package org.mule.weave.lsp.project.commands

import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.extension.protocol.OpenWindowsParams
import org.mule.weave.lsp.extension.protocol.WeaveOpenDialogParams
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.ui.utils.Buttons
import org.mule.weave.lsp.ui.wizard.DefaultWizardBuilder
import org.mule.weave.lsp.ui.wizard.DefaultWizardStepBuilder
import org.mule.weave.lsp.ui.wizard.InputWidgetBuilder
import org.mule.weave.lsp.ui.wizard.WidgetResult
import org.mule.weave.lsp.utils.Icons
import org.mule.weave.lsp.utils.Messages
import org.mule.weave.lsp.utils.Messages.NewDwProject

import java.io.File
import java.net.URI

class ProjectProvider(uIService: UIService, project: ProjectMetadata) {
  val icons = Icons.vscode

  def newProject(): Unit = {

    val createProjectTitle = "Create Project"

    val orgIdWidgetBuilder = new InputWidgetBuilder[ProjectCreationInfo](uIService).title(createProjectTitle).default("org.mycompany").prompt("Organization ID").result((projectInfo, result) => {
      projectInfo.groupId = result
      projectInfo
    }).selectionProvider(projectInfo => Some(projectInfo.groupId))

    val orgIdStep = new DefaultWizardStepBuilder[ProjectCreationInfo] //
      .widgetBuilder(orgIdWidgetBuilder)

    val artifactIdWidgetBuilder = new InputWidgetBuilder[ProjectCreationInfo](uIService).title(createProjectTitle).default("example").prompt("Artifact ID").result((projectInfo, result) => {
      projectInfo.artifactId = result
      projectInfo
    }).selectionProvider(projectInfo => Some(projectInfo.artifactId))

    val artifactIdStep = new DefaultWizardStepBuilder[ProjectCreationInfo] //
      .widgetBuilder(artifactIdWidgetBuilder)


    val versionWidget: InputWidgetBuilder[ProjectCreationInfo] = new InputWidgetBuilder[ProjectCreationInfo](uIService)
      .title(createProjectTitle)
      .default("1.0.0-SNAPSHOT")
      .prompt("Version")
      .selectionProvider(projectInfo =>
        Some(projectInfo.version)
      ).result((projectInfo, result) => {
        projectInfo.version = result
        projectInfo
      })

    val versionStep = new DefaultWizardStepBuilder[ProjectCreationInfo] //
      .widgetBuilder(versionWidget)

    val projectNameWidgetBuilder = new InputWidgetBuilder[ProjectCreationInfo](uIService).title(createProjectTitle).default("example-project").prompt("Project name").result((projectInfo, result) => {
      projectInfo.projectName = result
      projectInfo
    }).selectionProvider(projectInfo => Some(projectInfo.projectName))

    val projectNameStep = new DefaultWizardStepBuilder[ProjectCreationInfo] //
      .widgetBuilder(projectNameWidgetBuilder)

    val workspaceLocation: URI = project.home().toURI

    val projectPathWidgetBuilder = new InputWidgetBuilder[ProjectCreationInfo](uIService).title("Choose project path")
      .button(Buttons.browse(), previousResult => {
        val selectedPath = uIService.weaveFileBrowser(WeaveOpenDialogParams(false, true, false, previousResult.pathToCreate.toUri.toString, "Select", "Choose target folder")).get()
        previousResult.pathToCreate = new File(URI.create(selectedPath)).toPath
        new WidgetResult[ProjectCreationInfo](false, previousResult, Buttons.browse().id)
      })
      .result((projectInfo, result) => {
        projectInfo.pathToCreate = new File(result).toPath
        projectInfo
      })
      .selectionProvider(projectInfo => Some(projectInfo.pathToCreate.toFile.getAbsolutePath))

    val pathStep = new DefaultWizardStepBuilder[ProjectCreationInfo] //
      .widgetBuilder(projectPathWidgetBuilder)


    val info: ProjectCreationInfo = new DefaultWizardBuilder[ProjectCreationInfo] //
      .modelProvider(() => ProjectCreationInfo(groupId = "org.mycompany", artifactId = "example", version = "1.0.0-SNAPSHOT", projectName = "example-project", pathToCreate = new File(workspaceLocation).toPath))
      .step(orgIdStep).step(artifactIdStep).step(versionStep).step(projectNameStep).step(pathStep).create().open()

    if (info != null) {
      new DefaultProjectCreator().create(info)
      askForWindow(info.pathToCreate.resolve(info.projectName).toUri)
    }

  }


  private def askForWindow(projectPath: URI): Unit = {
    def openWindow(newWindow: Boolean): Unit = {
      val params = OpenWindowsParams(
        projectPath.toString,
        java.lang.Boolean.valueOf(newWindow)
      )
      uIService.openWindow(params)
    }

    val item = uIService
      .showMessageRequest(NewDwProject.askForNewWindowParams())
      .get()
    item match {
      case msg if msg == Messages.no =>
        openWindow(newWindow = false)
      case msg if msg == Messages.yes =>
        openWindow(newWindow = true)
      case _ =>
    }
  }
}
