package org.mule.weave.lsp.services

import org.eclipse.lsp4j
import org.eclipse.lsp4j.DidChangeConfigurationParams
import org.eclipse.lsp4j.DidChangeWatchedFilesParams
import org.eclipse.lsp4j.DidChangeWorkspaceFoldersParams
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.Location
import org.eclipse.lsp4j.SymbolInformation
import org.eclipse.lsp4j.SymbolKind
import org.eclipse.lsp4j.WorkspaceSymbolParams
import org.eclipse.lsp4j.services.WorkspaceService
import org.mule.weave.lsp.commands.CommandManagerService
import org.mule.weave.lsp.commands.Commands
import org.mule.weave.lsp.commands.InternalWeaveCommand
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolClient
import org.mule.weave.lsp.extension.protocol.SetContextParams
import org.mule.weave.lsp.extension.protocol.SetContextValue
import org.mule.weave.lsp.indexer.LSPWeaveIndexService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.jobs.Task
import org.mule.weave.lsp.project.DefaultProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.events.DocumentChangedEvent
import org.mule.weave.lsp.services.events.DocumentFocusChangedEvent
import org.mule.weave.lsp.services.events.DocumentOpenedEvent
import org.mule.weave.lsp.services.events.FileChangedEvent
import org.mule.weave.lsp.services.events.OnDocumentChanged
import org.mule.weave.lsp.services.events.OnDocumentFocused
import org.mule.weave.lsp.services.events.OnDocumentOpened
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.LSPConverters
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.editor.indexing.IdentifierType
import org.mule.weave.v2.editor.indexing.LocatedResult
import org.mule.weave.v2.editor.indexing.WeaveIdentifier
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.sdk.WeaveResourceResolver

import java.util
import java.util.concurrent.CompletableFuture
import java.util.logging.Level
import java.util.logging.Logger
import scala.collection.JavaConverters.seqAsJavaListConverter

/**
 * DataWeave Implementation of the LSP Workspace Service
 *
 */
class DataWeaveWorkspaceService(
                                 project: DefaultProjectMetadata,
                                 vfs: VirtualFileSystem,
                                 loggerFactory: ClientLoggerFactory,
                                 weaveProtocolClient: DataWeaveProtocolClient,
                                 jobManagerService: JobManagerService,
                                 indexService: LSPWeaveIndexService,
                                 commandManager: CommandManagerService
                               ) extends WorkspaceService {

  private val clientLogger = loggerFactory.createLogger(classOf[DataWeaveWorkspaceService])
  private val logger: Logger = Logger.getLogger(getClass.getName)
  private var eventBus: InternalEventBus = _
  private var projectKind: ProjectKind = _

  def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.eventBus = eventBus
    this.projectKind = projectKind

    eventBus.register(DocumentChangedEvent.DOCUMENT_CHANGED, new OnDocumentChanged {
      override def onDocumentChanged(vf: VirtualFile): Unit = {
        notifyContextChanged(vf)
      }
    })

    eventBus.register(DocumentOpenedEvent.DOCUMENT_OPENED, new OnDocumentOpened {
      override def onDocumentOpened(vf: VirtualFile): Unit = {
        notifyContextChanged(vf)
      }
    })

    eventBus.register(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, new OnDocumentFocused {
      override def onDocumentFocused(vf: VirtualFile): Unit = {
        notifyContextChanged(vf)
      }
    })
  }

  private def notifyContextChanged(vf: VirtualFile): Unit = {
    val params = SetContextParams(commandManager.commands.map(command => {
      val isEnabled = command match {
        case command: InternalWeaveCommand => command.enabled(vf)
        case _ => true
      }
      SetContextValue(command.commandId() + Commands.COMMAND_ENABLED_SUFFIX, isEnabled, vf.url())
    }) asJava)
    weaveProtocolClient.setContext(params)
  }

  override def symbol(params: WorkspaceSymbolParams): CompletableFuture[util.List[_ <: SymbolInformation]] = {
    CompletableFuture.supplyAsync(() => {
      val resourceResolver: WeaveResourceResolver = vfs.asResourceResolver
      val iterable: Iterable[LocatedResult[WeaveIdentifier]] = indexService.searchSymbol(params.getQuery)
      val result = new util.ArrayList[SymbolInformation]()
      iterable.foreach((lr) => {
        val information = new SymbolInformation()
        information.setName(lr.value.value)
        val kind: SymbolKind = lr.value.idType match {
          case IdentifierType.FUNCTION => SymbolKind.Function
          case IdentifierType.VARIABLE => SymbolKind.Variable
          case IdentifierType.NAMESPACE => SymbolKind.Namespace
          case IdentifierType.ANNOTATION => SymbolKind.Interface
          case _ => SymbolKind.Field
        }
        information.setKind(kind)
        val name = lr.moduleName
        val moduleResource = resourceResolver.resolve(name)
        moduleResource match {
          case Some(weaveResource) => {
            val range = new lsp4j.Range()
            val startPosition: Position = weaveResource.positionOf(lr.value.startLocation)
            val endPosition: Position = weaveResource.positionOf(lr.value.endLocation)
            range.setStart(LSPConverters.toPosition(startPosition))
            range.setEnd(LSPConverters.toPosition(endPosition))
            val location = new Location(weaveResource.url(), range)
            information.setLocation(location)
            result.add(information)
          }
          case _ =>
        }
      })
      result
    })
  }

  override def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {
    logger.log(Level.INFO, "didChangeConfiguration: " + params.getSettings)
    project.settings.update(params.getSettings)
  }

  override def executeCommand(params: ExecuteCommandParams): CompletableFuture[AnyRef] = {

    logger.log(Level.INFO, "executeCommand: " + params)
    CompletableFuture.supplyAsync(() => {
      commandManager.commandBy(params.getCommand)
        .map(c => {
          var result: AnyRef = null
          jobManagerService.execute(new Task() {
            override def run(cancelable: Status): Unit = {
              result = c.execute(params)
            }
          }, s"Running command: ${c.name()}", s"${c.description(params)}")
          result
        })
        .orNull
    })
  }

  override def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {
    logger.log(Level.INFO, "Changed Folders: " + params.getEvent)
  }

  override def didChangeWatchedFiles(params: DidChangeWatchedFilesParams): Unit = {
    params.getChanges.forEach((fe) => {
      if (projectKind.isProjectFile(fe.getUri)) {
        clientLogger.logDebug("Changed Watched File : " + fe.getUri + " - " + fe.getType)
        eventBus.fire(new FileChangedEvent(fe.getUri, fe.getType))
      } else {
        clientLogger.logDebug("Ignored Watched File : " + fe.getUri + " - " + fe.getType)
      }
    })
  }
}
