package org.mule.weave.lsp.actions

import org.eclipse.lsp4j.CodeAction
import org.eclipse.lsp4j.CodeActionParams
import org.mule.weave.extension.api.extension.action.WeaveCodeActionProvider
import org.mule.weave.extension.api.metadata.ContextMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.utils.InternalEventBus

import java.util.Optional
import java.util.logging.Level
import java.util.logging.Logger
import scala.util.Failure
import scala.util.Success
import scala.util.Try

class CodeActionsManagerService(toolingService: DataWeaveToolingService) extends ToolingService {

  private val LOGGER = Logger.getLogger(getClass.getName)

  private lazy val projectIndependentCodeActionProviders: Seq[WeaveCodeActionProvider] = Seq(
    new QuickFixAction(toolingService),
    new InsertDocumentationAction(toolingService),
    new InsertWeaveTypeAction(toolingService),
    new RefactorActionProvider(toolingService)
  )

  private var codeActionsProviders: Seq[WeaveCodeActionProvider] = {
    projectIndependentCodeActionProviders
  }

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    initProjectCodeActionProviders(projectKind)
  }

  private def initProjectCodeActionProviders(projectKind: ProjectKind): Unit = {
    codeActionsProviders = codeActionsProviders ++ projectKind.customCodeActions()
  }

  private def codeActionProvidersFor(params: CodeActionParams, maybeContextMetadata: Optional[ContextMetadata]): Seq[WeaveCodeActionProvider] = {
    codeActionsProviders.filter(cap => {
      val context = maybeContextMetadata.orElse(null)
      Try(cap.handles(params, context)) match {
        case Success(providers) => providers
        case Failure(e) =>
          LOGGER.log(Level.SEVERE, "Could not handle code action.", e)
          false
      }
    })
  }

  def codeActionsFor(params: CodeActionParams, maybeContextMetadata: Optional[ContextMetadata]): Seq[CodeAction] = {
    val providers = codeActionProvidersFor(params, maybeContextMetadata)
    val context = maybeContextMetadata.orElse(null)
    val codeActions = providers.flatMap(p => {
      Try(p.actions(params, context)) match {
        case Success(codeActions) => codeActions
        case Failure(e) =>
          LOGGER.log(Level.SEVERE, "Could get code action.", e)
          Seq.empty
      }
    })
    codeActions
  }
}
