package org.mule.weave.lsp.commands

import org.apache.commons.io.FilenameUtils
import org.apache.commons.lang3.StringUtils
import org.eclipse.lsp4j
import org.eclipse.lsp4j.ApplyWorkspaceEditParams
import org.eclipse.lsp4j.CreateFile
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.eclipse.lsp4j.Position
import org.eclipse.lsp4j.ResourceOperation
import org.eclipse.lsp4j.TextDocumentEdit
import org.eclipse.lsp4j.TextEdit
import org.eclipse.lsp4j.VersionedTextDocumentIdentifier
import org.eclipse.lsp4j.WorkspaceEdit
import org.eclipse.lsp4j.jsonrpc.messages.Either
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.lsp.commands.Commands.argAsString
import org.mule.weave.lsp.extension.protocol.OpenTextDocumentParams
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxParams
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxResult
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickItem
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickParams
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProjectStructureHelper
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl

import java.io.File
import java.util

abstract class AbstractCreateFileCommand extends WeaveCommand {

  val INTEGRATION_MAPPING_CONTENT =
    """|/**
       |* This mapping won't be shared through your library, but you can use it to try out your module and create integration tests.
       |*/"""


  def uIService: UIService

  def workspaceServiceManager: WorkspaceServiceManager

  def getInputLabel: String

  def getDefaultName(params: ExecuteCommandParams, project: ProjectKind): String

  def getTemplate(params: ExecuteCommandParams, project: ProjectKind): String

  def retrieveDefaultFolder(uIService: UIService): Option[(File, ProjectKind)] = {
    val projectKinds = workspaceServiceManager.projectKinds()
    var maybeProjectKind: Option[ProjectKind] = Option.empty
    if (!projectKinds.isEmpty) {
      if (projectKinds.length == 1) {
        maybeProjectKind = Option.apply(projectKinds.head)
      } else {
        val inputs: Array[WeaveQuickPickItem] = projectKinds.map((p) => {
          WeaveQuickPickItem(p.projectMetadata.home.getAbsolutePath, p.project.getName)
        })
        val result = uIService.weaveQuickPick(WeaveQuickPickParams(
          util.Arrays.asList(inputs: _*),
          "Select the Project where to create the File",
        )).get()
        if (!result.cancelled && !result.itemsId.isEmpty) {
          val projectPath = result.itemsId.get(0)
          maybeProjectKind = projectKinds.find(_.projectMetadata().home().getAbsolutePath.equals(projectPath))
        }
      }
    } else {
      uIService.showMessage(new MessageParams(MessageType.Error, "No source folder was detected. Please open a Project before trying to create an element."))
    }

    if (maybeProjectKind.isDefined) {
      val project = maybeProjectKind.get
      val modulesWithRoots = project.structure().modules().filter(_.roots().nonEmpty)
      if (modulesWithRoots.isEmpty) {
        uIService.showMessage(new MessageParams(MessageType.Info, "Selected project doesn't have modules with source folder. Please check the project before trying to create an element."))
        None
      } else if (modulesWithRoots.length == 1) {
        Option(ProjectStructureHelper.mainRoot(modulesWithRoots.head).get.defaultSourceFolder, project)
      } else {
        val inputs: Array[WeaveQuickPickItem] = modulesWithRoots.map((m) => {
          WeaveQuickPickItem(m.name(), m.name())
        })
        val result = uIService.weaveQuickPick(WeaveQuickPickParams(
          util.Arrays.asList(inputs: _*),
          "Select the Module where to create the File",
        )).get()
        if (result.cancelled || result.itemsId.isEmpty) {
          None
        } else {
          val moduleName = result.itemsId.get(0)
          val maybeStructure = modulesWithRoots.find(_.name().equals(moduleName))
          Option(maybeStructure.map(ProjectStructureHelper.mainRoot(_).get.defaultSourceFolder).get, project)
        }
      }
    } else {
      None
    }
  }

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val directoryProjectKindTuple: Option[(File, ProjectKind)] = if (params.getArguments != null && !params.getArguments.isEmpty) {
      val directoryUri = argAsString(params.getArguments, 0)
      val maybeProjectKind = workspaceServiceManager.projectKind(directoryUri)
      if (maybeProjectKind.isDefined) {
        URLUtils.toFile(directoryUri).map(directory => (directory, maybeProjectKind.get))
      } else {
        uIService.showMessage(new MessageParams(MessageType.Error, s"No project is associated to: ${directoryUri}. ${commandId} cannot be executed"))
        None
      }
    } else {
      retrieveDefaultFolder(uIService)
    }
    directoryProjectKindTuple match {
      case Some((folder: File, project: ProjectKind)) => {
        val result: WeaveInputBoxResult = uIService.weaveInputBox(WeaveInputBoxParams(getInputLabel, getDefaultName(params, project))).get()
        if (!result.cancelled) {
          var name = result.value
          if (StringUtils.isEmpty(FilenameUtils.getExtension(name))) {
            name = name + ".dwl"
          }

          val mappingFile = toLSPUrl(new File(folder, name))
          val createFile: Either[TextDocumentEdit, ResourceOperation] = Either.forRight[TextDocumentEdit, ResourceOperation](new CreateFile(mappingFile))

          val textEdit: TextEdit = new TextEdit(new lsp4j.Range(new Position(0, 0), new Position(0, 0)), getTemplate(params, project))
          val textDocumentEdit: TextDocumentEdit = new TextDocumentEdit(new VersionedTextDocumentIdentifier(mappingFile, 0), util.Arrays.asList(textEdit))
          val insertText = Either.forLeft[TextDocumentEdit, ResourceOperation](textDocumentEdit)

          val edits: util.List[Either[TextDocumentEdit, ResourceOperation]] = new util.ArrayList[Either[TextDocumentEdit, ResourceOperation]]()
          edits.add(createFile)
          edits.add(insertText)
          edits.addAll(util.Arrays.asList(project.newFile(folder, name): _*))

          val response = workspaceServiceManager.workspaceEditService.applyEdit(new ApplyWorkspaceEditParams(new WorkspaceEdit(edits))).get()
          if (response.isApplied) {
            uIService.openTextDocument(OpenTextDocumentParams(mappingFile))
          }
        }
      }
      case None =>
    }
    null

  }

}
