package org.mule.weave.lsp.services

import net.liftweb.json.DefaultFormats
import net.liftweb.json.parseOpt
import org.apache.commons.io.FileUtils
import org.apache.commons.io.filefilter.RegexFileFilter
import org.apache.commons.io.filefilter.TrueFileFilter
import org.eclipse.lsp4j.FileChangeType
import org.mule.weave.dsp.ProcessListener
import org.mule.weave.extension.api.component.structure.WeaveProjectStructure
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.extension.protocol.PublishTestItemsParams
import org.mule.weave.lsp.extension.protocol.PublishTestResultsParams
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolClient
import org.mule.weave.lsp.extension.protocol.WeaveTestItem
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.jobs.Task
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.Settings
import org.mule.weave.lsp.project.events.OnProjectStarted
import org.mule.weave.lsp.project.events.OnSettingsChanged
import org.mule.weave.lsp.project.events.ProjectStartedEvent
import org.mule.weave.lsp.project.events.SettingsChangedEvent
import org.mule.weave.lsp.services.DataWeaveTestService.TEST_FILES_PATTERN
import org.mule.weave.lsp.services.events.FileChangedEvent
import org.mule.weave.lsp.services.events.OnFileChanged
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.NoopWeaveTestIndexer
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.URLUtils.isChildOf
import org.mule.weave.lsp.utils.URLUtils.toFile
import org.mule.weave.lsp.utils.WeaveDirectoryUtils
import org.mule.weave.lsp.utils.WeaveSemanticTestIndexer
import org.mule.weave.lsp.utils.WeaveSyntacticTestIndexer
import org.mule.weave.lsp.utils.WeaveTestIndexer
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.v2.editor.VirtualFile

import java.io.File
import java.util
import java.util.concurrent.Semaphore
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean
import java.util.regex.Pattern
import scala.collection.JavaConverters._
import scala.collection.mutable

class DataWeaveTestService(
                            weaveLanguageClient: DataWeaveProtocolClient,
                            virtualFileSystem: ProjectFileSystemService,
                            loggerFactory: ClientLoggerFactory,
                            jobManagerService: JobManagerService,
                            agentService: WeaveAgentService,
                            dataWeaveToolingService: DataWeaveToolingService,
                            projectMetadata: ProjectMetadata,
                            eventBus: InternalEventBus) extends ToolingService with ProcessListener {

  private val clientLogger = loggerFactory.createLogger(classOf[DataWeaveTestService])
  private var projectKind: ProjectKind = _
  private val testsCache: mutable.HashMap[String, WeaveTestItem] = mutable.HashMap()
  private var semanticTestIndexer: WeaveTestIndexer = if (projectMetadata.settings().enableSemanticTestIndexer().booleanValue()) {
    WeaveSemanticTestIndexer(agentService, clientLogger)
  } else {
    NoopWeaveTestIndexer
  }
  private val syntacticTestIndexer: WeaveTestIndexer = WeaveSyntacticTestIndexer(dataWeaveToolingService)

  private val mutex = new Semaphore(1)

  eventBus.register(SettingsChangedEvent.SETTINGS_CHANGED, new OnSettingsChanged {
    override def onSettingsChanged(modifiedSettingsName: Array[String]): Unit = {
      if (modifiedSettingsName.contains(Settings.ENABLE_SEMANTIC_TEST_INDEXER_PROP_NAME)) {
        jobManagerService.schedule(new Task {
          override def run(cancelable: Status): Unit = {
            mutex.acquire()
            testsCache.clear()
            if (projectMetadata.settings().enableSemanticTestIndexer().booleanValue()) {
              semanticTestIndexer = WeaveSemanticTestIndexer(agentService, clientLogger)
            } else {
              semanticTestIndexer = NoopWeaveTestIndexer
            }
            discoverAllTests(projectKind.structure())
            mutex.release()
          }
        }, "Discovering Test", "Discovering all tests in project: " + projectMetadata.home())
      }
    }
  })

  private def discoverAllTests(projectStructure: WeaveProjectStructure): Unit = {
    clientLogger.logDebug("Discovering Tests ...")
    val testFolders: Array[File] = WeaveDirectoryUtils.wtfUnitTestFolder(projectStructure)
    testFolders.foreach(testFolder => {
      val start = System.currentTimeMillis()
      val dwTestsFolder = testFolder.getAbsolutePath;
      clientLogger.logInfo(s"Indexing tests from: $dwTestsFolder")
      FileUtils
        .listFiles(testFolder, new RegexFileFilter(DataWeaveTestService.TEST_FILES_REGEX), TrueFileFilter.INSTANCE)
        .forEach((f) => {
          discoverTestsIn(virtualFileSystem.file(f.toURI.normalize().toString))
        })
      clientLogger.logInfo(s"Indexing all tests from: $dwTestsFolder finished and took ${System.currentTimeMillis() - start} ms.")
    })

    clientLogger.logDebug("All tests Discovered ...")
  }

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.projectKind = projectKind
    eventBus.register(ProjectStartedEvent.PROJECT_STARTED, new OnProjectStarted {

      override def onProjectStarted(projectMetadata: ProjectMetadata): Unit = {
        jobManagerService.schedule(new Task {
          override def run(cancelable: Status): Unit = {
            mutex.acquire()
            discoverAllTests(projectKind.structure())
            mutex.release()
          }
        }, "Discovering Test", "Discovering all tests in project :" + projectMetadata.home())
      }
    })

    eventBus.register(FileChangedEvent.FILE_CHANGED_EVENT, new OnFileChanged {
      override def onFileChanged(uri: String, changeType: FileChangeType): Unit = {
        if (WeaveDirectoryUtils
          .wtfUnitTestFolder(projectKind.structure())
          .exists(file => URLUtils.isChildOf(uri, file) || toFile(uri).exists(URLUtils.isChildOf(file.getAbsolutePath, _)))) {
          changeType match {
            case FileChangeType.Created | FileChangeType.Changed =>
              discoverTest(uri)
            case FileChangeType.Deleted =>
              deleteRecursiveTests(uri)
          }
        }
      }
    })
  }

  private def deleteRecursiveTests(uri: String): Unit = {
    testsCache.retain((k, _) => !isChildOf(k, uri))
    publishTests()
  }

  private def discoverTestsIn(virtualFile: VirtualFile): Unit = {
    val weaveItem: WeaveTestItem = semanticTestIndexer
      .index(virtualFile)
      .orElse(syntacticTestIndexer.index(virtualFile))
    match {
      case Some(wti) => WeaveTestItem(label = virtualFile.getNameIdentifier.toString(), uri = virtualFile.url(), children = Seq(wti).asJava)
      case None => WeaveTestItem(label = virtualFile.getNameIdentifier.toString(), uri = virtualFile.url(), children = Seq().asJava)
    }
    testsCache.put(weaveItem.uri, weaveItem)
    publishTests()
  }

  private def discoverTest(uri: String): Unit = {
    if (TEST_FILES_PATTERN.matcher(uri).matches()) {
      val start = System.currentTimeMillis()
      clientLogger.logDebug(s"Indexing tests of $uri")
      val file = toFile(uri)
      if (file.exists(f => f.exists() && !f.isDirectory)) {
        testsCache.remove(uri).foreach(_ => publishTests())
        val virtualFile = virtualFileSystem.file(file.get.toURI.normalize().toString)
        jobManagerService.schedule(new Task {
          override def run(cancelable: Status): Unit = {
            discoverTestsIn(virtualFile)
          }
        }, "Indexing Test", "Indexing Test in at" + virtualFile.url())

      } else {
        clientLogger.logDebug(s"Skipping test indexing, it's a directory: $uri")
      }

      clientLogger.logDebug(s"Indexing tests of: $uri finished and took ${System.currentTimeMillis() - start} ms.")
    }
  }

  private def publishTests(): Unit = {
    val weaveTestItems: util.List[WeaveTestItem] = new util.ArrayList[WeaveTestItem]()
    testsCache.values.foreach(cachedItem => weaveTestItems.add(cachedItem))
    weaveLanguageClient.publishTestItems(PublishTestItemsParams(URLUtils.toLSPUrl(projectKind.projectMetadata().home()), weaveTestItems))
  }

  private def publishTestResult(event: TestEvent): Unit = {
    weaveLanguageClient.publishTestResults(PublishTestResultsParams(event.event, event.message.getOrElse(""), event.name, event.duration.getOrElse("0").toDouble.toInt, event.locationHint.getOrElse(""), event.status.getOrElse("")))
  }

  def output(jsonLine: String): Unit = {
    implicit val formats: DefaultFormats.type = DefaultFormats
    try {
      parseOpt(jsonLine) match {
        case Some(jValue) => publishTestResult(jValue.extract[TestEvent])
        case None =>
      }
    } catch {
      case exception: Throwable => clientLogger.logError("Error while outputting results", exception)
    }
  }

  override def onFinished(): Unit = {
    publishTestResult(TestEvent("testRunFinished", ""))
  }
}

object DataWeaveTestService {
  val TEST_FILES_REGEX = ".*(Test|TestCase|Spec)\\.dwl"
  val TEST_FILES_PATTERN: Pattern = Pattern.compile(TEST_FILES_REGEX)
  val SINGLE_TEST_FILE_DISCOVER_TIME_OUT = 10000L
  val DISCOVER_TIME_OUT_UNIT = TimeUnit.MILLISECONDS
}

case class TestEvent(event: String, name: String, message: Option[String] = None, error: Option[String] = None, duration: Option[String] = None, locationHint: Option[String] = None, nodeId: Option[String] = None, status: Option[String] = None, parentNodeId: Option[String] = None, captureStandardOutput: Option[String] = None, location: Option[String] = None)
