package org.mule.weave.lsp.services

import org.eclipse.lsp4j.DidChangeConfigurationParams
import org.eclipse.lsp4j.DidChangeWatchedFilesParams
import org.eclipse.lsp4j.DidChangeWorkspaceFoldersParams
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.SymbolInformation
import org.eclipse.lsp4j.WorkspaceSymbolParams
import org.eclipse.lsp4j.services.WorkspaceService
import org.mule.weave.lsp.commands.CommandManagerService
import org.mule.weave.lsp.commands.Commands
import org.mule.weave.lsp.commands.InternalWeaveCommand
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolClient
import org.mule.weave.lsp.extension.protocol.SetContextParams
import org.mule.weave.lsp.extension.protocol.SetContextValue
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.jobs.Task
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.events.DocumentChangedEvent
import org.mule.weave.lsp.services.events.DocumentFocusChangedEvent
import org.mule.weave.lsp.services.events.DocumentOpenedEvent
import org.mule.weave.lsp.services.events.OnDocumentChanged
import org.mule.weave.lsp.services.events.OnDocumentFocused
import org.mule.weave.lsp.services.events.OnDocumentOpened
import org.mule.weave.v2.editor.VirtualFile

import java.util
import java.util.concurrent.CompletableFuture
import java.util.logging.Level
import java.util.logging.Logger
import scala.collection.JavaConverters.seqAsJavaListConverter
import scala.collection.concurrent.TrieMap
import scala.collection.mutable

/**
 * DataWeave Implementation of the LSP Workspace Service
 */
class DataWeaveWorkspaceService(weaveProtocolClient: DataWeaveProtocolClient,
                                jobManagerService: JobManagerService,
                                commandManager: CommandManagerService)
  extends WorkspaceService with WorkspaceServiceContributorRegistry {

  private val logger: Logger = Logger.getLogger(getClass.getName)
  private val projectWorkspaceContributors: TrieMap[ProjectKind, WorkspaceServiceContributor] = TrieMap()
  private val workspaceServiceListeners: mutable.Buffer[WorkspaceServiceListener] = mutable.Buffer()

  private val onDocumentChanged: OnDocumentChanged = (vf: VirtualFile) => {
    notifyContextChanged(vf)
  }

  private val onDocumentOpened: OnDocumentOpened = (vf: VirtualFile) => {
    notifyContextChanged(vf)
  }

  private val onDocumentFocused: OnDocumentFocused = (vf: VirtualFile) => {
    notifyContextChanged(vf)
  }

  override def addWorkspaceServiceListener(workspaceServiceListener: WorkspaceServiceListener): Unit = {
    workspaceServiceListeners += workspaceServiceListener
  }

  def onProjectKindCreated(projectKind: ProjectKind, workspaceServiceContributor: WorkspaceServiceContributor): Unit = {
    this.projectWorkspaceContributors.putIfAbsent(projectKind, workspaceServiceContributor) match {
      case Some(_) => // Already registered, therefore no need to register the listeners again...
      case None => registerDocumentListeners(projectKind)
    }
  }

  def onProjectKindRemoved(projectKind: ProjectKind): Unit = {
    this.projectWorkspaceContributors.remove(projectKind) match {
      case Some(_) => unregisterDocumentListeners(projectKind)
      case None =>
    }
  }

  private def registerDocumentListeners(projectKind: ProjectKind): Unit = {
    projectKind.eventBus.register(DocumentChangedEvent.DOCUMENT_CHANGED, onDocumentChanged)
    projectKind.eventBus.register(DocumentOpenedEvent.DOCUMENT_OPENED, onDocumentOpened)
    projectKind.eventBus.register(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, onDocumentFocused)
  }

  private def unregisterDocumentListeners(projectKind: ProjectKind): Unit = {
    projectKind.eventBus.unRegister(DocumentChangedEvent.DOCUMENT_CHANGED, onDocumentChanged)
    projectKind.eventBus.unRegister(DocumentOpenedEvent.DOCUMENT_OPENED, onDocumentOpened)
    projectKind.eventBus.unRegister(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, onDocumentFocused)
  }

  private def notifyContextChanged(vf: VirtualFile): Unit = {
    val params = SetContextParams(commandManager.commands.map(command => {
      val isEnabled = command match {
        case command: InternalWeaveCommand => command.enabled(vf.url())
        case _ => true
      }
      SetContextValue(command.commandId() + Commands.COMMAND_ENABLED_SUFFIX, isEnabled, vf.url())
    }) asJava)
    weaveProtocolClient.setContext(params)
  }

  override def symbol(params: WorkspaceSymbolParams): CompletableFuture[util.List[_ <: SymbolInformation]] = {
    logger.log(Level.INFO, "symbol: " + params.getQuery)

    val futures: Array[CompletableFuture[util.List[_ <: SymbolInformation]]] =
      projectWorkspaceContributors.values.map(projectWorkspaceContributor => projectWorkspaceContributor.symbol(params)).toArray

    val allFutures = CompletableFuture.allOf(futures: _*)

    allFutures.thenApply { _ =>
      val symbolsInformation: util.ArrayList[SymbolInformation] = new util.ArrayList()
      futures.foreach(future => symbolsInformation.addAll(future.join()))
      symbolsInformation
    }
  }

  override def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {
    logger.log(Level.INFO, "didChangeConfiguration: " + params.getSettings)

    workspaceServiceListeners
      .foreach(listener => listener.didChangeConfiguration(params))

    projectWorkspaceContributors.values
      .foreach(projectWorkspaceContributor => projectWorkspaceContributor.didChangeConfiguration(params))
  }

  override def executeCommand(params: ExecuteCommandParams): CompletableFuture[AnyRef] = {
    logger.log(Level.INFO, "executeCommand: " + params)
    CompletableFuture.supplyAsync(() => {
      commandManager.commandBy(params.getCommand)
        .map(c => {
          var result: AnyRef = null
          jobManagerService.execute(new Task() {
            override def run(cancelable: Status): Unit = {
              result = c.execute(params)
            }
          }, s"Running command: ${c.name()}", s"${c.description(params)}")
          result
        })
        .orNull
    })
  }

  override def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {
    logger.log(Level.INFO, "didChangeWorkspaceFolders: " + params.getEvent)
    workspaceServiceListeners
      .foreach(listener => listener.didChangeWorkspaceFolders(params))
  }

  override def didChangeWatchedFiles(params: DidChangeWatchedFilesParams): Unit = {
    logger.log(Level.INFO, "didChangeWatchedFiles: " + params.getChanges)
    projectWorkspaceContributors.values.foreach(projectWorkspaceContributor => {
      projectWorkspaceContributor.didChangeWatchedFiles(params)
    })
  }

}

trait WorkspaceServiceListener {
  def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {}

  def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {}
}