package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.extension.protocol.LaunchConfiguration
import org.mule.weave.lsp.extension.protocol.LaunchConfiguration._
import org.mule.weave.lsp.extension.protocol.LaunchConfigurationProperty
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.ConfigurationService
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.utils.Icons
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.v2.editor.VirtualFileSystem

import java.util

class LaunchWeaveTestCommand(uIService: UIService, configurationService: ConfigurationService, workspaceServiceManager: WorkspaceServiceManager) extends WeaveCommand {

  val icon: Icons = Icons.vscode

  override def commandId(): String = Commands.DW_LAUNCH_TEST

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val urls: String = Commands.argAsString(params.getArguments(), 0)
    urls.split(",")
      .groupBy(url => workspaceServiceManager.projectKind(url))
      .foreach(entry => {
        entry._1 match {
          case Some(projectKind: ProjectKind) => {
            val names = entry._2
              .flatMap((url) => Option(projectKind.vfs().file(url)))
              .map((vf) => vf.getNameIdentifier.toString())
              .mkString(",")
            val noDebug: Boolean = Commands.argAsBoolean(params.getArguments(), 1)
            val launchConfiguration = LaunchWeaveTestCommand.createDefaultConfiguration(URLUtils.toLSPUrl(projectKind.projectMetadata().home()), names, noDebug, projectKind.projectMetadata())
            configurationService.runConfiguration(launchConfiguration)
          }
          case None =>
            uIService.showMessage(new MessageParams(MessageType.Warning, s"Unable to find a project for test URL files: ${entry._2.mkString("")}."))
        }
      })
    null
  }


  override def name(): String = "Launch DataWeave Test."

  override def description(params: ExecuteCommandParams): String = "Launching DataWeave Test."
}

object LaunchWeaveTestCommand {
  def createDefaultConfiguration(projectUri: String, testName: String, noDebug: Boolean, projectMetadata: ProjectMetadata): LaunchConfiguration = {
    val mapping: LaunchConfigurationProperty = LaunchConfigurationProperty(MAIN_FILE_NAME, testName)
    val testRunnerEnvVarFile: LaunchConfigurationProperty = LaunchConfigurationProperty(TEST_RUNNER_ENV_VAR_FILE, projectMetadata.settings().testRunnerEnvVarFileName.value())
    LaunchConfiguration(LaunchConfiguration.WTF_CONFIG_TYPE_NAME, projectUri, "Debugging " + testName, LAUNCH_REQUEST_TYPE, noDebug, util.Arrays.asList(mapping, testRunnerEnvVarFile))
  }
}
