package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.mule.weave.lsp.commands.Commands.argAsString
import org.mule.weave.lsp.commands.CreateMappingTestCommand.NEW_ID
import org.mule.weave.lsp.commands.CreateMappingTestCommand.UPDATE_ID
import org.mule.weave.lsp.extension.protocol.PreviewResult
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxParams
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickItem
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickParams
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickResult
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProjectStructureHelper
import org.mule.weave.lsp.project.components.Scenario
import org.mule.weave.lsp.services.PreviewService
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

import java.util
import java.util.concurrent.CompletableFuture

class CreateMappingTestCommand(workspaceServiceManager: WorkspaceServiceManager,
                               uIService: UIService
                              ) extends InternalWeaveCommand {

  override def commandId(): String = Commands.DW_CREATE_MAPPING_TEST

  private def newScenario(uIService: UIService): Option[String] = {
    val nameOfTheTestScenarioNameInputBoxResult = uIService.weaveInputBox(WeaveInputBoxParams(title = "Specify the name of the test scenario", value = "NewTestScenario", prompt = "Name of the mapping test")).get()
    if (!nameOfTheTestScenarioNameInputBoxResult.cancelled) {
      Some(nameOfTheTestScenarioNameInputBoxResult.value)
    } else {
      None
    }
  }

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val uri = argAsString(params.getArguments, 0)
    workspaceServiceManager.projectKind(uri) match {
      case Some(projectKind: ProjectKind) => {
        val scenariosManager: WeaveScenarioManagerService = projectKind.toolingService(classOf[WeaveScenarioManagerService])
        val virtualFile: VirtualFileSystem = projectKind.vfs()
        val previewService: PreviewService = projectKind.toolingService(classOf[PreviewService])

        val file: VirtualFile = virtualFile.file(uri)
        if (!previewService.canRunPreview(file)) {
          return null
        }
        val nameIdentifier = file.getNameIdentifier
        val maybeActiveScenario = scenariosManager.activeScenario(nameIdentifier)
        var nameOfTheTestScenario: Option[String] = None
        if (maybeActiveScenario.isEmpty || maybeActiveScenario.exists((s) => s.name.equals(Scenario.PLAYGROUND_SCENARIO))) {
          nameOfTheTestScenario = newScenario(uIService)
        } else {
          val quickPickResult: CompletableFuture[WeaveQuickPickResult] = uIService.weaveQuickPick(
            WeaveQuickPickParams(
              util.Arrays.asList(
                WeaveQuickPickItem(UPDATE_ID, s"Update `${maybeActiveScenario.get.name}` expected output", description = s"Updates ${maybeActiveScenario.get.name} expected output.", picked = true),
                WeaveQuickPickItem(NEW_ID, "Create new test", "Creates a new test scenario")
              )
            )
          )

          val pickResult: WeaveQuickPickResult = quickPickResult.get()
          if (!pickResult.cancelled && !pickResult.itemsId.isEmpty) {
            val pressedId = pickResult.itemsId.get(0)
            if (UPDATE_ID.equals(pressedId)) {
              nameOfTheTestScenario = Some(maybeActiveScenario.get.name)
            } else {
              nameOfTheTestScenario = newScenario(uIService)
            }
          }
        }

        if (nameOfTheTestScenario.isDefined) {
          val previewResult: PreviewResult = previewService.runPreview(file)
          if (previewResult.success) {
            createMappingTest(nameIdentifier, nameOfTheTestScenario.get, previewResult.mimeType, previewResult.content, maybeActiveScenario, previewResult.fileExtension, scenariosManager)
          } else {
            uIService.showMessage(new MessageParams(MessageType.Warning, "Can't create mapping test, there was an error running the current mapping."))
          }
        }
      }
      case None => {
        uIService.showMessage(new MessageParams(MessageType.Warning, "Can't create mapping test, selected URI file does not belong to a project."))
      }
    }
    null
  }

  private def createMappingTest(nameIdentifier: NameIdentifier, nameOfTheTestScenario: String, mimeType: String, content: String, maybeActiveScenario: Option[Scenario], fileExtension: String, scenariosManager: WeaveScenarioManagerService): Unit = {
    val outputName = s"out${fileExtension}"
    if (maybeActiveScenario.isDefined) {
      val activeScenario = maybeActiveScenario.get
      scenariosManager.copyScenarioTo(nameIdentifier, activeScenario, nameOfTheTestScenario)
    } else {
      scenariosManager.createScenario(nameIdentifier, nameOfTheTestScenario, WeaveScenarioManagerService.DEFAULT_INPUT)
    }
    scenariosManager.saveOutput(nameIdentifier, nameOfTheTestScenario, outputName, content)
    scenariosManager.createOrUpdateMappingTest(nameIdentifier, nameOfTheTestScenario, mimeType)
  }

  override def name(): String = "Create mapping test."

  override def description(params: ExecuteCommandParams): String = "Creating mapping test."

  override def enabled(uri: String): Boolean = {
    val maybeProjectKind = workspaceServiceManager.projectKind(uri)
    if (maybeProjectKind.isDefined) {
      val projectKind = maybeProjectKind.get
      projectKind.toolingService(classOf[PreviewService]).canRunPreview(projectKind.toolingService(classOf[ProjectFileSystemService]).file(uri))
    } else {
      false
    }
  }
}

object CreateMappingTestCommand {
  val UPDATE_ID = "update"
  val NEW_ID = "new"
}
