package org.mule.weave.lsp.project.components

import org.mule.weave.extension.api.component.structure.WeaveModuleStructure
import org.mule.weave.extension.api.component.structure.WeaveProjectStructure
import org.mule.weave.extension.api.component.structure.WeaveRootKind
import org.mule.weave.extension.api.component.structure.WeaveRootStructure
import org.mule.weave.extension.api.component.structure.WeaveTargetFolder
import org.mule.weave.extension.api.component.structure.WeaveTargetKind
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.v2.editor.VirtualFile

import java.io.File

object ProjectStructureHelper {

  /**
   * Returns true if the Virtual File belongs to a source or resource folder
   *
   * @param vf The virtual file to validate
   * @return True if the file is part of the project
   */
  def isAProjectFile(vf: VirtualFile, projectStructure: WeaveProjectStructure): Boolean = {
    isAProjectFile(vf.url(), projectStructure)
  }

  /**
   * Returns true if the Virtual File belongs to a source or resource folder
   *
   * @param url The url
   * @return True if the file is part of the project
   */
  def isAProjectFile(url: String, projectStructure: WeaveProjectStructure): Boolean = {
    projectStructure.modules.exists((module) => {
      module.roots.exists((root) => {
        URLUtils.isChildOfAny(url, root.sources) || URLUtils.isChildOfAny(url, root.resources)
      }) || (module.descriptors().exists((descriptor) => {
        URLUtils.toFile(url).exists(file => file.getAbsolutePath.equals(descriptor.getFile.getAbsolutePath))
      }))
    })
  }

  def getOrCreateTestSourceFolder(projectStructure: WeaveProjectStructure, uri: String): File = {
    val module = moduleOfFile(projectStructure, uri)
    if (module.isEmpty) {
      throw new RuntimeException("Couldn't find module for file " + uri)
    }

    val root = module.get.roots.find(rootStructure => rootStructure.kind == WeaveRootKind.TEST)

    if (root.isEmpty) {
      throw new RuntimeException("No test directory defined for this project")
    }
    val testDir = root.map(_.sources()).head.head
    if (!testDir.exists()) {
      testDir.mkdir()
    }
    testDir
  }

  def testsSourceFolders(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules.flatMap((module) => {
      module.roots.flatMap((root) => {
        if (root.kind == WeaveRootKind.TEST) {
          root.sources
        } else {
          Array.empty[File]
        }
      })
    })
  }

  def defaultTestSourceFolder(projectStructure: WeaveProjectStructure): Option[File] = {
    projectStructure.modules.flatMap(p => p.roots) //
      .find(rootStructure => rootStructure.kind == WeaveRootKind.TEST) //
      .map(testRootStructure => testRootStructure.defaultSourceFolder)
  }


  def mainRoot(module: WeaveModuleStructure): Option[WeaveRootStructure] = {
    module.roots.find(_.kind == WeaveRootKind.MAIN)
  }

  def mainSourceFolders(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules.flatMap((module) => {
      module.roots.flatMap((root) => {
        if (root.kind == WeaveRootKind.MAIN) {
          root.sources
        } else {
          Array.empty[File]
        }
      })
    })
  }

  def mainResourcesFolders(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules().flatMap((module) => {
      module.roots.flatMap((root) => {
        if (root.kind == WeaveRootKind.MAIN) {
          root.resources
        } else {
          Array.empty[File]
        }
      })
    })
  }

  /**
   * Returns the list of all target folders without tests only main
   */
  def mainTargetFolders(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules.flatMap((module) => {
      val targets: Array[WeaveTargetFolder] = module.targets
        .filter((target) => target.kind == WeaveTargetKind.CLASS || target.kind == WeaveTargetKind.RESOURCES)
      targets.flatMap(_.files())
    })
  }

  /**
   * Returns the list of all target folders including tests
   */
  def targetFolders(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules.flatMap((module) => {
      val targets: Array[WeaveTargetFolder] = module.targets
      targets.flatMap(_.files())
    })
  }

  def moduleOfFile(projectStructure: WeaveProjectStructure, url: String): Option[WeaveModuleStructure] = {
    val candidates = projectStructure.modules().filter(module => {
      module.roots.exists(root => {
        URLUtils.isChildOfAny(url, root.sources) || URLUtils.isChildOfAny(url, root.resources)
      }) || module.descriptors().exists((descriptor) => {
        URLUtils.toFile(url).exists(file => file.getAbsolutePath.equals(descriptor.getFile.getAbsolutePath))
      })
    })

    candidates.headOption
  }
}

