package org.mule.weave.lsp.services

import org.eclipse.lsp4j.FileChangeType
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolClient
import org.mule.weave.lsp.extension.protocol.PreviewResult
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.events.OnProjectStarted
import org.mule.weave.lsp.project.events.ProjectStartedEvent
import org.mule.weave.lsp.services.events.ActiveScenarioChangedEvent
import org.mule.weave.lsp.services.events.DocumentChangedEvent
import org.mule.weave.lsp.services.events.DocumentFocusChangedEvent
import org.mule.weave.lsp.services.events.FileChangedEvent._
import org.mule.weave.lsp.services.events.OnActiveScenarioChanged
import org.mule.weave.lsp.services.events.OnDocumentChanged
import org.mule.weave.lsp.services.events.OnDocumentFocused
import org.mule.weave.lsp.services.events.OnFileChanged
import org.mule.weave.lsp.services.exception.RunPreviewException
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.URLUtils.isChildOf
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl
import org.mule.weave.lsp.utils.WeaveASTQueryUtils
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.WeaveDocumentToolingService
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.slf4j.LoggerFactory

import java.util.Collections
import java.util.concurrent.Future
import java.util.concurrent.TimeUnit

class PreviewService(agentService: WeaveAgentService, weaveLanguageClient: DataWeaveProtocolClient, toolingServices: DataWeaveToolingService, jobManager: JobManagerService) extends ToolingService {
  private val logger = LoggerFactory.getLogger(getClass)
  private var eventBus: InternalEventBus = _
  private var projectKind: ProjectKind = _

  @volatile
  private var enableValue: Boolean = false
  private var pendingProjectStart: Option[VirtualFile] = None
  private var currentVfPreview: Option[VirtualFile] = None
  private val previewDebouncer = new Debouncer[NameIdentifier]

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.eventBus = eventBus
    this.projectKind = projectKind
    eventBus.register(DocumentChangedEvent.DOCUMENT_CHANGED, new OnDocumentChanged {
      override def onDocumentChanged(vf: VirtualFile): Unit = {
        if (enableValue && canRunPreview(vf)) {
          scheduleRunPreview(vf)
        }
      }
    })

    eventBus.register(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, new OnDocumentFocused {
      override def onDocumentFocused(vf: VirtualFile): Unit = {
        if (enableValue && canRunPreview(vf)) {
          scheduleRunPreview(vf)
        }
      }
    })

    eventBus.register(FILE_CHANGED_EVENT, new OnFileChanged {
      override def onFileChanged(uri: String, changeType: FileChangeType): Unit = {
        if (enableValue) {
          currentVfPreview.map(currentVFPreview => {
            projectKind.sampleDataManager().searchSampleDataFolderFor(currentVFPreview.getNameIdentifier).map(scenarioFolder => {
              if (isChildOf(uri, scenarioFolder) || isChildOf(toLSPUrl(scenarioFolder.getPath), uri)) {
                scheduleRunPreview(currentVFPreview)
              }
            })
          })
        }
      }
    })

    eventBus.register(ProjectStartedEvent.PROJECT_STARTED, new OnProjectStarted {
      override def onProjectStarted(metadata: ProjectMetadata): Unit = {
        if (pendingProjectStart.isDefined) {
          scheduleRunPreview(pendingProjectStart.get)
          pendingProjectStart = None
        }
      }
    })

    eventBus.register(ActiveScenarioChangedEvent.ACTIVE_SCENARIO_CHANGED, new OnActiveScenarioChanged {
      override def onActiveScenarioChanged(vf: VirtualFile): Unit = {
        if (enableValue && currentVfPreview.isDefined && currentVfPreview.get.url().equals(vf.url()) && canRunPreview(vf)) {
          scheduleRunPreview(vf)
        }
      }
    })
  }

  def runPreview(vf: VirtualFile): PreviewResult = {
    val fileUrl: String = vf.url()
    if (!projectKind.isStarted) {
      PreviewResult(
        uri = fileUrl,
        success = false,
        logs = Collections.emptyList(),
        errorMessage = "Project is not yet initialized.\nPreview is going to be executed once project initializes."
      )
    } else {
      val identifier: NameIdentifier = vf.getNameIdentifier
      val content: String = vf.read()
      logger.debug(s"Trigger run preview for `$identifier`.")
      agentService.run(identifier, content, fileUrl)
    }
  }

  def canRunPreview(documentToolingService: WeaveDocumentToolingService): Boolean = {
    if (documentToolingService == null) {
      false
    } else {
      canRunPreview(documentToolingService.file, () => fileKind(documentToolingService))
    }
  }
  
  def canRunPreview(vf: VirtualFile): Boolean = {
    canRunPreview(vf, () => fileKind(vf.url()))
  }

  private def canRunPreview(vf: VirtualFile, fileKindResolver: () => Option[String] ): Boolean = {
    if (vf == null) {
      false
    } else {
      val fileUrl = URLUtils.toURI(vf.url())
      if (fileUrl.exists(p => p.getScheme == "preview")) { //If file is a preview file then don't schedule anything

        false
      } else {
        val mayBeKind: Option[String] = fileKindResolver.apply()
        val isMapping: Boolean = mayBeKind.forall(kind => kind.equals(WeaveASTQueryUtils.MAPPING))
        !vf.readOnly() && isMapping
      }
    }
  }

  private def fileKind(fileUrl: String): Option[String] = {
    fileKind(toolingServices.openDocument(fileUrl))
  }

  private def fileKind(documentToolingService: WeaveDocumentToolingService): Option[String] = {
    val maybeAstNode = documentToolingService.ast().map(_.astNode)
    val mayBeKind = WeaveASTQueryUtils.fileKind(maybeAstNode)
    mayBeKind
  }

  def scheduleRunPreview(vf: VirtualFile, sync: Boolean = false): PreviewResult = {
    val fileUrl: String = vf.url()
    if (!canRunPreview(vf)) {
      val previewResult = PreviewResult(
        uri = fileUrl,
        success = false,
        logs = Collections.emptyList(),
        errorMessage = s"File `${vf.getNameIdentifier.name}` is not an executable mapping file."
      )
      if (sync) {
        return previewResult
      } else {
        weaveLanguageClient.showPreviewResult(previewResult)
      }
    } else if (!projectKind.isStarted) {
      if (sync) {
        throw new RunPreviewException("Project is not yet initialized, preview cannot be executed at this time.")
      } else {
        pendingProjectStart = Some(vf)
        val previewResult = runPreview(vf)
        weaveLanguageClient.showPreviewResult(previewResult)
      }
    } else {
      if (sync) {
        return runPreview(vf)
      } else {
        val identifier: NameIdentifier = vf.getNameIdentifier
        //Debounce the changes for 300ms
        previewDebouncer.debounce(identifier, () => {
          jobManager.execute((_: Status) => {
            weaveLanguageClient.showPreviewResult(runPreview(vf))
          }, "Running preview", s"Running preview of $identifier"
          )
        }, 300, TimeUnit.MILLISECONDS)
      }
    }
    currentVfPreview = Some(vf)
    null
  }

  def enable(): Unit = {
    this.enableValue = true
  }

  def disable(): Unit = {
    this.enableValue = false
    this.currentVfPreview = None
  }
}

import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.Executors

class Debouncer[T] {
  final private val scheduler = Executors.newSingleThreadScheduledExecutor
  final private val delayedMap = new ConcurrentHashMap[T, Future[_]]
  private val logger = LoggerFactory.getLogger(getClass)

  def debounce(key: T, runnable: Runnable, delay: Long, unit: TimeUnit): Unit = {
    val prev: Future[_] = delayedMap.put(key, scheduler.schedule(new Runnable() {
      override def run(): Unit = {
        try {
          runnable.run()
        }
        finally {
          delayedMap.remove(key)
        }
      }
    }, delay, unit))
    if (prev != null) {
      logger.info("Canceling previous execution.")
      prev.cancel(true)
    }
  }

  def shutdown(): Unit = {
    scheduler.shutdownNow
  }
}
