package org.mule.weave.lsp.services

import org.eclipse.lsp4j
import org.eclipse.lsp4j.DidChangeConfigurationParams
import org.eclipse.lsp4j.DidChangeWatchedFilesParams
import org.eclipse.lsp4j.Location
import org.eclipse.lsp4j.SymbolInformation
import org.eclipse.lsp4j.SymbolKind
import org.eclipse.lsp4j.WorkspaceSymbolParams
import org.eclipse.lsp4j.services.LanguageClient
import org.eclipse.lsp4j.services.WorkspaceService
import org.mule.weave.extension.api.WeaveLanguageServerBuilder.TextDocumentTranslator
import org.mule.weave.lsp.commands.CommandManagerService
import org.mule.weave.lsp.commands.Commands
import org.mule.weave.lsp.commands.InternalWeaveCommand
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolClient
import org.mule.weave.lsp.extension.protocol.SetContextParams
import org.mule.weave.lsp.extension.protocol.SetContextValue
import org.mule.weave.lsp.indexer.LSPWeaveIndexService
import org.mule.weave.lsp.project.DefaultProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.events.DocumentChangedEvent
import org.mule.weave.lsp.services.events.DocumentFocusChangedEvent
import org.mule.weave.lsp.services.events.DocumentOpenedEvent
import org.mule.weave.lsp.services.events.FileChangedEvent
import org.mule.weave.lsp.services.events.OnDocumentChanged
import org.mule.weave.lsp.services.events.OnDocumentFocused
import org.mule.weave.lsp.services.events.OnDocumentOpened
import org.mule.weave.lsp.utils.LSPConverters
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.indexing.IdentifierType
import org.mule.weave.v2.editor.indexing.LocatedResult
import org.mule.weave.v2.editor.indexing.WeaveIdentifier
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.slf4j.LoggerFactory

import java.util
import java.util.concurrent.CompletableFuture
import scala.collection.JavaConverters.seqAsJavaListConverter

trait WorkspaceServiceManager {
  def workspaceService(): WorkspaceService
  def uIService(): UIService

  def diagnosticsPublisherService(): DiagnosticsPublisherService
  def workspaceEditService(): WorkspaceEditService
  def textDocumentTranslatorService(): TextDocumentTranslatorService
  def configurationService(): ConfigurationService
  def commandManagerService(): CommandManagerService

  def onLanguageClientConnected(languageClient: LanguageClient)

  def standaloneContribution(workspaceFolderUri: String, projectKind: ProjectKind, workspaceServiceContributor: WorkspaceServiceContributor): Unit
  def onProjectKindCreated(workspaceFolderUri: String, projectKind: ProjectKind, workspaceServiceContributor: WorkspaceServiceContributor): Unit
  def onProjectKindRemoved(workspaceFolderUri: String): Option[ProjectKind]
  def projectKinds(): Array[ProjectKind]
  def projectKind(uri: String): Option[ProjectKind]

  def addWorkspaceServiceListeners(listener: WorkspaceServiceListener): Unit
  def removeWorkspaceServiceListeners(listener: WorkspaceServiceListener): Unit

  def getStandaloneProjectKind: Option[ProjectKind]
  def isStandaloneProjectFile(uri: String): Boolean
}

trait WorkspaceServiceContributor {
  def symbol(params: WorkspaceSymbolParams): CompletableFuture[util.List[_ <: SymbolInformation]]
  def didChangeConfiguration(params: DidChangeConfigurationParams): Unit
  def didChangeWatchedFiles(params: DidChangeWatchedFilesParams): Unit
}

/**
 * DataWeave Implementation of the LSP Workspace Service Contributor
 */
class DefaultBaseWorkspaceServiceContributor(projectKind: ProjectKind,
                                             projectMetadata: DefaultProjectMetadata,
                                             weaveProtocolClient: DataWeaveProtocolClient,
                                             commandManager: CommandManagerService,
                                             maybeTextDocumentTranslator: Option[TextDocumentTranslator] = Option.empty)
  extends WorkspaceServiceContributor {

  private val logger = LoggerFactory.getLogger(getClass)

  projectKind.eventBus().register(DocumentChangedEvent.DOCUMENT_CHANGED, new OnDocumentChanged {
    override def onDocumentChanged(vf: VirtualFile): Unit = {
      notifyContextChanged(vf)
    }
  })

  projectKind.eventBus().register(DocumentOpenedEvent.DOCUMENT_OPENED, new OnDocumentOpened {
    override def onDocumentOpened(vf: VirtualFile): Unit = {
      notifyContextChanged(vf)
    }
  })

  projectKind.eventBus().register(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, new OnDocumentFocused {
    override def onDocumentFocused(vf: VirtualFile): Unit = {
      notifyContextChanged(vf)
    }
  })

  private def notifyContextChanged(vf: VirtualFile): Unit = {
    val uri = maybeTextDocumentTranslator.map(translator => translator.translate(vf.url())).getOrElse(vf.url())
    val params = SetContextParams(commandManager.commands.map(command => {
      val isEnabled = command match {
        case command: InternalWeaveCommand => command.enabled(uri)
        case _ => true
      }
      SetContextValue(command.commandId() + Commands.COMMAND_ENABLED_SUFFIX, isEnabled, vf.url())
    }) asJava)
    weaveProtocolClient.setContext(params)
  }

  override def symbol(params: WorkspaceSymbolParams): CompletableFuture[util.List[_ <: SymbolInformation]] = {
    CompletableFuture.supplyAsync(() => {
      val resourceResolver: WeaveResourceResolver = projectKind.vfs.asResourceResolver
      val iterable: Iterable[LocatedResult[WeaveIdentifier]] = projectKind.toolingService(classOf[LSPWeaveIndexService]).searchSymbol(params.getQuery)
      val result = new util.ArrayList[SymbolInformation]()
      iterable.foreach((lr) => {
        val information = new SymbolInformation()
        information.setName(lr.value.value)
        val kind: SymbolKind = lr.value.idType match {
          case IdentifierType.FUNCTION => SymbolKind.Function
          case IdentifierType.VARIABLE => SymbolKind.Variable
          case IdentifierType.NAMESPACE => SymbolKind.Namespace
          case IdentifierType.ANNOTATION => SymbolKind.Interface
          case _ => SymbolKind.Field
        }
        information.setKind(kind)
        val name = lr.moduleName
        val moduleResource = resourceResolver.resolve(name)
        moduleResource match {
          case Some(weaveResource) => {
            val range = new lsp4j.Range()
            val startPosition: Position = weaveResource.positionOf(lr.value.startLocation)
            val endPosition: Position = weaveResource.positionOf(lr.value.endLocation)
            range.setStart(LSPConverters.toPosition(startPosition))
            range.setEnd(LSPConverters.toPosition(endPosition))
            val location = new Location(weaveResource.url(), range)
            information.setLocation(location)
            result.add(information)
          }
          case _ =>
        }
      })
      result
    })
  }

  override def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {
    projectMetadata.settings.update(params.getSettings)
  }

  override def didChangeWatchedFiles(params: DidChangeWatchedFilesParams): Unit = {
    params.getChanges.forEach((fe) => {
      if (projectKind.isProjectFile(fe.getUri)) {
        logger.debug("Changed Watched File : " + fe.getUri + " - " + fe.getType)
        projectKind.eventBus.fire(new FileChangedEvent(fe.getUri, fe.getType))
      } else {
        logger.debug("Ignored Watched File : " + fe.getUri + " - " + fe.getType)
      }
    })
  }

}
