package org.mule.weave.lsp.indexer

import org.mule.weave.extension.api.component.structure.WeaveProjectStructure
import org.mule.weave.lsp.indexer.events.IndexingFinishedEvent
import org.mule.weave.lsp.indexer.events.IndexingStartedEvent
import org.mule.weave.lsp.indexer.events.IndexingType
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProjectStructureHelper.isAProjectFile
import org.mule.weave.lsp.project.events.{OnProjectStructureChanged, ProjectStructureChangedEvent}
import org.mule.weave.lsp.services.ClientLoggerFactory
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.VFUtils
import org.mule.weave.lsp.vfs.ArtifactVirtualFileSystem
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.lsp.vfs.events.LibrariesAddedEvent
import org.mule.weave.lsp.vfs.events.LibrariesRemovedEvent
import org.mule.weave.lsp.vfs.events.OnLibrariesAdded
import org.mule.weave.lsp.vfs.events.OnLibrariesRemoved
import org.mule.weave.v2.editor.ChangeListener
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.editor.indexing.DefaultWeaveIndexer
import org.mule.weave.v2.editor.indexing.IdentifierKind
import org.mule.weave.v2.editor.indexing.LocatedResult
import org.mule.weave.v2.editor.indexing.WeaveDocument
import org.mule.weave.v2.editor.indexing.WeaveIdentifier
import org.mule.weave.v2.editor.indexing.WeaveIndexService
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.slf4j.LoggerFactory

import java.util
import java.util.concurrent.Callable
import java.util.concurrent.ForkJoinPool
import java.util.concurrent.ForkJoinTask
import scala.collection.JavaConverters.asJavaIteratorConverter
import scala.collection.JavaConverters.asScalaIteratorConverter
import scala.collection.concurrent.TrieMap

class LSPWeaveIndexService(loggerFactory: ClientLoggerFactory,
                           projectVirtualFileSystem: ProjectFileSystemService,
                           jobManagerService: JobManagerService
                          ) extends WeaveIndexService with ToolingService {

  private val logger = LoggerFactory.getLogger(classOf[LSPWeaveIndexService])
  private val clientLogger = loggerFactory.createLogger(classOf[LSPWeaveIndexService])

  /**
   * Contains all the root symbols inside each Module for each library (jar)
   *
   * MyLib.jar
   * |-> MyModule
   * |----> myVariable
   * |----> otherFunction
   */
  private val identifiersInLibraries: TrieMap[VirtualFileSystem, TrieMap[String, Array[LocatedResult[WeaveIdentifier]]]] = TrieMap()
  private val identifiersInProject: TrieMap[String, Array[LocatedResult[WeaveIdentifier]]] = TrieMap()

  /**
   * Contains the Logical names for each element
   */
  private val namesInLibraries: TrieMap[VirtualFileSystem, TrieMap[String, LocatedResult[WeaveDocument]]] = TrieMap()
  private val namesInProject: TrieMap[String, LocatedResult[WeaveDocument]] = TrieMap()

  private val indexedPool = new ForkJoinPool()
  private var projectKind: ProjectKind = _
  private var eventBus: InternalEventBus = _

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.eventBus = eventBus
    this.projectKind = projectKind
    projectVirtualFileSystem.changeListener(new ChangeListener {
      override def onDeleted(vf: VirtualFile): Unit = {
        eventBus.fire(new IndexingStartedEvent(IndexingType.Project))
        identifiersInProject.remove(vf.url())
        namesInProject.remove(vf.url())
        eventBus.fire(new IndexingFinishedEvent(IndexingType.Project))
      }

      override def onChanged(vf: VirtualFile): Unit = {
        val projectStructure = projectKind.structure()

        if (projectVirtualFileSystem.isInMemoryFile(vf.url()) || isAProjectFile(vf, projectStructure)) {
          eventBus.fire(new IndexingStartedEvent(IndexingType.Project))
          indexProjectFile(vf)
          eventBus.fire(new IndexingFinishedEvent(IndexingType.Project))
        }
      }

      override def onCreated(vf: VirtualFile): Unit = {
        val projectStructure: WeaveProjectStructure = projectKind.structure()
        if (projectVirtualFileSystem.isInMemoryFile(vf.url()) || isAProjectFile(vf, projectStructure)) {
          eventBus.fire(new IndexingStartedEvent(IndexingType.Project))
          indexProjectFile(vf)
          eventBus.fire(new IndexingFinishedEvent(IndexingType.Project))
        }
      }
    })

    eventBus.register(ProjectStructureChangedEvent.PROJECT_STRUCTURE_CHANGED, new OnProjectStructureChanged {
      override def onProjectStructureChanged(previousProjectStructure: WeaveProjectStructure, newProjectStructure: WeaveProjectStructure): Unit = {
        identifiersInProject.clear()
        namesInProject.clear()
        initialized()
      }
    })

    eventBus.register(LibrariesRemovedEvent.LIBRARIES_REMOVED, new OnLibrariesRemoved {
      override def onLibrariesRemoved(libs: Array[ArtifactVirtualFileSystem]): Unit = {
        libs.foreach(lib => {
          identifiersInLibraries.remove(lib)
          namesInLibraries.remove(lib)
        })
      }
    })

    eventBus.register(LibrariesAddedEvent.LIBRARIES_ADDED, new OnLibrariesAdded {
      override def onLibrariesAdded(libaries: Array[ArtifactVirtualFileSystem]): Unit = {
        jobManagerService.execute((status: Status) => {
          val forks: Array[Callable[Unit]] = libaries.map((vfs) => {
            new Callable[Unit] {
              override def call(): Unit = {
                if (status.canceled) {
                  return
                }
                val virtualFiles: Iterator[VirtualFile] = vfs.listFiles().asScala
                val vfsIdentifiers: TrieMap[String, Array[LocatedResult[WeaveIdentifier]]] = identifiersInLibraries.getOrElseUpdate(vfs, TrieMap())
                val vfsNames: TrieMap[String, LocatedResult[WeaveDocument]] = namesInLibraries.getOrElseUpdate(vfs, TrieMap())
                logger.debug(s"Start Indexing `${vfs.artifactId()}`.")
                val startTime = System.currentTimeMillis()
                virtualFiles.foreach((vf) => {
                  val indexer: DefaultWeaveIndexer = new DefaultWeaveIndexer()
                  //Only index DW files for now
                  //TODO figure out how to support other formats like .class or .raml etc
                  if (VFUtils.isDWFile(vf.url()) && indexer.parse(vf, ParsingContextFactory.createParsingContext(false))) {
                    vfsIdentifiers.put(vf.url(), index(vf, indexer))
                    vfsNames.put(vf.url(), LocatedResult(vf.getNameIdentifier, indexer.document()))
                  }
                })
                logger.debug(s"Indexing `${vfs.artifactId()}` took ${System.currentTimeMillis() - startTime}ms")
              }
            }
          })
          val start = System.currentTimeMillis()
          eventBus.fire(new IndexingStartedEvent(IndexingType.Dependencies))
          indexedPool.invokeAll(util.Arrays.asList(forks: _*))
          eventBus.fire(new IndexingFinishedEvent(IndexingType.Dependencies))
          clientLogger.logInfo(s"Indexing all libraries finished and took ${System.currentTimeMillis() - start}ms.")
        }, "Indexing DataWeave Libraries [" + projectKind.project().getName + "]", "Indexing Libraries")
      }
    })
  }


  override def initialized(): Unit = {
    eventBus.fire(new IndexingStartedEvent(IndexingType.Project))
    indexedPool.invoke(ForkJoinTask.adapt(
      new Runnable {
        override def run(): Unit = {
          val value: util.Iterator[VirtualFile] = projectVirtualFileSystem.listFiles()
          while (value.hasNext) {
            indexProjectFile(value.next())
          }
        }
      })
    )
    eventBus.fire(new IndexingFinishedEvent(IndexingType.Project))
  }

  private def indexProjectFile(vf: VirtualFile) = {
    val indexer: DefaultWeaveIndexer = new DefaultWeaveIndexer()
    try {
      if (VFUtils.isDWFile(vf.url()) && indexer.parse(vf, ParsingContextFactory.createParsingContext(false))) {
        identifiersInProject.put(vf.url(), index(vf, indexer))
        namesInProject.put(vf.url(), LocatedResult(vf.getNameIdentifier, indexer.document()))
      }
    } catch {
      case e: Exception => {
        //Don't stop if there is an exception indexing a file.
        clientLogger.logError(s"Error while trying to index `${vf.url()}`", e)
      }
    }
  }


  private def index(vf: VirtualFile, indexer: DefaultWeaveIndexer): Array[LocatedResult[WeaveIdentifier]] = {
    val nameIdentifier = vf.getNameIdentifier
    val weaveSymbols: Array[WeaveIdentifier] = indexer.identifiers()
    val locatedWeaveSymbols = weaveSymbols.map((symbol) => {
      LocatedResult[WeaveIdentifier](nameIdentifier, symbol)
    })
    locatedWeaveSymbols
  }


  def searchSymbol(getQuery: String): Iterable[LocatedResult[WeaveIdentifier]] = {
    identifiersInProject.values.flatMap((elements) => {
      elements.filter((e) => {
        val value = e.value.value
        e.value.kind == IdentifierKind.DEFINITION &&
          matchesQuery(getQuery, value)
      })
    }) ++
      identifiersInLibraries.values.flatMap((elements) => {
        elements.flatMap((e) => {
          e._2.filter((d) => {
            val value = d.value.value
            d.value.kind == IdentifierKind.DEFINITION &&
              matchesQuery(getQuery, value)
          })
        })
      })
  }

  private def matchesQuery(getQuery: String, value: String) = {
    value.contains(getQuery)
  }

  override def searchReferences(name: String): Array[LocatedResult[WeaveIdentifier]] = {
    val librarySymbols = identifiersInLibraries.values.flatMap((vf) => {
      val allSymbols = vf.values
      val result = searchIn(allSymbols, name, IdentifierKind.REFERENCE)
      result
    })
    val projectSymbols = searchIn(identifiersInProject.values, name, IdentifierKind.REFERENCE)
    (projectSymbols ++ librarySymbols).toArray
  }

  private def searchIn(identifiers: Iterable[Array[LocatedResult[WeaveIdentifier]]], name: String, kind: Int): Iterable[LocatedResult[WeaveIdentifier]] = {
    identifiers
      .flatMap((identifiers) => {
        identifiers.filter((identifier) => {
          identifier.value.kind == kind &&
            identifier.value.value.equals(name)
        })
      })
  }

  override def searchDefinitions(name: String): Array[LocatedResult[WeaveIdentifier]] = {
    val librarySymbols = identifiersInLibraries.values.flatMap((vf) => {
      val allSymbols = vf.values
      val result = searchIn(allSymbols, name, IdentifierKind.DEFINITION)
      result
    })
    val projectSymbols = searchIn(identifiersInProject.values, name, IdentifierKind.DEFINITION)
    (projectSymbols ++ librarySymbols).toArray
  }

  override def searchDocumentContainingName(pattern: String): util.Iterator[LocatedResult[WeaveDocument]] = {
    val libraryNames = namesInLibraries.values.flatMap((names) => {
      names.values.filter((name) => {
        name.moduleName.name.contains(pattern)
      })
    })
    val projectNames = namesInProject.values.filter((name) => {
      name.moduleName.name.contains(pattern)
    })
    (projectNames.toIterator ++ libraryNames.toIterator).asJava
  }
}
