package org.mule.weave.lsp.vfs

import org.eclipse.lsp4j.FileChangeType
import org.mule.weave.lsp.services.events.{FileChangedEvent, OnFileChanged}
import org.mule.weave.lsp.utils.{InternalEventBus, URLUtils, VFUtils}
import org.mule.weave.lsp.vfs.resource.CustomLoaderResourceResolver
import org.mule.weave.v2.editor.{ChangeListener, VirtualFile, VirtualFileSystem}
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.WeaveResourceResolver

import java.io.File
import java.util
import scala.collection.JavaConverters.asJavaIteratorConverter
import scala.collection.mutable.ArrayBuffer
import scala.io.{BufferedSource, Source}

/**
  * This virtual file system handles all the files that are generated by the custom loader resource resolver
  *
  */
class CustomLoaderVirtualFileSystem(resourceResolver: CustomLoaderResourceResolver, eventBus: InternalEventBus) extends ReadOnlyVirtualFileSystem {

  val listeners: ArrayBuffer[ChangeListener] = ArrayBuffer[ChangeListener]()

  eventBus.register(FileChangedEvent.FILE_CHANGED_EVENT, new OnFileChanged {
    override def onFileChanged(uri: String, changeType: FileChangeType): Unit = {
      if (uri.endsWith("." + resourceResolver.fileExtension)) {
        val maybeVF = file(uri, resourceResolver.nameIdentifierOfOriginalFile)
        Option(maybeVF)
          .foreach(virtualFile => {
            listeners.foreach(_.onChanged(virtualFile))
          })
      }
    }
  })

  private def file(path: String, uriToIdentifier: String => Option[NameIdentifier]): VirtualFile ={
    val theFile: Option[File] = URLUtils.toFile(path)
    if (theFile.isEmpty) {
      null
    } else {
      if (theFile.get.exists()) {
        val maybeIdentifier = uriToIdentifier(theFile.get.toURI.toString)
        if (maybeIdentifier.isDefined) {
          new RemoteVirtualFile(theFile.get, this, maybeIdentifier.get)
        } else {
          null
        }
      } else {
        null
      }
    }
  }

  override def file(path: String): VirtualFile = {
    file(path, resourceResolver.nameIdentifierOfStub)
  }

  override def changeListener(cl: ChangeListener): Unit = {
    listeners.+=(cl)
  }

  override def onChanged(vf: VirtualFile): Unit = {
    listeners.foreach((cl) => cl.onChanged(vf))
  }

  override def removeChangeListener(service: ChangeListener): Unit = {
    listeners.remove(listeners.indexOf(service))
  }

  override def asResourceResolver: WeaveResourceResolver = {
    resourceResolver
  }

  override def listFiles(): util.Iterator[VirtualFile] = {
    VFUtils.listFiles(resourceResolver.stubsModulesFolder, this).asJava
  }

  class RemoteVirtualFile(file: File, val fs: VirtualFileSystem, nameIdentifier: NameIdentifier) extends VirtualFile {

    override def read(): String = {
      val source: BufferedSource = Source.fromFile(file, "UTF-8")
      try {
        source.mkString
      } catch {
        case e: Exception => {
          throw new RuntimeException(s"Unable to read content of `${file.toURI}`", e)
        }
      } finally {
        source.close()
      }
    }

    override def write(content: String): Boolean = {
      false
    }

    override def readOnly(): Boolean = true

    override def url(): String = {
      file.toURI.toString
    }

    override def getNameIdentifier: NameIdentifier = {
      nameIdentifier
    }

    override def path(): String = file.getPath
  }
}
