package org.mule.weave.lsp.services

import org.eclipse.lsp4j.CodeAction
import org.eclipse.lsp4j.CodeActionParams
import org.eclipse.lsp4j.CodeLens
import org.eclipse.lsp4j.CodeLensParams
import org.eclipse.lsp4j.Command
import org.eclipse.lsp4j.CompletionItem
import org.eclipse.lsp4j.CompletionList
import org.eclipse.lsp4j.CompletionParams
import org.eclipse.lsp4j.DefinitionParams
import org.eclipse.lsp4j.DidChangeTextDocumentParams
import org.eclipse.lsp4j.DidCloseTextDocumentParams
import org.eclipse.lsp4j.DidOpenTextDocumentParams
import org.eclipse.lsp4j.DidSaveTextDocumentParams
import org.eclipse.lsp4j.DocumentFormattingParams
import org.eclipse.lsp4j.DocumentRangeFormattingParams
import org.eclipse.lsp4j.DocumentSymbol
import org.eclipse.lsp4j.DocumentSymbolParams
import org.eclipse.lsp4j.FoldingRange
import org.eclipse.lsp4j.FoldingRangeRequestParams
import org.eclipse.lsp4j.Hover
import org.eclipse.lsp4j.HoverParams
import org.eclipse.lsp4j.Location
import org.eclipse.lsp4j.LocationLink
import org.eclipse.lsp4j.ReferenceParams
import org.eclipse.lsp4j.RenameParams
import org.eclipse.lsp4j.SignatureHelp
import org.eclipse.lsp4j.SignatureHelpParams
import org.eclipse.lsp4j.SymbolInformation
import org.eclipse.lsp4j.TextDocumentItem
import org.eclipse.lsp4j.TextEdit
import org.eclipse.lsp4j.WorkspaceEdit
import org.eclipse.lsp4j.jsonrpc.ResponseErrorException
import org.eclipse.lsp4j.jsonrpc.messages
import org.eclipse.lsp4j.jsonrpc.messages.ResponseError
import org.eclipse.lsp4j.jsonrpc.messages.ResponseErrorCode
import org.eclipse.lsp4j.jsonrpc.messages.{Either => JEither}
import org.mule.weave.extension.api.WeaveLanguageServerBuilder.TextDocumentTranslator
import org.mule.weave.lsp.extension.services.DidFocusChangeParams
import org.mule.weave.lsp.extension.services.WeaveTextDocumentService
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.vfs.LibrariesVirtualFileSystem

import java.util
import java.util.Collections
import java.util.concurrent.CompletableFuture
import scala.collection.concurrent.TrieMap

class DataWeaveDocumentServiceDispatcher(supportCodeLens: Boolean = true,
                                         maybeTextDocumentTranslator: Option[TextDocumentTranslator] = Option.empty,
                                         fallbackDocumentService: (String) => WeaveTextDocumentService = (textDocumentUri) => null)
  extends WeaveTextDocumentService  {

  private val projectDocumentServices: TrieMap[ProjectKind, WeaveTextDocumentService] = TrieMap()

  def onProjectKindCreated(projectKind: ProjectKind, weaveTextDocumentService: WeaveTextDocumentService) = {
    this.projectDocumentServices += ((projectKind, weaveTextDocumentService))
  }

  def onProjectRemoved(projectKind: ProjectKind) = {
    projectDocumentServices.remove(projectKind)
  }

  private def weaveTextDocumentServiceFor(textDocumentUri: String) = {
    val projectFileUri = maybeTextDocumentTranslator.map(translator => translator.translate(textDocumentUri)).getOrElse(textDocumentUri)

    val maybeDocumentService = projectDocumentServices
      .find(service => service._1.isProjectFile(projectFileUri) || service._1.toolingService(classOf[LibrariesVirtualFileSystem]).file(projectFileUri) != null)
      .map(service => service._2)
    if (maybeDocumentService.isDefined) {
      maybeDocumentService
    } else {
      // Case of playground mode
      Option.apply(fallbackDocumentService.apply(projectFileUri))
    }
  }

  //FS Changes
  override def didOpen(params: DidOpenTextDocumentParams): Unit = {
    val textDocument: TextDocumentItem = params.getTextDocument
    val uri: String = textDocument.getUri

    weaveTextDocumentServiceFor(uri) match {
      case Some(value) => value.didOpen(params)
      case None =>
    }
  }

  override def didChange(params: DidChangeTextDocumentParams): Unit = {
    val textDocument = params.getTextDocument
    val uri = textDocument.getUri

    weaveTextDocumentServiceFor(uri) match {
      case Some(value) => value.didChange(params)
      case None =>
    }
  }

  override def didClose(params: DidCloseTextDocumentParams): Unit = {
    val uri = params.getTextDocument.getUri

    weaveTextDocumentServiceFor(uri) match {
      case Some(value) => value.didClose(params)
      case None =>
    }
  }

  override def didSave(params: DidSaveTextDocumentParams): Unit = {
    val uri = params.getTextDocument.getUri

    weaveTextDocumentServiceFor(uri) match {
      case Some(value) => value.didSave(params)
      case None =>
    }
  }

  override def didFocusChange(params: DidFocusChangeParams): Unit = {
    val uri: String = params.textDocumentIdentifier.getUri

    weaveTextDocumentServiceFor(uri) match {
      case Some(value) => value.didFocusChange(params)
      case None =>
    }
  }

  override def completion(position: CompletionParams): CompletableFuture[messages.Either[util.List[CompletionItem], CompletionList]] = {
    weaveTextDocumentServiceFor(position.getTextDocument.getUri) match {
      case Some(value: DataWeaveDocumentService) => value.completion(position)
      case _ =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request completion(${position.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def signatureHelp(params: SignatureHelpParams): CompletableFuture[SignatureHelp] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.signatureHelp(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request signatureHelp(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def codeLens(params: CodeLensParams): CompletableFuture[util.List[_ <: CodeLens]] = {
    if (supportCodeLens) {
      weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
        case Some(value) => value.codeLens(params)
        case None =>
          val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request codeLens(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
          CompletableFuture.failedFuture(new ResponseErrorException(error))
      }
    } else {
      CompletableFuture.completedFuture(Collections.emptyList)
    }
  }

  override def resolveCodeLens(unresolved: CodeLens): CompletableFuture[CodeLens] = {
    CompletableFuture.completedFuture(unresolved)
  }

  override def foldingRange(params: FoldingRangeRequestParams): CompletableFuture[util.List[FoldingRange]] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.foldingRange(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request foldingRange(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def codeAction(params: CodeActionParams): CompletableFuture[util.List[JEither[Command, CodeAction]]] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.codeAction(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request codeAction(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def resolveCompletionItem(unresolved: CompletionItem): CompletableFuture[CompletionItem] = {
    CompletableFuture.completedFuture(unresolved)
  }

  override def hover(params: HoverParams): CompletableFuture[Hover] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.hover(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request hover(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def documentSymbol(params: DocumentSymbolParams): CompletableFuture[util.List[messages.Either[SymbolInformation, DocumentSymbol]]] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.documentSymbol(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request documentSymbol(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def definition(params: DefinitionParams): CompletableFuture[messages.Either[util.List[_ <: Location], util.List[_ <: LocationLink]]] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.definition(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request definition(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def formatting(params: DocumentFormattingParams): CompletableFuture[java.util.List[_ <: TextEdit]] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.formatting(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request formatting(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def rangeFormatting(params: DocumentRangeFormattingParams): CompletableFuture[util.List[_ <: TextEdit]] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.rangeFormatting(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request rangeFormatting(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def rename(params: RenameParams): CompletableFuture[WorkspaceEdit] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.rename(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request rename(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

  override def references(params: ReferenceParams): CompletableFuture[util.List[_ <: Location]] = {
    weaveTextDocumentServiceFor(params.getTextDocument.getUri) match {
      case Some(value) => value.references(params)
      case None =>
        val error = new ResponseError(ResponseErrorCode.InvalidRequest, s"Invalid request references(${params.getTextDocument.getUri}), URI does not belong to the ProjectKinds registered.", null)
        CompletableFuture.failedFuture(new ResponseErrorException(error))
    }
  }

}
