package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.Command
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.mule.weave.lsp.commands.Commands.argAsBooleanOrDefault
import org.mule.weave.lsp.commands.Commands.argAsString
import org.mule.weave.lsp.services.PreviewService
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.services.exception.RunPreviewException
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem

import java.util
import scala.collection.JavaConverters

class RunPreviewCommand(workspaceServiceManager: WorkspaceServiceManager, uIService: UIService) extends InternalWeaveCommand {
  override def commandId(): String = Commands.DW_RUN_PREVIEW

  override def execute(params: ExecuteCommandParams): Object = {
    val uri = argAsString(params.getArguments, 0)
    val sync = argAsBooleanOrDefault(params.getArguments, 1, false)

    val maybeProjectKind = workspaceServiceManager.projectKind(uri).orElse(workspaceServiceManager.getStandaloneProjectKind)
    if (maybeProjectKind.isDefined) {
      val projectKind = maybeProjectKind.get

      val previewService: PreviewService = projectKind.toolingService(classOf[PreviewService])
      val virtualFile: VirtualFileSystem = projectKind.vfs()

      val file: VirtualFile = virtualFile.file(uri)
      if (file != null) {
        val result = previewService.scheduleRunPreview(file, sync)
        if (sync) {
          return JavaConverters.mapAsJavaMap(result.toMap)
        }
      } else {
        val message = s"Unable to run preview because file `$uri` was not found."
        if (sync) {
          throw new RunPreviewException(message)
        } else {
          uIService.showMessage(new MessageParams(MessageType.Warning, message))
        }
      }
    } else {
      uIService.showMessage(new MessageParams(MessageType.Error, s"No project is associated to: ${uri}. ${commandId} cannot be executed"))
    }
    null
  }

  override def name(): String = "Runs Preview."

  override def description(params: ExecuteCommandParams): String = "Running Preview."

  override def enabled(uri: String): Boolean = {
    val maybeProjectKind = workspaceServiceManager.projectKind(uri).orElse(workspaceServiceManager.getStandaloneProjectKind)
    if (maybeProjectKind.isDefined) {
      val projectKind = maybeProjectKind.get
      projectKind.toolingService(classOf[PreviewService]).canRunPreview(projectKind.toolingService(classOf[ProjectFileSystemService]).file(uri))
    } else {
      false
    }
  }
}

object RunPreviewCommand {
  val LABEL: String = "Run Preview"

  def createCommand(uri: String, sync: Boolean = false): Command = {
    new Command(LABEL, Commands.DW_RUN_PREVIEW, util.Arrays.asList(uri, sync.toString))
  }

}
