package org.mule.weave.lsp.project.components

import org.mule.weave.dsp.JavaExecutableHelper
import org.mule.weave.dsp.LauncherConfig
import org.mule.weave.dsp.RunMappingConfiguration
import org.mule.weave.dsp.RunWTFConfiguration
import org.mule.weave.extension.api.component.dependency.WeaveDependencyComponent
import org.mule.weave.extension.api.component.structure.WeaveProjectStructure
import org.mule.weave.lsp.extension.protocol.LaunchConfiguration.DATA_WEAVE_CONFIG_TYPE_NAME
import org.mule.weave.lsp.extension.protocol.LaunchConfiguration.WITF_CONFIG_TYPE_NAME
import org.mule.weave.lsp.extension.protocol.LaunchConfiguration.WTF_CONFIG_TYPE_NAME
import org.mule.weave.lsp.extension.protocol.LaunchConfiguration
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickItem
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickParams
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.JavaWeaveLauncher.WEAVE_RUNNER_MAIN_CLASS
import org.mule.weave.lsp.project.components.JavaWeaveLauncher.buildJavaProcessBaseArgs
import org.mule.weave.lsp.project.components.ProjectStructureHelper.defaultTestSourceFolder
import org.mule.weave.lsp.services.ClientLoggerFactory
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.utils.Icons
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.WeaveDirectoryUtils
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.v2.agent.client.tcp.TcpClientProtocol
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

import java.io.File
import java.io.FileNotFoundException
import java.io.Reader
import java.util
import java.util.Properties
import scala.collection.JavaConverters.asScalaIteratorConverter
import scala.collection.JavaConverters.iterableAsScalaIterableConverter
import scala.io.Source
import scala.util.Try

/**
 * Handles the initialization of the weave process
 */
trait ProcessLauncher {

  type ConfigType <: LauncherConfig

  /**
   * Parses the arguments and build a LauncherConfig for this Launcher
   *
   * @param args The args to be parsed
   * @return The parsed Config
   */
  def parseArgs(args: Map[String, AnyRef]): ConfigType

  /**
   * Launches a process
   *
   * @param config The config taken from the parseArgs
   */
  def launch(config: ConfigType, debugging: Boolean): Option[Process]

}

object ProcessLauncher {
  def createLauncherByType(configType: String, projectKind: ProjectKind,
                           clientLogger: ClientLoggerFactory,
                           uIService: UIService,
                           vfs: ProjectFileSystemService): ProcessLauncher = {
    configType match {
      case DATA_WEAVE_CONFIG_TYPE_NAME => new DefaultWeaveLauncher(projectKind, clientLogger, uIService, vfs)
      case WTF_CONFIG_TYPE_NAME => new WTFLauncher(projectKind, clientLogger, uIService, vfs)
      case WITF_CONFIG_TYPE_NAME => new DefaultWeaveLauncher(projectKind, clientLogger, uIService, vfs)
      case _ => throw new RuntimeException(s"Unable to found a valid launcher for ${configType}.")
    }
  }
}

case class ClassPathConfig(shouldIncludeTarget: Boolean = true,
                           shouldIncludeSources: Boolean = false,
                           jars: Seq[String] = Seq.empty)

object JavaWeaveLauncher {

  val WEAVE_RUNNER_MAIN_CLASS = "org.mule.weave.v2.runtime.utils.WeaveRunner"

  def buildJavaProcessBaseArgs(projectKind: ProjectKind, configs: ClassPathConfig, mainClass: String, properties: Seq[String] = Seq()): util.ArrayList[String] = {
    val javaHome = JavaExecutableHelper.currentJavaHome()
    val javaExec = new File(new File(javaHome, "bin"), "java")
    val args = new util.ArrayList[String]()
    args.add(javaExec.toString)
    //We should take args from the user?
    args.add("-Xms64m")
    args.add("-Xmx2G")
    args.add("-XX:+HeapDumpOnOutOfMemoryError")
    if (java.lang.Boolean.getBoolean("com.dw.debugAgent")) {
      args.add("-agentlib:jdwp=transport=dt_socket,server=y,suspend=n,address=7010")
    }

    ///
    args.add("-cp")
    val dependencyManager: WeaveDependencyComponent = projectKind.dependencyManager()
    var classpath: Array[String] = dependencyManager.dependencies().map((dep) => {
      dep.file.getAbsolutePath
    })
    // Add jars from ClassPathConfig
    classpath = classpath ++ configs.jars
    val projectStructure: WeaveProjectStructure = projectKind.structure()
    if (configs.shouldIncludeTarget) {
      //If there is a target use the target folder, if not we need to use the root folders
      val targets: Array[String] = ProjectStructureHelper.targetFolders(projectStructure).map(_.getAbsolutePath)
      classpath = classpath ++ targets
    }

    if (configs.shouldIncludeSources) {
      val sources: Array[String] = projectStructure.modules.flatMap((module) => {
        module.roots.flatMap((r) => {
          val sources = r.sources.map((s) => {
            s.getAbsolutePath
          })
          val resources = r.resources.map((s) => {
            s.getAbsolutePath
          })
          sources ++ resources
        })
      })
      classpath = classpath ++ sources
    }

    args.add(classpath.mkString(File.pathSeparator))

    /// Common system properties
    //    args.add(s"-Duser.dir='${projectKind.structure().projectHome.getAbsolutePath}'")
    ///
    properties.foreach(args.add)
    defaultTestSourceFolder(projectStructure).map(dwTestFile => args.add(s"-DdwtestDir=${dwTestFile.getAbsolutePath}"))
    WeaveDirectoryUtils.wtfResourcesTestFolder(projectStructure).map(dwTestFile => args.add(s"-DdwtestResources=${dwTestFile.getAbsolutePath}"))
    args.add(mainClass)
    args
  }
}

class DefaultWeaveLauncher(projectKind: ProjectKind,
                           loggerFactory: ClientLoggerFactory,
                           uIService: UIService,
                           vfs: ProjectFileSystemService) extends ProcessLauncher {

  val icon = Icons.vscode

  private val clientLogger = loggerFactory.createLogger(classOf[DefaultWeaveLauncher])

  override type ConfigType = RunMappingConfiguration


  override def launch(config: RunMappingConfiguration, debugging: Boolean): Option[Process] = {

    val builder = new ProcessBuilder()
    val args: util.ArrayList[String] = buildJavaProcessBaseArgs(projectKind, ClassPathConfig(), WEAVE_RUNNER_MAIN_CLASS)

    if (debugging) {
      args.add("-debug")
    }

    var nameIdentifier: Option[String] = None
    val maybeMapping: Option[String] = config.mayBeMapping
    maybeMapping match {
      case Some(mappingNameID) if (mappingNameID.nonEmpty) => {
        nameIdentifier = maybeMapping
      }
      case _ => {
        val sourceFolders = ProjectStructureHelper.mainSourceFolders(projectKind.structure())
        val items: Array[WeaveQuickPickItem] = vfs.listFiles().asScala
          .filter((vf) => {
            //Filter for test only files
            vf.url().endsWith(".dwl") && URLUtils.isChildOfAny(vf.url(), sourceFolders)
          })
          .map((s) => {
            WeaveQuickPickItem(s.getNameIdentifier.toString, icon.file + s.getNameIdentifier.toString)
          }).toArray
        val result = uIService.weaveQuickPick(WeaveQuickPickParams(
          items = util.Arrays.asList(items: _*),
          title = "Select The DataWeave Script To Run"
        )).get()
        if (!result.cancelled) {
          nameIdentifier = Option(result.itemsId.get(0))
        }
      }
    }
    if (nameIdentifier.isDefined) {
      val theNameIdentifier = NameIdentifier(nameIdentifier.get)
      var scenarioPath: Option[String] = None

      val sampleManager: SampleDataComponent = projectKind.sampleDataManager()
      config.scenario match {
        case Some(scenario) if (scenario.nonEmpty) => {
          val maybeScenario = sampleManager.searchScenarioByName(theNameIdentifier, scenario)
          if (maybeScenario.isEmpty) {
            scenarioPath = askToPickScenario(theNameIdentifier)
          } else {
            scenarioPath = maybeScenario.map(_.file.getAbsolutePath)
          }
        }
        case _ => {
          val scenarios = sampleManager.listScenarios(theNameIdentifier)
          if (scenarios.length == 1) {
            scenarioPath = Some(scenarios.head.file.getAbsolutePath)
          } else {
            scenarioPath = askToPickScenario(theNameIdentifier)
          }
        }
      }
      if (scenarioPath.isDefined) {
        args.add("-scenario")
        args.add(scenarioPath.get)
      }

      args.add(nameIdentifier.get)
      builder.command(args)
      clientLogger.logDebug(s"Executing: ${args.asScala.mkString(" ")}")
      Some(builder.start())
    } else {
      None
    }
  }

  private def askToPickScenario(theNameIdentifier: NameIdentifier): Option[String] = {
    val scenarios: Array[Scenario] = projectKind.sampleDataManager().listScenarios(theNameIdentifier)
    if (scenarios.nonEmpty) {
      val items: Array[WeaveQuickPickItem] = scenarios.map((s) => {
        WeaveQuickPickItem(s.file.getAbsolutePath, s.name)
      })
      val response = uIService.weaveQuickPick(WeaveQuickPickParams(
        items = util.Arrays.asList(items: _*),
        title = "Select The Sample Data To Use"
      ))
      val result = response.get()
      if (result.cancelled) {
        None
      } else {
        Some(result.itemsId.get(0))
      }
    } else {
      None
    }
  }

  override def parseArgs(args: Map[String, AnyRef]): RunMappingConfiguration = {
    RunMappingConfiguration(
      args.get(LaunchConfiguration.MAIN_FILE_NAME).map(_.toString),
      args.get("scenario").map(_.toString),
      args.get(LaunchConfiguration.BUILD_BEFORE_PROP_NAME).forall((value) => value.toString == "true"),
      TcpClientProtocol.DEFAULT_PORT,
      args.get(LaunchConfiguration.TEST_RUNNER_ENV_VAR_FILE).map(path => new File(path.toString))
    )
  }
}


class WTFLauncher(projectKind: ProjectKind,
                  loggerFactory: ClientLoggerFactory,
                  uIService: UIService,
                  vfs: ProjectFileSystemService) extends ProcessLauncher {

  val icon = Icons.vscode

  private val clientLogger = loggerFactory.createLogger(classOf[WTFLauncher])

  override type ConfigType = RunWTFConfiguration

  override def launch(config: RunWTFConfiguration, debugging: Boolean): Option[Process] = {
    val builder = new ProcessBuilder()
    val props: util.ArrayList[String] = new util.ArrayList()
    if (config.dryRun) {
      props.add("-DskipAll=true")
    }
    config.testToRun match {
      case Some(testToRun) if testToRun.nonEmpty =>
        props.add(s"-DtestToRun='$testToRun'")
      case _ =>
      // Nothing to do
    }

    val args: util.ArrayList[String] = buildJavaProcessBaseArgs(projectKind, ClassPathConfig(), WEAVE_RUNNER_MAIN_CLASS, props.asScala.toSeq)
    args.add("--wtest")

    if (debugging) {
      args.add("-debug")
    }

    val launchProcess: Boolean = (config.dwTestFolder, config.mayBeTests) match {
      case (Some(dwTestFolder), _) =>
        args.add("-R")
        args.add(dwTestFolder)
        true

      case (None, Some(theTests)) =>
        val tests = theTests.split(",")
        tests.foreach(theTest => {
          args.add("-test")
          args.add(theTest)
        })
        true

      case (None, None) =>
        val items: Array[WeaveQuickPickItem] = vfs.listFiles().asScala
          .filter(vf => vf.url().endsWith(".dwl"))
          .map(s => WeaveQuickPickItem(s.getNameIdentifier.toString, icon.file + s.getNameIdentifier.toString)).toArray
        val result = uIService.weaveQuickPick(
          WeaveQuickPickParams(
            items = util.Arrays.asList(items: _*),
            title = "Select The Test To Run"
          )).get()
        if (!result.cancelled) {
          args.add("-test")
          args.add(result.itemsId.get(0))
          true
        } else {
          clientLogger.logDebug("No Test specified")
          false
        }
    }

    if (launchProcess) {
      args.add("-testlistener")
      args.add("json")
      builder.command(args)
      config.testRunnerEnvVarFile match {
        case Some(file: File) => {
          var reader: Reader = null
          try {
            reader = Source.fromFile(file).bufferedReader()
            clientLogger.logInfo("Launching WTF with \"testRunnerEnvVarFile\": " + file.getAbsolutePath)
            val properties = new Properties()
            properties.load(reader)
            properties.entrySet().stream()
              .forEach(e => builder.environment().put(e.getKey.asInstanceOf[String], e.getValue.asInstanceOf[String]))
          } catch {
            case _: FileNotFoundException => clientLogger.logInfo("No test runner environment variable file found.")
            case t: Throwable => clientLogger.logWarning("Could not read \"testRunnerEnvVarFile\" from file: " + file.getAbsolutePath, t)
          } finally {
            if (reader != null) {
              Try(reader.close())
            }
          }
        }
        case None =>
      }
      Some(builder.start())
    } else {
      None
    }
  }

  override def parseArgs(args: Map[String, AnyRef]): RunWTFConfiguration = {
    RunWTFConfiguration(
      mayBeTests = args.get(LaunchConfiguration.MAIN_FILE_NAME).map(_.toString),
      testToRun = args.get(LaunchConfiguration.TEST_TO_RUN).map(_.toString),
      buildBefore = args.get(LaunchConfiguration.BUILD_BEFORE_PROP_NAME).forall((value) => value.toString == "true"),
      debuggerPort = TcpClientProtocol.DEFAULT_PORT,
      dwTestFolder = None,
      testRunnerEnvVarFile = args.get(LaunchConfiguration.TEST_RUNNER_ENV_VAR_FILE).map(path => new File(path.toString))
    )
  }
}

