package org.mule.weave.lsp

import org.eclipse.lsp4j.CodeLensOptions
import org.eclipse.lsp4j.CompletionOptions
import org.eclipse.lsp4j.DidChangeConfigurationParams
import org.eclipse.lsp4j.DidChangeWorkspaceFoldersParams
import org.eclipse.lsp4j.InitializeParams
import org.eclipse.lsp4j.InitializeResult
import org.eclipse.lsp4j.InitializedParams
import org.eclipse.lsp4j.ServerCapabilities
import org.eclipse.lsp4j.ServerInfo
import org.eclipse.lsp4j.SignatureHelpOptions
import org.eclipse.lsp4j.TextDocumentSyncKind
import org.eclipse.lsp4j.WorkspaceFolder
import org.eclipse.lsp4j.WorkspaceFoldersOptions
import org.eclipse.lsp4j.WorkspaceServerCapabilities
import org.eclipse.lsp4j.jsonrpc.services.JsonDelegate
import org.eclipse.lsp4j.services.LanguageClient
import org.eclipse.lsp4j.services.LanguageServer
import org.eclipse.lsp4j.services.WorkspaceService
import org.mule.dx.platform.api.protocol.ClientAware
import org.mule.weave.extension.api.WeaveLanguageServerBuilder
import org.mule.weave.lsp.extension.client.LanguageClientDelegate
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolService
import org.mule.weave.lsp.extension.services.WeaveTextDocumentService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.project.DefaultProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.ProjectKindContext
import org.mule.weave.lsp.project.ProjectKindFactory
import org.mule.weave.lsp.project.ProjectSettings
import org.mule.weave.lsp.services.DiagnosticsPublisher
import org.mule.weave.lsp.services._
import org.mule.weave.lsp.services.delegate.TextDocumentServiceDelegate
import org.slf4j.LoggerFactory

import java.util
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutorService

class EmbeddedWeaveLanguageServer(projectKindFactory: ProjectKindFactory,
                                  configuration: EmbeddedLanguageServerConfiguration,
                                  protocolService: DataWeaveProtocolService,
                                  workspaceServiceManager: WorkspaceServiceManager,
                                  jobManagerService: JobManagerService)
  extends LanguageServer with ClientAware[LanguageClient] with WorkspaceServiceListener {

  private val logger = LoggerFactory.getLogger(classOf[EmbeddedWeaveLanguageServer])

  private val textDocumentService: TextDocumentServiceDelegate = new TextDocumentServiceDelegate()
  private val textDocumentServiceImpl: DataWeaveDocumentServiceDispatcher =
    new DataWeaveDocumentServiceDispatcher(configuration.codeLens, configuration.textDocumentLocator)

  private val languageClient: LanguageClientDelegate = new LanguageClientDelegate()

  private var clientLoggerFactory: ClientLoggerFactory = _
  private var clientLogger: ClientLogger = _

  private val workspaceSettings: ProjectSettings = ProjectSettings()

  private val workspaceEditApplier = new WorkspaceEditApplier(languageClient, configuration.fileScheme)
  private val diagnosticsPublisher = new DiagnosticsPublisher(languageClient, configuration.fileScheme)

  override def initialize(params: InitializeParams): CompletableFuture[InitializeResult] = {
    logger.debug(s"Initialize ${configuration.lspID}")
    CompletableFuture.supplyAsync(() => {
      logger.debug(s"Initializing ${configuration.lspID}")
      Runtime.getRuntime.addShutdownHook(new Thread(() => this.shutdown().get()))
      clientLoggerFactory = new ClientLoggerFactory(languageClient)
      clientLogger = clientLoggerFactory.createLogger(classOf[WeaveLanguageServer], configuration.lspID)

      workspaceServiceManager.addWorkspaceServiceListeners(this)

      workspaceSettings.load(params.getInitializationOptions)

      if (params.getWorkspaceFolders != null) {
        params.getWorkspaceFolders.forEach(workspaceFolder => discoverAndRegisterProjectKind(workspaceFolder))
      }

      textDocumentService.delegate = textDocumentServiceImpl

      if (configuration.fileScheme.isDefined) {
        workspaceServiceManager.diagnosticsPublisherService().addPublisher(diagnosticsPublisher)
        workspaceServiceManager.workspaceEditService().addApplier(workspaceEditApplier)
        workspaceServiceManager.textDocumentTranslatorService().addTranslator(configuration.fileScheme.get, configuration.textDocumentLocator.get)
      }

      val capabilities = new ServerCapabilities
      capabilities.setTextDocumentSync(TextDocumentSyncKind.Full)
      capabilities.setCompletionProvider(new CompletionOptions(true, java.util.Arrays.asList(".")))
      capabilities.setHoverProvider(true)
      capabilities.setDocumentSymbolProvider(true)
      capabilities.setDefinitionProvider(true)
      capabilities.setDocumentFormattingProvider(true)
      capabilities.setFoldingRangeProvider(true)
      capabilities.setCodeActionProvider(configuration.codeActions)
      capabilities.setWorkspaceSymbolProvider(true)
      capabilities.setCodeLensProvider(new CodeLensOptions(configuration.codeLens))
      capabilities.setRenameProvider(true)
      capabilities.setDocumentRangeFormattingProvider(true)
      capabilities.setReferencesProvider(true)
      capabilities.setSignatureHelpProvider(new SignatureHelpOptions(util.Arrays.asList("("), util.Arrays.asList(",")))

      val workspaceFoldersOptions = new WorkspaceFoldersOptions()
      workspaceFoldersOptions.setSupported(true)
      workspaceFoldersOptions.setChangeNotifications(true)
      val workspaceServerCapabilities = new WorkspaceServerCapabilities(workspaceFoldersOptions)
      capabilities.setWorkspace(workspaceServerCapabilities)

      val serverInfo = new ServerInfo(s"Embedded DataWeave (${configuration.lspID})")
      logger.debug(s"Initialize: Finished. Server: $serverInfo, Capabilities: $capabilities")
      new InitializeResult(capabilities, serverInfo)
    })
  }

  override def initialized(params: InitializedParams): Unit = {
    clientLogger.logInfo("Initialized")
  }

  override def shutdown(): CompletableFuture[AnyRef] = {
    CompletableFuture.supplyAsync(() => {
      if (configuration.fileScheme.isDefined) {
        workspaceServiceManager.diagnosticsPublisherService().removePublisher(diagnosticsPublisher)
        workspaceServiceManager.workspaceEditService().removeApplier(workspaceEditApplier)
        workspaceServiceManager.textDocumentTranslatorService().removeTranslator(configuration.fileScheme.get)
      }
      workspaceServiceManager.removeWorkspaceServiceListeners(this)
      logger.debug("Stopped")
      null
    })
  }

  override def exit(): Unit = {
  }

  @JsonDelegate
  override def getTextDocumentService: WeaveTextDocumentService = {
    textDocumentService
  }

  override def getWorkspaceService: WorkspaceService = {
    workspaceServiceManager.workspaceService()
  }

  override def connect(client: LanguageClient): Unit = {
    languageClient.delegate = client
  }

  def executor(): ExecutorService = {
    IDEExecutors.defaultExecutor()
  }

  override def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {
    workspaceSettings.update(params.getSettings)
  }

  override def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {
    params.getEvent.getRemoved.forEach(removedWorkspaceFolder => {
      workspaceServiceManager.onProjectKindRemoved(removedWorkspaceFolder.getUri) match {
        case Some(projectKind) =>
          textDocumentServiceImpl.onProjectRemoved(projectKind)
          projectKind.shutdown()
        case None => logger.debug(s"There is no projectKind registered for URI: ${removedWorkspaceFolder.getUri}")
      }
    })

    params.getEvent.getAdded.forEach(addedWorkspaceFolder => {
      try {
        discoverAndRegisterProjectKind(addedWorkspaceFolder)
      } catch {
        case e: Exception =>
          clientLogger.logError(s"Unable to discover project for workspaceFolder: ${addedWorkspaceFolder.getUri}", e)
      }
    })
  }

  private def discoverAndRegisterProjectKind(addedWorkspaceFolder: WorkspaceFolder) = {
    val projectMetadata = DefaultProjectMetadata(addedWorkspaceFolder.getUri, workspaceSettings.cloneProjectSettings())
    val projectKind = createProjectKind(projectMetadata, clientLoggerFactory)

    textDocumentServiceImpl.onProjectKindCreated(projectKind,
      new DataWeaveDocumentService(
        projectKind,
        executor(),
        supportAddUnitTestCodeLens = true,
        supportRunMappingLens = true,
      )
    )

    workspaceServiceManager.onProjectKindCreated(addedWorkspaceFolder.getUri, projectKind,
      new DefaultBaseWorkspaceServiceContributor(projectKind, projectMetadata, protocolService.getClient(),
        workspaceServiceManager.commandManagerService(),configuration.textDocumentLocator))

    projectKind
  }

  private def createProjectKind(projectMetadata: DefaultProjectMetadata, clientLoggerFactory: ClientLoggerFactory): ProjectKind = {
    val workspaceEditsService = workspaceServiceManager.workspaceEditService()

    val projectKind = projectKindFactory.createKind(
      ProjectKindContext(
        projectMetadata,
        clientLoggerFactory,
        protocolService.getClient(),
        jobManagerService,
        workspaceEditsService
      )
    )
    projectKind.initialize()

    projectKind
  }


}

case class EmbeddedLanguageServerConfiguration(
                                        lspID: String,
                                        codeActions: Boolean = true,
                                        codeLens: Boolean = true,
                                        fileScheme: Option[String] = None,
                                        textDocumentLocator: Option[WeaveLanguageServerBuilder.TextDocumentTranslator] = None
                                      ) {}
