package org.mule.weave.lsp.project.components

import org.apache.commons.io.FileUtils
import org.apache.commons.io.FilenameUtils
import org.mule.weave.extension.api.component.WeaveComponent
import org.mule.weave.lsp.extension.protocol.SampleInput
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.WeaveDirectoryUtils
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

import java.io.File
import java.io.FilenameFilter
import scala.collection.JavaConverters.asScalaIteratorConverter

/**
 * Provides the scenarios for Sample Data  for a given mapping
 */
trait SampleDataComponent extends WeaveComponent {

  /**
   * Returns the sample data folder for the given NameIdentifier
   *
   * @param nameIdentifier
   * @return
   */
  def searchSampleDataFolderFor(nameIdentifier: NameIdentifier): Option[File]

  /**
   * Create the sample folder for the given name identifier
   *
   * @param nameIdentifier The name Identifier
   * @return
   */
  def createSampleDataFolderFor(nameIdentifier: NameIdentifier): File


  def createSampleDataFolderFor(nameIdentifier: NameIdentifier, uri: Option[String]): File

  /**
   * List all the scenarios for a given NameIdentifier
   *
   * @param nameIdentifier The NameIdentifier of the mapping
   * @return
   */
  def listScenarios(nameIdentifier: NameIdentifier): Array[Scenario]


  /**
   * Searches for a scenario for a given mapping with the given name
   *
   * @param nameIdentifier The NameIdentifier of the mapping
   * @param scenarioName   The name of the scenario
   * @return
   */
  def searchScenarioByName(nameIdentifier: NameIdentifier, scenarioName: String): Option[Scenario]
}

case class Scenario(file: File, name: String) {

  def inputsDirectory(): File = {
    WeaveDirectoryUtils.inputsFolder(file)
  }

  def inputs(): Array[SampleInput] = {
    if (!inputsDirectory().isDirectory()) {
      Array()
    } else {
      val files = FileUtils.listFiles(inputsDirectory(), null, true)
      files.iterator().asScala
        .map(file => {
          val relativePath = inputsDirectory().toPath.relativize(file.toPath)
          val inputName = relativePath.iterator().asScala.map((p) => FilenameUtils.getBaseName(p.toFile.getName)).mkString(".")
          SampleInput(URLUtils.toLSPUrl(file), inputName)
        }).toArray
    }
  }

  def expected(): Option[File] = {
    val files = file.listFiles(new FilenameFilter {
      override def accept(dir: File, name: String): Boolean = {
        name.startsWith("out.")
      }
    })
    Option(files).flatMap(_.headOption)
  }

}

object Scenario {
  val PLAYGROUND_SCENARIO = "Playground"
}

