package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.lsp.extension.protocol.OpenTextDocumentParams
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxParams
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxResult
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

class CreateScenarioCommand(workspaceServiceManager: WorkspaceServiceManager, uIService: UIService) extends WeaveCommand {

  override def commandId(): String = Commands.DW_CREATE_SCENARIO

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val nameIdentifier: String = Commands.argAsString(params.getArguments, 0)
    val uri: String  = Commands.argAsString(params.getArguments, 1)
    val mayBeScenarioName: Option[String] = Commands.optionalArgAsString(params.getArguments, 2)

    val maybeProjectKind = workspaceServiceManager.projectKind(uri).orElse(workspaceServiceManager.getStandaloneProjectKind)
    if (maybeProjectKind.isDefined) {
      val scenariosManager: WeaveScenarioManagerService = maybeProjectKind.get.toolingService(classOf[WeaveScenarioManagerService])
      if (mayBeScenarioName.isDefined) {
        doCreateScenario(nameIdentifier, uri, mayBeScenarioName.get, uIService, scenariosManager)
      } else {
        val nameOfTheScenario: WeaveInputBoxResult = uIService.weaveInputBox(WeaveInputBoxParams(value = "NewTestScenario", title = "Specify the name of the Scenario")).get()
        if (!nameOfTheScenario.cancelled) {
          doCreateScenario(nameIdentifier, uri, nameOfTheScenario.value, uIService, scenariosManager)
        }
      }
    } else {
      uIService.showMessage(new MessageParams(MessageType.Error, s"No project is associated to: ${uri} and 'Playground' mode has not been started. ${commandId} cannot be executed"))
    }

    null
  }

  private def doCreateScenario(nameIdentifier: String, uri: String, nameOfTheScenario: String, weaveLanguageClient: UIService, scenariosManager: WeaveScenarioManagerService): Unit = {
    val sampleName = weaveLanguageClient.weaveInputBox(WeaveInputBoxParams(title = "Specify The Sample Name", value = WeaveScenarioManagerService.DEFAULT_INPUT, prompt = "Name of sample i.e payload.json or vars.foo.csv")).get()
    if (!sampleName.cancelled) {
      val maybeFile = scenariosManager.createScenario(NameIdentifier(nameIdentifier), nameOfTheScenario, sampleName.value, None, Some(uri))
      if (maybeFile.isDefined) {
        weaveLanguageClient.openTextDocument(OpenTextDocumentParams(toLSPUrl(maybeFile.get)))
      }
    }
  }

  override def name(): String = "Create Scenario."

  override def description(params: ExecuteCommandParams): String = "Creating Scenario."
}
