package org.mule.weave.lsp.vfs

import org.eclipse.lsp4j.FileChangeType
import org.mule.weave.extension.api.component.structure.WeaveProjectStructure
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProjectStructureHelper
import org.mule.weave.lsp.project.components.ProjectStructureHelper.isAProjectFile
import org.mule.weave.lsp.project.events.{OnProjectStructureChanged, ProjectStructureChangedEvent}
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.services.events.FileChangedEvent
import org.mule.weave.lsp.services.events.OnFileChanged
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.URLUtils.isChildOf
import org.mule.weave.lsp.utils.URLUtils.toFile
import org.mule.weave.lsp.utils.VFUtils
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileChangedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileCreatedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileDeletedEvent
import org.mule.weave.lsp.vfs.resource.FolderWeaveResourceResolver
import org.mule.weave.v2.editor.ChangeListener
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.ChainedWeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.slf4j.LoggerFactory

import java.io.File
import java.nio.file.Path
import java.util
import scala.collection.JavaConverters.asJavaIteratorConverter
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

/**
  * This Virtual File System handles the project interactions.
  *
  * There are two kind of files:
  *
  *  - InMemory ones, this is when a File was either modified or created by still not being saved in the persisted FS.
  *  - Persisted Files, this are the files that are haven't been modified and are the one persisted in the FS
  *
  */
class ProjectFileSystemService(jarFileNameIdentifierResolver: JarFileNameIdentifierResolver) extends VirtualFileSystem with ToolingService {
  private val logger = LoggerFactory.getLogger(getClass)

  private val inMemoryFiles: mutable.Map[String, ProjectVirtualFile] = mutable.Map[String, ProjectVirtualFile]()
  private val openedFiles: mutable.Set[String] = mutable.Set()
  private val vfsChangeListeners: ArrayBuffer[ChangeListener] = ArrayBuffer[ChangeListener]()

  private var projectKind: ProjectKind = _
  private var eventBus: InternalEventBus = _

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.projectKind = projectKind
    this.eventBus = eventBus
    eventBus.register(ProjectStructureChangedEvent.PROJECT_STRUCTURE_CHANGED, new OnProjectStructureChanged {
      override def onProjectStructureChanged(previousProjectStructure: WeaveProjectStructure, newProjectStructure: WeaveProjectStructure): Unit = {
        val filesNotPresentInProject = openedFiles
          .filter(uri => !ProjectStructureHelper.isAProjectFile(uri, newProjectStructure))
          .++(inMemoryFiles.keys.filter(uri => !ProjectStructureHelper.isAProjectFile(uri, newProjectStructure)).toArray)
        filesNotPresentInProject.foreach(toBeDeleted => deleted(toBeDeleted))
      }
    })

    eventBus.register(FileChangedEvent.FILE_CHANGED_EVENT, new OnFileChanged {
      override def onFileChanged(uri: String, changeType: FileChangeType): Unit = {
        val projectStructure = projectKind.structure()
        changeType match {
          case FileChangeType.Created => {
            if (isAProjectFile(uri, projectStructure) && toFile(uri).exists(f => f.exists() && f.isFile)) {
              created(uri)
            }
          }
          case FileChangeType.Changed => {
            if (isAProjectFile(uri, projectStructure)) {
              openedFiles.iterator.filter(p => isChildOf(p, uri)).foreach(changed)
            }
          }
          case FileChangeType.Deleted => {
            if (isAProjectFile(uri, projectStructure)) {
              openedFiles.iterator.filter(p => isChildOf(p, uri)).foreach(deleted)
            }
          }
        }
      }
    })
  }

  private def createNameIdentifierResolver(): NameIdentifierResolver =
    new CompositeNameIdentifierResolver(
      Array(
        new ProjectFileNameIdentifierResolver(projectKind.structure()),
        jarFileNameIdentifierResolver,
        SimpleNameIdentifierResolver
      )
    )

  def update(uri: String, content: String): Option[VirtualFile] = {
    logger.debug(s"Update `${uri}` -> ${content}")
    Option(file(uri)) match {
      case Some(vf) => {
        val written: Boolean = vf.write(content)
        if (written) {
          triggerChanges(vf)
        }
        Some(vf)
      }
      case None => {
        val virtualFile: ProjectVirtualFile = new ProjectVirtualFile(this, createNameIdentifierResolver(), URLUtils.toCanonicalString(uri), None, Some(content), readOnly = !VFUtils.isSupportedEditableScheme(uri))
        doUpdateFile(uri, virtualFile)
        triggerChanges(virtualFile)
        Some(virtualFile)
      }
    }
  }

  /**
    * Mark the specified Uri as closed. All memory representation should be cleaned
    *
    * @param uri The Uri of the file
    */
  def closed(uri: String): Option[VirtualFile] = {
    logger.debug(s"closed ${uri}")
    inMemoryFiles.remove(uri)
  }

  /**
    * Mark the given uri as saved into the persisted FS
    *
    * @param uri The uri to be marked as saved
    */
  def saved(uri: String): Option[VirtualFile] = {
    logger.debug(s"saved: ${uri}")
    inMemoryFiles.get(uri).map(_.save())
  }

  /**
    * Mark a given uri was changed by an external even
    *
    * @param uri The uri to be marked as changed
    */
  private def changed(uri: String): Unit = {
    logger.debug(s"changed: ${uri}")
    val virtualFile = file(uri)
    triggerChanges(virtualFile)
  }

  def isInMemoryFile(uri: String): Boolean =
    inMemoryFiles.contains(uri)

  /**
    * Mark a given uri was deleted by an external event
    *
    * @param uri The uri to be deleted
    */
  private def deleted(uri: String): Unit = {
    logger.debug(s"deleted ${uri}")
    inMemoryFiles.remove(uri)
    openedFiles.remove(uri)
    val virtualFile = new ProjectVirtualFile(this, createNameIdentifierResolver(), uri, URLUtils.toFile(uri))
    triggerDeleted(virtualFile)
  }

  private def created(uri: String): Unit = {
    logger.debug(s"created: ${uri}")
    openedFiles.add(uri)
    val virtualFile = new ProjectVirtualFile(this, createNameIdentifierResolver(), uri, URLUtils.toFile(uri))
    triggerCreated(virtualFile)
  }

  override def changeListener(cl: ChangeListener): Unit = {
    vfsChangeListeners.+=(cl)
  }

  override def onChanged(vf: VirtualFile): Unit = {
    triggerChanges(vf)
  }

  private def triggerChanges(vf: VirtualFile): Unit = {
    vfsChangeListeners.foreach(listener => {
      listener.onChanged(vf)
    })
    if (eventBus != null) {
      eventBus.fire(new ProjectVirtualFileChangedEvent(vf))
    }
  }

  private def triggerCreated(vf: VirtualFile): Unit = {
    vfsChangeListeners.foreach(listener => {
      listener.onCreated(vf)
    })
    if (eventBus != null) {
      eventBus.fire(new ProjectVirtualFileCreatedEvent(vf))
    }
  }

  private def triggerDeleted(vf: VirtualFile): Unit = {
    vfsChangeListeners.foreach(listener => {
      listener.onDeleted(vf)
    })
    if (eventBus != null) {
      eventBus.fire(new ProjectVirtualFileDeletedEvent(vf))
    }
  }

  override def removeChangeListener(service: ChangeListener): Unit = {
    vfsChangeListeners.remove(vfsChangeListeners.indexOf(service))
  }


  override def file(uri: String): VirtualFile = {
    logger.debug(s"file ${uri}")
    openedFiles.add(uri)
    //absolute path
    if (containsFile(uri)) {
      doGetFile(uri)
    } else {
      //It may not be a valid url then just try on nextone
      val maybeFile = URLUtils.toFile(uri)
      if (maybeFile.isEmpty) {
        openedFiles.remove(uri)
        null
      } else {
        if (maybeFile.get.exists()) {
          val virtualFile = new ProjectVirtualFile(this, createNameIdentifierResolver(), URLUtils.toCanonicalString(uri), maybeFile, readOnly = !VFUtils.isSupportedEditableScheme(uri))
          doUpdateFile(uri, virtualFile)
          virtualFile
        } else {
          openedFiles.remove(uri)
          null
        }
      }
    }
  }

  private def containsFile(uri: String) = {
    inMemoryFiles.contains(URLUtils.toCanonicalString(uri))
  }

  private def doGetFile(uri: String) = {
    inMemoryFiles(URLUtils.toCanonicalString(uri))
  }


  private def doUpdateFile(uri: String, virtualFile: ProjectVirtualFile) = {
    inMemoryFiles.put(URLUtils.toCanonicalString(uri), virtualFile)
  }

  def sourceRootOf(uri: String): Option[File] = {
    val rootFolders = projectKind.structure().modules
      .flatMap((module) => {
        module.roots.flatMap(_.sources) ++ module.roots.flatMap(_.resources)
      })
    rootFolders
      .find((root) => {
        val maybePath: Option[Path] = URLUtils.toPath(uri)
        maybePath
          .exists((path) => {
            path.startsWith(root.toPath)
          })
      })
  }

  override def asResourceResolver: WeaveResourceResolver = {
    val resolvers: Array[FolderWeaveResourceResolver] = projectKind.structure().modules.flatMap((module) => {
      module.roots.flatMap((root) => {
        root.sources.map((root) => {
          new FolderWeaveResourceResolver(root, this)
        }) ++
          root.resources.map((root) => {
            new FolderWeaveResourceResolver(root, this)
          })
      })
    })
    val inMemoryVirtualFileResourceResolver = new InMemoryVirtualFileResourceResolver(inMemoryFiles)
    new ChainedWeaveResourceResolver(inMemoryVirtualFileResourceResolver +: resolvers)
  }


  override def listFiles(): util.Iterator[VirtualFile] = {
    val result = projectKind.structure().modules.toIterator.flatMap((module) => {
      module.roots.toIterator.flatMap((root) => {
        root.sources.toIterator.flatMap((root) => {
          VFUtils.listFiles(root, this)
        }) ++
          root.resources.toIterator.flatMap((root) => {
            VFUtils.listFiles(root, this)
          })
      })
    })

    val wrapper = new Iterator[VirtualFile] {
      override def hasNext: Boolean = result.hasNext

      override def next(): VirtualFile = {
        val next = result.next()
        openedFiles.add(next.url)
        next
      }
    }

    wrapper.asJava
  }
}

class InMemoryVirtualFileResourceResolver(inMemoryFiles: mutable.Map[String, ProjectVirtualFile]) extends WeaveResourceResolver {

  override def resolvePath(path: String): Option[WeaveResource] = {
    this.inMemoryFiles.get(path).map((f) => WeaveResource(f))
  }

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    super.resolveAll(name)
  }

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    inMemoryFiles
      .find((urlPfvPair) => urlPfvPair._2.getNameIdentifier.equals(name))
      .map(_._2.asResource())
  }
}



