package org.mule.weave.lsp.utils

import org.slf4j.LoggerFactory

import java.util.concurrent.ExecutorService
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.util.Failure
import scala.util.Success
import scala.util.Try

class InternalEventBus(executorService: ExecutorService) {

  private val logger = LoggerFactory.getLogger(getClass)
  private val listeners = new mutable.HashMap[InternalEventType[_], ArrayBuffer[InternalEventHandler]]

  def unRegister[T <: InternalEventHandler](evenType: InternalEventType[T], eventHandler: T): Unit = {
    listeners.get(evenType).foreach(a => {
      a.-=(eventHandler)
    })
  }

  def register[T <: InternalEventHandler](evenType: InternalEventType[T], handler: T): EventRegistrationHandler[T] = {
    if (handler == null) {
      throw new IllegalArgumentException("Handler can not be null")
    }
    listeners.getOrElseUpdate(evenType, ArrayBuffer[InternalEventHandler]()).+=(handler)
    EventRegistrationHandler(this, evenType, handler)
  }

  def fire(event: InternalEvent): Unit = {
    logger.debug(s"Fire event [type: ${event.getType}, hash-code: ${System.identityHashCode(event)}].")
    listeners.get(event.getType) match {
      case Some(listeners) =>
        executorService.execute(() => {
          listeners.foreach(l => {
            Try({
              val handler = l.asInstanceOf[event.T]
              logger.debug(s"Dispatching event [type: ${event.getType}, hash-code: ${System.identityHashCode(event)}] to [$handler].")
              event.dispatch(l.asInstanceOf[event.T])
              logger.debug(s"Finished event [type: ${event.getType}, hash-code: ${System.identityHashCode(event)}] to [$handler].")
            }) match {
              case Success(_) =>
              case Failure(exception) =>
                logger.warn("Exception on dispatching :" + event, exception)
            }
          })
        })
      case None =>
    }
  }
}


trait InternalEvent {

  type T <: InternalEventHandler

  def getType: InternalEventType[T]

  def dispatch(handler: T): Unit
}

case class InternalEventType[T <: InternalEventHandler](kind: String) {}

trait InternalEventHandler {}

case class EventRegistrationHandler[T <: InternalEventHandler](bus: InternalEventBus, evenType: InternalEventType[T], handler: T) {
  def unregister(): Unit = {
    bus.unRegister(evenType, handler)
  }

  def register(): Unit = {
    bus.register(evenType, handler)
  }
}