package org.mule.weave.lsp.utils

import org.codehaus.plexus.util.FileUtils
import org.mule.weave.v2.utils.StringHelper.toStringTransformer

object InputScaffoldingFactory {

  private val SCAFFOLDING =
    Map(
      "json" -> new JsonInputScaffolding(),
      "xml" -> new XmlInputScaffolding(),
      "csv" -> new CSVInputScaffolding(),
      "dwl" -> new WeaveInputScaffolding(),
      "yaml" -> new YamlInputScaffolding()
    )

  def create(filename: String): Option[InputScaffolding] = {
    val extension = FileUtils.extension(filename)
    SCAFFOLDING.get(extension)
  }
}

trait InputScaffolding {
  /**
    * Returns a template snippet of code for a given format
    *
    * @return The snippet of code
    */
  def template(): String
}

class JsonInputScaffolding extends InputScaffolding {
  override def template(): String =
    """{
      | "messages": "Hello world!"
      |}""".stripMarginAndNormalizeEOL
}

class WeaveInputScaffolding extends InputScaffolding {
  override def template(): String =
    """{
      | messages: "Hello world!"
      |}""".stripMarginAndNormalizeEOL
}

class XmlInputScaffolding extends InputScaffolding {
  override def template(): String = {
    """<?xml version='1.0' encoding='UTF-8'?>
      |<root>Hello world!</root>""".stripMarginAndNormalizeEOL
  }
}

class CSVInputScaffolding extends InputScaffolding {
  override def template(): String = {
    """name,lastname
      |data,weave""".stripMarginAndNormalizeEOL
  }
}

class YamlInputScaffolding extends InputScaffolding {
  override def template(): String = {
    """
      |messages: Hello world!""".stripMarginAndNormalizeEOL
  }
}