package org.mule.weave.lsp.services

import org.eclipse.lsp4j.DidChangeConfigurationParams
import org.eclipse.lsp4j.DidChangeWatchedFilesParams
import org.eclipse.lsp4j.DidChangeWorkspaceFoldersParams
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.SymbolInformation
import org.eclipse.lsp4j.WorkspaceSymbolParams
import org.eclipse.lsp4j.services.WorkspaceService
import org.mule.weave.lsp.commands.CommandManagerService
import org.mule.weave.lsp.commands.Commands
import org.mule.weave.lsp.commands.InternalWeaveCommand
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolClient
import org.mule.weave.lsp.extension.protocol.SetContextParams
import org.mule.weave.lsp.extension.protocol.SetContextValue
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.jobs.Task
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.events.DocumentChangedEvent
import org.mule.weave.lsp.services.events.DocumentFocusChangedEvent
import org.mule.weave.lsp.services.events.DocumentOpenedEvent
import org.mule.weave.lsp.services.events.OnDocumentChanged
import org.mule.weave.lsp.services.events.OnDocumentFocused
import org.mule.weave.lsp.services.events.OnDocumentOpened
import org.mule.weave.v2.editor.VirtualFile

import java.util
import java.util.concurrent.CompletableFuture
import scala.collection.JavaConverters.seqAsJavaListConverter
import scala.collection.concurrent.TrieMap

/**
 * DataWeave Implementation of the LSP Workspace Service
 */
class DataWeaveWorkspaceService(weaveProtocolClient: DataWeaveProtocolClient,
                                jobManagerService: JobManagerService,
                                commandManager: CommandManagerService,
                                workspaceServiceListener: WorkspaceServiceListener)
  extends WorkspaceService with WorkspaceServiceContributorRegistry {

  private val projectWorkspaceContributors: TrieMap[ProjectKind, WorkspaceServiceContributor] = TrieMap()

  private val onDocumentChanged: OnDocumentChanged = (vf: VirtualFile) => {
    notifyContextChanged(vf)
  }

  private val onDocumentOpened: OnDocumentOpened = (vf: VirtualFile) => {
    notifyContextChanged(vf)
  }

  private val onDocumentFocused: OnDocumentFocused = (vf: VirtualFile) => {
    notifyContextChanged(vf)
  }

  def onProjectKindCreated(projectKind: ProjectKind, workspaceServiceContributor: WorkspaceServiceContributor): Unit = {
    this.projectWorkspaceContributors.putIfAbsent(projectKind, workspaceServiceContributor) match {
      case Some(_) => // Already registered, therefore no need to register the listeners again...
      case None => registerDocumentListeners(projectKind)
    }
  }

  def onProjectKindRemoved(projectKind: ProjectKind): Unit = {
    this.projectWorkspaceContributors.remove(projectKind) match {
      case Some(_) => unregisterDocumentListeners(projectKind)
      case None =>
    }
  }

  private def registerDocumentListeners(projectKind: ProjectKind): Unit = {
    projectKind.eventBus().register(DocumentChangedEvent.DOCUMENT_CHANGED, onDocumentChanged)
    projectKind.eventBus().register(DocumentOpenedEvent.DOCUMENT_OPENED, onDocumentOpened)
    projectKind.eventBus().register(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, onDocumentFocused)
  }

  private def unregisterDocumentListeners(projectKind: ProjectKind): Unit = {
    projectKind.eventBus().unRegister(DocumentChangedEvent.DOCUMENT_CHANGED, onDocumentChanged)
    projectKind.eventBus().unRegister(DocumentOpenedEvent.DOCUMENT_OPENED, onDocumentOpened)
    projectKind.eventBus().unRegister(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, onDocumentFocused)
  }

  private def notifyContextChanged(vf: VirtualFile): Unit = {
    val values = commandManager.commands.map(command => {
      val isEnabled = command match {
        case command: InternalWeaveCommand => command.enabled(vf.url())
        case _ => true
      }
      SetContextValue(command.commandId() + Commands.COMMAND_ENABLED_SUFFIX, isEnabled, vf.url())
    })
    val params = SetContextParams(util.Arrays.asList(values: _*))
    weaveProtocolClient.setContext(params)
  }

  override def symbol(params: WorkspaceSymbolParams): CompletableFuture[util.List[_ <: SymbolInformation]] = {
    val futures: Array[CompletableFuture[util.List[_ <: SymbolInformation]]] =
      projectWorkspaceContributors.values.map(projectWorkspaceContributor => projectWorkspaceContributor.symbol(params)).toArray

    val allFutures = CompletableFuture.allOf(futures: _*)

    allFutures.thenApply { _ =>
      val symbolsInformation: util.ArrayList[SymbolInformation] = new util.ArrayList()
      futures.foreach(future => symbolsInformation.addAll(future.join()))
      symbolsInformation
    }
  }

  override def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {
    workspaceServiceListener.didChangeConfiguration(params)

    projectWorkspaceContributors.values
      .foreach(projectWorkspaceContributor => projectWorkspaceContributor.didChangeConfiguration(params))
  }

  override def executeCommand(params: ExecuteCommandParams): CompletableFuture[AnyRef] = {
    CompletableFuture.supplyAsync(() => {
      commandManager.commandBy(params.getCommand)
        .map(c => {
          var result: AnyRef = null
          jobManagerService.execute(new Task() {
            override def run(cancelable: Status): Unit = {
              result = c.execute(params)
            }
          }, s"Running command: ${c.name()}", s"${c.description(params)}")
          result
        })
        .orNull
    })
  }

  override def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {
    workspaceServiceListener.didChangeWorkspaceFolders(params)
  }

  override def didChangeWatchedFiles(params: DidChangeWatchedFilesParams): Unit = {
    projectWorkspaceContributors.values.foreach(projectWorkspaceContributor => {
      projectWorkspaceContributor.didChangeWatchedFiles(params)
    })
  }

}

trait WorkspaceServiceListener {
  def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {}

  def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {}
}