package org.mule.weave.lsp

import org.eclipse.lsp4j._
import org.eclipse.lsp4j.jsonrpc.services.JsonDelegate
import org.eclipse.lsp4j.services.{LanguageClient, LanguageServer, WorkspaceService}
import org.mule.dx.platform.api.protocol.ClientAware
import org.mule.weave.lsp.extension.client.LanguageClientDelegate
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolService
import org.mule.weave.lsp.extension.services.WeaveTextDocumentService
import org.mule.weave.lsp.jobs.{JobManagerService, Status}
import org.mule.weave.lsp.project._
import org.mule.weave.lsp.services._
import org.mule.weave.lsp.services.delegate.TextDocumentServiceDelegate
import org.slf4j.{Logger, LoggerFactory}

import java.util
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutorService
import scala.collection.concurrent.TrieMap

/**
 * Base class for DataWeave Language Server, standalone and embedded.
 */
abstract class BaseWeaveLanguageServer(
  protected val projectKindFactory: ProjectKindFactory,
  protected val protocolService: DataWeaveProtocolService,
  protected val workspaceServiceManager: WorkspaceServiceManager,
  protected val jobManagerService: JobManagerService
) extends LanguageServer with ClientAware[LanguageClient] with WorkspaceServiceListener {

  protected val logger: Logger = LoggerFactory.getLogger(getClass)
  private lazy val textDocumentServiceDispatcher: DataWeaveDocumentServiceDispatcher = createTextDocumentServiceDispatcher()
  private val textDocumentService: TextDocumentServiceDelegate = new TextDocumentServiceDelegate()
  protected val languageClient: LanguageClientDelegate = new LanguageClientDelegate()
  protected val projectKindsByWorkspaceFolderUri: TrieMap[String, ProjectKind] = TrieMap()
  protected val workspaceSettings: ProjectSettings = ProjectSettings()

  protected var clientLoggerFactory: ClientLoggerFactory = _
  protected var clientLogger: ClientLogger = _

  // Abstract methods that subclasses must implement
  protected def getServerId: String
  protected def getServerInfo: ServerInfo
  protected def createTextDocumentServiceDispatcher(): DataWeaveDocumentServiceDispatcher
  protected def setupWorkspaceServices(): Unit
  protected def createServerCapabilities(): ServerCapabilities
  protected def onLanguageClientConnected(): Unit
  protected def onShutdown(): Unit
  protected def onExit(): Unit

  override def initialize(params: InitializeParams): CompletableFuture[InitializeResult] = {
    logger.debug(s"Initialize ${getServerId}")
    CompletableFuture.supplyAsync(() => {
      logger.debug(s"Initializing ${getServerId}")
      Runtime.getRuntime.addShutdownHook(new Thread(() => this.shutdown().get()))
      
      clientLoggerFactory = new ClientLoggerFactory(languageClient)
      clientLogger = clientLoggerFactory.createLogger(getClass, getServerId)

      workspaceServiceManager.addWorkspaceServiceListeners(this)
      workspaceSettings.load(params.getInitializationOptions)

      if (params.getWorkspaceFolders != null) {
        params.getWorkspaceFolders.forEach(workspaceFolder => discoverAndRegisterProjectKind(workspaceFolder))
      }

      textDocumentService.delegate = textDocumentServiceDispatcher
      setupWorkspaceServices()

      val capabilities = createServerCapabilities()
      val serverInfo = getServerInfo
      
      logger.debug(s"Initialize: Finished. Server: $serverInfo, Capabilities: $capabilities")
      new InitializeResult(capabilities, serverInfo)
    })
  }

  override def initialized(params: InitializedParams): Unit = {
    projectKindsByWorkspaceFolderUri.foreach(entry => {
      jobManagerService
        .schedule((_: Status) => {
          initializedProjectKind(entry._2)
        }, s"Starting DataWeave Project [" + entry._2.project().getName + "]", "Starting DataWeave Project.")
    })
    clientLogger.logInfo("Initialized")
  }

  protected def initializedProjectKind(projectKind: ProjectKind): Unit = {
    try {
      clientLogger.logInfo(s"Project: '${projectKind.project().getName()}' starting.")
      projectKind.initialized()
      clientLogger.logInfo(s"Project: '${projectKind.project().getName()}' started.")
    } catch {
      case e: Exception =>
        clientLogger.logError(s"Unable to start project: '${projectKind.project().getName()}'.", e)
    }
  }

  override def shutdown(): CompletableFuture[AnyRef] = {
    CompletableFuture.supplyAsync(() => {
      projectKindsByWorkspaceFolderUri.foreach(entry => {
        workspaceServiceManager.onProjectKindRemoved(entry._1)
        textDocumentServiceDispatcher.onProjectKindRemoved(entry._2)
        entry._2.shutdown()
      })
      workspaceServiceManager.removeWorkspaceServiceListeners(this)
      onShutdown()
      logger.debug("Stopped")
      null
    })
  }

  override def exit(): Unit = {
    onExit()
  }

  @JsonDelegate
  override def getTextDocumentService: WeaveTextDocumentService = {
    textDocumentService
  }

  override def getWorkspaceService: WorkspaceService = {
    workspaceServiceManager.workspaceService()
  }

  override def connect(client: LanguageClient): Unit = {
    languageClient.delegate = client
    onLanguageClientConnected()
  }

  def executor(): ExecutorService = {
    IDEExecutors.defaultExecutor()
  }

  override def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {
    workspaceSettings.update(params.getSettings)
  }

  override def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {
    params.getEvent.getRemoved.forEach(removedWorkspaceFolder => {
      workspaceServiceManager.onProjectKindRemoved(removedWorkspaceFolder.getUri) match {
        case Some(projectKind) =>
          textDocumentServiceDispatcher.onProjectKindRemoved(projectKind)
          projectKind.shutdown()
        case None => logger.debug(s"There is no projectKind registered for URI: ${removedWorkspaceFolder.getUri}")
      }
    })

    params.getEvent.getAdded.forEach(addedWorkspaceFolder => {
      try {
        discoverAndRegisterProjectKind(addedWorkspaceFolder)
          .foreach(pk => initializedProjectKind(pk))
      } catch {
        case e: Exception =>
          clientLogger.logError(s"Unable to discover project for workspaceFolder: ${addedWorkspaceFolder.getUri}", e)
      }
    })
  }

  private final def discoverAndRegisterProjectKind(addedWorkspaceFolder: WorkspaceFolder): Option[ProjectKind] = {
    val projectMetadata = DefaultProjectMetadata(addedWorkspaceFolder.getUri, workspaceSettings.cloneProjectSettings())
    val maybeProjectKind = createProjectKind(projectMetadata, clientLoggerFactory)

    maybeProjectKind.foreach(projectKind => {
      textDocumentServiceDispatcher.onProjectKindCreated(projectKind,
        new DataWeaveDocumentService(
          projectKind,
          executor(),
          supportAddUnitTestCodeLens = true,
          supportRunMappingLens = true,
        )
      )

      workspaceServiceManager.onProjectKindCreated(addedWorkspaceFolder.getUri, projectKind,
        createWorkspaceServiceContributor(projectMetadata, projectKind))

      projectKindsByWorkspaceFolderUri.putIfAbsent(addedWorkspaceFolder.getUri, projectKind)
    })
    maybeProjectKind
  }

  private def createProjectKind(projectMetadata: DefaultProjectMetadata, clientLoggerFactory: ClientLoggerFactory): Option[ProjectKind] = {
    val workspaceEditsService = workspaceServiceManager.workspaceEditService()

    val maybeProjectKind = projectKindFactory.createKind(
      ProjectKindContext(
        projectMetadata,
        clientLoggerFactory,
        protocolService.getClient(),
        jobManagerService,
        workspaceEditsService
      )
    )
    maybeProjectKind.foreach(pk => pk.initialize())
    maybeProjectKind
  }

  protected def createWorkspaceServiceContributor(projectMetadata: DefaultProjectMetadata, projectKind: ProjectKind): WorkspaceServiceContributor = {
    new DefaultWorkspaceServiceContributor(projectKind, projectMetadata, protocolService.getClient(),
      workspaceServiceManager.commandManagerService())
  }

  protected def createBaseServerCapabilities(): ServerCapabilities = {
    val capabilities = new ServerCapabilities
    capabilities.setTextDocumentSync(TextDocumentSyncKind.Full)
    capabilities.setCompletionProvider(new CompletionOptions(true, java.util.Arrays.asList(".")))
    capabilities.setHoverProvider(true)
    capabilities.setDocumentSymbolProvider(true)
    capabilities.setDefinitionProvider(true)
    capabilities.setDocumentFormattingProvider(true)
    capabilities.setFoldingRangeProvider(true)
    capabilities.setCodeActionProvider(true)
    capabilities.setWorkspaceSymbolProvider(true)
    capabilities.setCodeLensProvider(new CodeLensOptions(true))
    capabilities.setRenameProvider(true)
    capabilities.setDocumentRangeFormattingProvider(true)
    capabilities.setReferencesProvider(true)
    capabilities.setSignatureHelpProvider(new SignatureHelpOptions(util.Arrays.asList("("), util.Arrays.asList(",")))

    val workspaceFoldersOptions = new WorkspaceFoldersOptions()
    workspaceFoldersOptions.setSupported(true)
    workspaceFoldersOptions.setChangeNotifications(true)
    val workspaceServerCapabilities = new WorkspaceServerCapabilities(workspaceFoldersOptions)
    capabilities.setWorkspace(workspaceServerCapabilities)
    
    capabilities
  }
}
