package org.mule.weave.lsp

import org.eclipse.lsp4j.{ExecuteCommandOptions, ServerCapabilities, ServerInfo}
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.project.{DefaultProjectMetadata, ProjectKind, ProjectKindContext, ProjectKindFactory}
import org.mule.weave.lsp.services._
import org.mule.weave.lsp.utils.{URLUtils, WeaveDirectoryUtils}

import java.io.File
import java.util

class WeaveLanguageServer(
  projectKindFactory: ProjectKindFactory,
  protocolService: DataWeaveProtocolService,
  workspaceServiceManager: WorkspaceServiceManager,
  jobManagerService: JobManagerService
) extends BaseWeaveLanguageServer(projectKindFactory, protocolService, workspaceServiceManager, jobManagerService) {

  private val ID = "DATA_WEAVE_LSP"
  private lazy val languageServerDocumentService: DataWeaveDocumentService = createLanguageServerDocumentService()

  override protected def getServerId: String = ID

  override protected def getServerInfo: ServerInfo = {
    new ServerInfo("DataWeave", workspaceSettings.languageLevelVersion.value())
  }

  override protected def createTextDocumentServiceDispatcher(): DataWeaveDocumentServiceDispatcher = {
    new DataWeaveDocumentServiceDispatcher(fallbackDocumentService = (_) => languageServerDocumentService)
  }

  override protected def setupWorkspaceServices(): Unit = {
    workspaceServiceManager.diagnosticsPublisherService().setDefaultPublisher(new DiagnosticsPublisher(languageClient, Option.empty))
    workspaceServiceManager.workspaceEditService().setDefaultApplier(new WorkspaceEditApplier(languageClient, Option.empty))
  }

  override protected def createServerCapabilities(): ServerCapabilities = {
    val capabilities = createBaseServerCapabilities()
    val commandManagerService = workspaceServiceManager.commandManagerService()
    val list = util.Arrays.asList(commandManagerService.commandIds(): _*)
    capabilities.setExecuteCommandProvider(new ExecuteCommandOptions(list))
    capabilities
  }

  override protected def onLanguageClientConnected(): Unit = {
    workspaceServiceManager.onLanguageClientConnected(languageClient)
  }

  override protected def onShutdown(): Unit = {
    // No additional shutdown logic needed
  }

  override protected def onExit(): Unit = {
    System.exit(0)
  }

  private def createLanguageServerDocumentService(): DataWeaveDocumentService = {
    val url = URLUtils.toLSPUrl(new File(WeaveDirectoryUtils.getWorkingHome(), "Playground"))
    val projectMetadata = DefaultProjectMetadata(url, workspaceSettings.cloneProjectSettings())

    val workspaceEditsService = workspaceServiceManager.workspaceEditService()

    val projectKind = projectKindFactory.createDefaultProjectKind(
      ProjectKindContext(
        projectMetadata,
        clientLoggerFactory,
        protocolService.getClient(),
        jobManagerService,
        workspaceEditsService
      )
    )

    val languageServerDataWeaveDocumentService =
      new DataWeaveDocumentService(
        projectKind,
        executor(),
        supportAddUnitTestCodeLens = false,
        supportRunMappingLens = false
      )

    val uri = URLUtils.toLSPUrl(projectMetadata.home())
    workspaceServiceManager.standaloneContribution(uri, projectKind,
      new DefaultWorkspaceServiceContributor(projectKind, projectMetadata, protocolService.getClient(), workspaceServiceManager.commandManagerService()))

    projectKindsByWorkspaceFolderUri.putIfAbsent(uri, projectKind)

    projectKind.initialize()

    val currentContextClassLoader = Thread.currentThread.getContextClassLoader
    try {
      Thread.currentThread.setContextClassLoader(classOf[WeaveLanguageServer].getClassLoader)
      initializedProjectKind(projectKind)
    } finally {
      Thread.currentThread.setContextClassLoader(currentContextClassLoader)
    }

    languageServerDataWeaveDocumentService
  }
}