package org.mule.weave.lsp

import org.eclipse.lsp4j.{ServerCapabilities, ServerInfo}
import org.mule.weave.extension.api.WeaveLanguageServerBuilder
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.project.{DefaultProjectMetadata, ProjectKind, ProjectKindContext, ProjectKindFactory}
import org.mule.weave.lsp.services._

class EmbeddedWeaveLanguageServer(
  projectKindFactory: ProjectKindFactory,
  configuration: EmbeddedLanguageServerConfiguration,
  protocolService: DataWeaveProtocolService,
  workspaceServiceManager: WorkspaceServiceManager,
  jobManagerService: JobManagerService
) extends BaseWeaveLanguageServer(projectKindFactory, protocolService, workspaceServiceManager, jobManagerService) {

  private val workspaceEditApplier = new WorkspaceEditApplier(languageClient, configuration.fileScheme)
  private val diagnosticsPublisher = new DiagnosticsPublisher(languageClient, configuration.fileScheme)

  override protected def getServerId: String = configuration.lspID

  override protected def getServerInfo: ServerInfo = {
    new ServerInfo(s"Embedded DataWeave (${configuration.lspID})")
  }

  override protected def createTextDocumentServiceDispatcher(): DataWeaveDocumentServiceDispatcher = {
    new DataWeaveDocumentServiceDispatcher(configuration.codeLens, configuration.textDocumentLocator)
  }

  override protected def setupWorkspaceServices(): Unit = {
    if (configuration.fileScheme.isDefined) {
      workspaceServiceManager.diagnosticsPublisherService().addPublisher(diagnosticsPublisher)
      workspaceServiceManager.workspaceEditService().addApplier(workspaceEditApplier)
      workspaceServiceManager.textDocumentTranslatorService().addTranslator(configuration.fileScheme.get, configuration.textDocumentLocator.get)
    }
  }

  override protected def createServerCapabilities(): ServerCapabilities = {
    val capabilities = createBaseServerCapabilities()
    capabilities.setCodeActionProvider(configuration.codeActions)
    capabilities.setCodeLensProvider(new org.eclipse.lsp4j.CodeLensOptions(configuration.codeLens))
    capabilities
  }

  override protected def onLanguageClientConnected(): Unit = {
    // No additional connection logic needed for embedded server
  }

  override protected def onShutdown(): Unit = {
    if (configuration.fileScheme.isDefined) {
      workspaceServiceManager.diagnosticsPublisherService().removePublisher(diagnosticsPublisher)
      workspaceServiceManager.workspaceEditService().removeApplier(workspaceEditApplier)
      workspaceServiceManager.textDocumentTranslatorService().removeTranslator(configuration.fileScheme.get)
    }
  }

  override protected def onExit(): Unit = {
    // No exit logic needed for embedded server
  }

  override protected def createWorkspaceServiceContributor(projectMetadata: DefaultProjectMetadata, projectKind: ProjectKind): WorkspaceServiceContributor =
    new DefaultWorkspaceServiceContributor(projectKind, projectMetadata, protocolService.getClient(),
      workspaceServiceManager.commandManagerService(), configuration.textDocumentLocator)
}

case class EmbeddedLanguageServerConfiguration(
  lspID: String,
  codeActions: Boolean = true,
  codeLens: Boolean = true,
  fileScheme: Option[String] = None,
  textDocumentLocator: Option[WeaveLanguageServerBuilder.TextDocumentTranslator] = None
) {}
