package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ApplyWorkspaceEditParams
import org.eclipse.lsp4j.CreateFile
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.eclipse.lsp4j.Position
import org.eclipse.lsp4j.ResourceOperation
import org.eclipse.lsp4j.TextDocumentEdit
import org.eclipse.lsp4j.TextEdit
import org.eclipse.lsp4j.VersionedTextDocumentIdentifier
import org.eclipse.lsp4j.WorkspaceEdit
import org.eclipse.lsp4j.jsonrpc.messages.Either
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.lsp.extension.protocol.OpenTextDocumentParams
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxParams
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxResult
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickItem
import org.mule.weave.lsp.extension.protocol.WeaveQuickPickParams
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProjectStructureHelper
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl
import org.mule.weave.lsp.utils.VFUtils
import org.mule.weave.lsp.utils.WeaveDirectoryUtils
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

import java.io.File
import java.util
import scala.io.Source

class CreateTestCommand(workspaceServiceManager: WorkspaceServiceManager, uIService: UIService) extends WeaveCommand {

  private val TEMPLATE_TEST: String = {
    val source = Source.fromInputStream(getClass.getClassLoader.getResourceAsStream("dw-template-test.dwl"), "UTF-8")
    try {
      source.mkString
    } finally {
      source.close()
    }
  }

  override def commandId(): String = Commands.DW_CREATE_TEST

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val maybeProjectKind = selectProjectKind()
    if (maybeProjectKind.isDefined) {
      val value: WeaveInputBoxResult = uIService.weaveInputBox(WeaveInputBoxParams(title = "Insert The Name Of The Test")).get()
      if (!value.cancelled) {
        val projectKind = maybeProjectKind.get
        val nameOfTheTest = value.value
        var path = nameOfTheTest.replace(NameIdentifier.SEPARATOR, File.separator)
        if (!path.endsWith(VFUtils.DWL_EXTENSION)) {
          path = path + VFUtils.DWL_EXTENSION
        }
        val maybeFile: Option[File] = ProjectStructureHelper.testsSourceFolders(projectKind.structure()).find((f) => f.getName == WeaveDirectoryUtils.DWTest_FOLDER)
        maybeFile match {
          case Some(weaveTestFolder) => {
            val testFile = new File(weaveTestFolder, path)
            val testFileURL = toLSPUrl(testFile)
            val testScript = TEMPLATE_TEST.replace("${syntaxVersion}", projectKind.getWeaveVersion())
            val createFile = Either.forRight[TextDocumentEdit, ResourceOperation](new CreateFile(testFileURL))
            val textEdit = new TextEdit(new org.eclipse.lsp4j.Range(new Position(0, 0), new Position(0, 0)), testScript)
            val textDocumentEdit = new TextDocumentEdit(new VersionedTextDocumentIdentifier(testFileURL, 0), util.Arrays.asList(textEdit))
            val insertText = Either.forLeft[TextDocumentEdit, ResourceOperation](textDocumentEdit)
            val edits = util.Arrays.asList(createFile, insertText)
            val response = workspaceServiceManager.workspaceEditService.applyEdit(new ApplyWorkspaceEditParams(new WorkspaceEdit(edits))).get()
            if (response.isApplied) {
              uIService.openTextDocument(OpenTextDocumentParams(testFileURL))
            }
          }
          case None => {
            uIService.showMessage(new MessageParams(MessageType.Warning, "Unable to find test folder for project."))
          }
        }
      } else {

      }
    }
    null
  }

  private def selectProjectKind(): Option[ProjectKind] = {
    val projectKinds = workspaceServiceManager.projectKinds()
    var maybeProjectKind: Option[ProjectKind] = Option.empty
    if (!projectKinds.isEmpty) {
      if (projectKinds.length == 1) {
        maybeProjectKind = Option.apply(projectKinds.head)
      } else {
        val inputs: Array[WeaveQuickPickItem] = projectKinds.map((p) => {
          WeaveQuickPickItem(p.projectMetadata.home.getAbsolutePath, p.project.getName)
        })
        val result = uIService.weaveQuickPick(WeaveQuickPickParams(
          util.Arrays.asList(inputs: _*),
          "Select the Project where to create the File",
        )).get()
        if (!result.cancelled && !result.itemsId.isEmpty) {
          val projectPath = result.itemsId.get(0)
          maybeProjectKind = projectKinds.find(_.projectMetadata().home().getAbsolutePath.equals(projectPath))
        }
      }
    }
    maybeProjectKind
  }
  override def name(): String = "Creates Unit Test."

  override def description(params: ExecuteCommandParams): String = "Creating Unit Test."
}
