package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.mule.weave.dsp.DataWeaveDebuggerAdapterProtocolLauncher.launch
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.project.DefaultProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProcessLauncher
import org.mule.weave.lsp.services.ClientLoggerFactory
import org.mule.weave.lsp.services.DataWeaveTestService
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.utils.NetUtils
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.v2.editor.VirtualFileSystem

import java.nio.file.Paths
import java.util.concurrent.CountDownLatch

//LSP .....
// LaunchDebugger
// [VSCODE] <-> [LSPADI] <-> [PDW]

class RunWeaveCommand(workspaceServiceManager: WorkspaceServiceManager,
                      loggerFactory: ClientLoggerFactory,
                      jobManagerService: JobManagerService,
                      uIService: UIService) extends WeaveCommand {

  override def commandId(): String = Commands.DW_RUN_MAPPING

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val configType = Commands.argAsString(params.getArguments, 0)
    val maybeProjectFilePath = Commands.optionalArgAsString(params.getArguments, 1)

    runMapping(configType, maybeProjectFilePath, uIService)
  }

  private def runMapping(config: String, maybeProjectFilePath: Option[String], uIService: UIService): Integer = {
    if (maybeProjectFilePath.isEmpty) {
      -1
    }
    else {
      val projectFolder = maybeProjectFilePath.map(path => Paths.get(path).toFile).get
      val maybeProjectKind = workspaceServiceManager.projectKinds().find(projectKind =>
        projectKind.projectMetadata().home().equals(projectFolder))
      if (maybeProjectKind.isEmpty) {
        uIService.showMessage(new MessageParams(MessageType.Warning, s"Can not run a DW script, no projectKind registered for file path: ${maybeProjectFilePath.get}"))
        -1
      } else {
        val projectKind = maybeProjectKind.get
        if (!projectKind.isStarted) {
          uIService.showMessage(new MessageParams(MessageType.Warning, "Can not run a DW script until Project was initialized."))
          -1
        } else {
          val port: Int = NetUtils.freePort()
          val latch = new CountDownLatch(1)
          jobManagerService.schedule((status: Status) => {
            val launcher: ProcessLauncher = ProcessLauncher.createLauncherByType(config, projectKind, loggerFactory, uIService, projectKind.toolingService(classOf[ProjectFileSystemService]))
            launch(projectKind.vfs(), loggerFactory, uIService, launcher, projectKind, jobManagerService, projectKind.toolingService(classOf[DataWeaveTestService]), () => latch.countDown(), port)
          }, "Starting Debugger Server", "Starting Debugger Server")
          latch.await()
          port
        }
      }
    }
  }


  override def name(): String = "Run DataWeave."

  override def description(params: ExecuteCommandParams): String = "Running DataWeave Command."

}
