package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.Command
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.extension.protocol.OpenTextDocumentParams
import org.mule.weave.lsp.extension.protocol.PreviewResult
import org.mule.weave.lsp.extension.protocol.WeaveInputBoxParams
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.AgentBasedDataFormatDescriptorProviderService
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.services.WorkspaceServiceManager
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl
import org.mule.weave.v2.completion.DataFormatDescriptor
import org.mule.weave.v2.editor.MappingInput
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.scaffolding.ScaffoldingConfiguration
import org.mule.weave.v2.scaffolding.ScaffoldingService
import org.mule.weave.v2.ts.WeaveType

import java.io.File
import java.util

class CreateInputSampleCommand(workspaceServiceManager: WorkspaceServiceManager, uIService: UIService) extends WeaveCommand {

  override def commandId(): String = Commands.DW_CREATE_INPUT_SAMPLE

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val nameIdentifier: String = Commands.argAsString(params.getArguments, 0)
    val nameOfScenario: String = Commands.argAsString(params.getArguments, 1)
    val uri: String = Commands.argAsString(params.getArguments, 2)

    val maybeProjectKind = workspaceServiceManager.projectKind(uri).orElse(workspaceServiceManager.getStandaloneProjectKind)
    if (maybeProjectKind.isDefined) {
      val inputName: Option[String] = Commands.optionalArgAsString(params.getArguments, 3)
      val mayBeInputFileName: Option[String] = Commands.optionalArgAsString(params.getArguments, 4)

      if (mayBeInputFileName.isEmpty) {
        val sampleName = uIService.weaveInputBox(WeaveInputBoxParams(title = "Specify The Sample Name", value = "payload.json", prompt = "Name of sample i.e payload.json or vars.foo.csv")).get()
        if (!sampleName.cancelled) {
          val inputFileName: String = sampleName.value
          createInput(nameIdentifier, nameOfScenario, inputFileName, uri, inputName, uIService, maybeProjectKind.get)
        }
      } else {
        createInput(nameIdentifier, nameOfScenario, mayBeInputFileName.get, uri, inputName, uIService, maybeProjectKind.get)
      }
    } else {
      uIService.showMessage(new MessageParams(MessageType.Error, s"No project is associated to: ${uri} and 'Playground' mode has not been started. ${commandId} cannot be executed"))
    }
    null
  }

  private def createInput(nameIdentifier: String, nameOfScenario: String, inputFileName: String, uri: String, inputName: Option[String],
                          uiService: UIService, projectKind: ProjectKind): Unit = {
    val content: Option[String] =
      if (inputName.isEmpty) {
        None
      } else {
        val maybeInputDirective: Option[MappingInput] = projectKind.toolingService(classOf[DataWeaveToolingService]).inputOf(uri, inputName.get)
        val inputType: Option[WeaveType] = maybeInputDirective.flatMap(_.wtype)
        val inputMimeType: Option[String] = maybeInputDirective.flatMap(_.dataFormat.map(_.value))
        if (inputType.isDefined && inputMimeType.isDefined) {
          val dataFormatValue = inputMimeType.get
          val scaffoldingService: ScaffoldingService = new ScaffoldingService()
          val scaffoldingConfiguration: ScaffoldingConfiguration = ScaffoldingConfiguration(projectKind.projectMetadata().settings.scaffoldingElementNumber.intValue())

          val maybeFormatDescriptor: Option[DataFormatDescriptor] =
            projectKind.toolingService(classOf[AgentBasedDataFormatDescriptorProviderService])
              .dataFormats()
              .find((d) => d.mimeType.equals(dataFormatValue) || d.id.exists(_.equals(dataFormatValue)))

          val writerOptions = maybeFormatDescriptor match {
            case Some(formatDescriptor) => {
              maybeInputDirective.map((id) => {
                id.properties
                  .filter((p) => formatDescriptor.writerProperties.exists(_.name == p.name))
                  .map((p) => {
                    val value = p.value
                    val descriptorProp = formatDescriptor.writerProperties.find(_.name == p.name).get
                    descriptorProp.wtype match {
                      case "String" => (p.name, value)
                      case "Boolean" => (p.name, value.toBoolean)
                      case "Number" => (p.name, value.toInt)
                      case _ => (p.name, value)
                    }
                  }).toMap
              }).getOrElse(Map())
            }
            case None => Map[String, Any]()
          }

          val scaffolding: String = scaffoldingService.scaffold(inputType.get, dataFormatValue, writerOptions, scaffoldingConfiguration)
          val result: PreviewResult = projectKind.toolingService(classOf[WeaveAgentService]).run(NameIdentifier.ANONYMOUS_NAME, scaffolding, "fake://scaffold.dwl")
          if (result.success) {
            Some(result.content)
          } else {
            None
          }
        } else {
          None
        }
      }

    val maybeFile: Option[File] = projectKind.toolingService(classOf[WeaveScenarioManagerService]).createScenario(NameIdentifier(nameIdentifier), nameOfScenario, inputFileName, content, Some(uri))
    if (maybeFile.isDefined) {
      uiService.openTextDocument(OpenTextDocumentParams(toLSPUrl(maybeFile.get)))
    } else {
      uiService.showMessage(new MessageParams(MessageType.Error, s"Unable to create sample data file ${inputFileName}."))
    }
  }

  override def name(): String = CreateInputSampleCommand.LABEL

  override def description(params: ExecuteCommandParams): String = s"Creating Sampling Data for: ${Commands.argAsString(params.getArguments, 0)}."
}

object CreateInputSampleCommand {
  val LABEL: String = "Create Sample Data."

  def createCommand(nameIdentifier: String,
                               nameOfScenario: String,
                                maybeUri: String,
                               maybeInputName: String,
                               mayBeInputFileName: String): Command = {
    new Command(LABEL,
      Commands.DW_CREATE_INPUT_SAMPLE,
      util.Arrays.asList(
        nameIdentifier,
        nameOfScenario,
        maybeUri,
        maybeInputName,
        mayBeInputFileName
      )
    )
  }
}

