package org.mule.weave.lsp.project

import com.google.gson.JsonObject
import org.eclipse.lsp4j.InitializeParams
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.extension.api.project.WeaveProjectSettings
import org.mule.weave.extension.api.project.WeaveProjectSettings.OptionalStringSetting
import org.mule.weave.extension.api.project.WeaveProjectSettings.StringSetting
import org.mule.weave.lsp.project.Settings._
import org.mule.weave.lsp.project.events.SettingsChangedEvent
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.WeaveDirectoryUtils

import java.io.File
import java.util.Optional
import scala.collection.mutable.ArrayBuffer

case class DefaultProjectMetadata(private val uri: String, override val settings: ProjectSettings) extends ProjectMetadata {
  private val homeFile: File = URLUtils.toFile(uri).get

  def home(): File = {
    homeFile
  }

}

case class ProjectSettings(wlangVersion: DefaultStringSetting = DefaultStringSetting(WLANG_VERSION_PROP_NAME, DEFAULT_VERSION),
                           languageLevelVersion: DefaultStringSetting = DefaultStringSetting(LANGUAGE_LEVEL_PROP_NAME, DEFAULT_VERSION),
                           validationLevel: DefaultStringSetting = DefaultStringSetting(VALIDATION_LEVEL_PROP_NAME, TYPE_LEVEL),
                           enableSemanticTestIndexer: DefaultStringSetting = DefaultStringSetting(ENABLE_SEMANTIC_TEST_INDEXER_PROP_NAME, ENABLE_SEMANTIC_TEST_INDEXER),
                           batVersion: DefaultStringSetting = DefaultStringSetting(BAT_VERSION_PROP_NAME, DEFAULT_BAT_VERSION),
                           batWrapperVersion: DefaultStringSetting = DefaultStringSetting(BAT_WRAPPER_VERSION_PROP_NAME, DEFAULT_BAT_WRAPPER_VERSION),
                           previewTimeout: DefaultStringSetting = DefaultStringSetting(PREVIEW_TIMEOUT, PREVIEW_TIMEOUT_VALUE),
                           useLiteralOnInsertType: DefaultStringSetting = DefaultStringSetting(USE_LITERAL_ON_INSERT, false.toString),
                           scaffoldingElementNumber: DefaultStringSetting = DefaultStringSetting(SCAFFOLDING_REPEATED_ELEMENT, 5.toString),
                           testRunnerEnvVarFileName: DefaultStringSetting = DefaultStringSetting(TEST_RUNNER_ENV_VAR_FILE_NAME, DEFAULT_TEST_RUNNER_ENV_VAR_FILE_NAME),
                           agentFolder: DefaultOptionalStringSetting = DefaultOptionalStringSetting(AGENT_FOLDER_KEY),
                          ) extends WeaveProjectSettings {

  def cloneProjectSettings(): ProjectSettings =
    copy(
      wlangVersion.deepCopy(),
      languageLevelVersion.deepCopy(),
      validationLevel.deepCopy(),
      enableSemanticTestIndexer.deepCopy(),
      batVersion.deepCopy(),
      batWrapperVersion.deepCopy(),
      previewTimeout.deepCopy(),
      useLiteralOnInsertType.deepCopy(),
      scaffoldingElementNumber.deepCopy(),
      testRunnerEnvVarFileName.deepCopy(),
      agentFolder.deepCopy())

  val PROPERTIES: Seq[SettingValue] =
    Seq(
      wlangVersion,
      languageLevelVersion,
      validationLevel,
      enableSemanticTestIndexer,
      batVersion,
      batWrapperVersion,
      previewTimeout,
      useLiteralOnInsertType,
      scaffoldingElementNumber,
      agentFolder
    )

  def load(settings: AnyRef): Array[String] = {
    val modifiedProps = new ArrayBuffer[String]()
    val weaveSettings: JsonObject = settings.asInstanceOf[JsonObject]
    if (weaveSettings != null) {
      PROPERTIES.foreach(prop => {
        if (prop.updateValue(weaveSettings)) {
          modifiedProps.+=(prop.settingName())
        }
      })
    }
    modifiedProps.toArray
  }

  def update(settings: AnyRef): Array[String] = {
    val allSettings: JsonObject = settings.asInstanceOf[JsonObject]
    val modified = ArrayBuffer[String]()
    if (allSettings != null) {
      val weaveSettings = allSettings.getAsJsonObject("data-weave")
      if (weaveSettings != null) {
        modified.++=(load(weaveSettings))
      }
    }
    modified.toArray
  }
}

object Settings {
  val DEFAULT_VERSION = "2.9.0"
  val DEFAULT_BAT_VERSION = "1.24.63"
  val DEFAULT_BAT_WRAPPER_VERSION = "1.0.58"
  val PREVIEW_TIMEOUT_VALUE: String = 50000.toString
  val DEFAULT_BAT_HOME = ".bat"
  val DEFAULT_TEST_RUNNER_ENV_VAR_FILE_NAME = "${workspaceFolder}/.env"
  val TYPE_LEVEL = "type"
  val ENABLE_SEMANTIC_TEST_INDEXER = "false"
  val SCOPE_LEVEL = "scope"
  val PARSE_LEVEL = "parse"
  val WLANG_VERSION_PROP_NAME = "wlangVersion"
  val LANGUAGE_LEVEL_PROP_NAME = "languageLevel"
  val VALIDATION_LEVEL_PROP_NAME = "validationLevel"
  val ENABLE_SEMANTIC_TEST_INDEXER_PROP_NAME = "enableSemanticTestIndexer"
  val BAT_VERSION_PROP_NAME = "batVersion"
  val BAT_WRAPPER_VERSION_PROP_NAME = "batWrapperVersion"
  val PREVIEW_TIMEOUT = "previewTimeout"
  val USE_LITERAL_ON_INSERT = "userLiteralOnInsertAction"
  val SCAFFOLDING_REPEATED_ELEMENT = "scaffoldingRepeatedElement"
  val TEST_RUNNER_ENV_VAR_FILE_NAME = "testRunnerEnvVarFileName"
  val AGENT_FOLDER_KEY = "agentFolder"

  def isTypeLevel(settings: WeaveProjectSettings): Boolean = settings.validationLevel.value() == TYPE_LEVEL

  def isScopeLevel(settings: WeaveProjectSettings): Boolean = settings.validationLevel.value() == SCOPE_LEVEL

  def isParseLevel(settings: WeaveProjectSettings): Boolean = settings.validationLevel.value() == PARSE_LEVEL
}

trait SettingValue {
  def updateValue(v: JsonObject): Boolean

  def settingName(): String
}

case class DefaultStringSetting(settingName: String, initialValue: String) extends StringSetting with SettingValue {

  private var valueHolder: String = initialValue

  def value(): String = valueHolder

  def intValue(): Int = valueHolder.toInt

  def booleanValue(): Boolean = valueHolder.toBoolean

  def updateValue(v: JsonObject): Boolean = {
    Option(v.get(settingName)) match {
      case Some(value) =>
        val newValue = value.getAsString
        if (!newValue.equals(valueHolder)) {
          valueHolder = newValue
          true
        } else {
          false
        }
      case None =>
        false
    }
  }

  def deepCopy(): DefaultStringSetting = {
    val clone = copy()
    clone.valueHolder = this.valueHolder
    clone
  }
}


case class DefaultOptionalStringSetting(settingName: String) extends OptionalStringSetting with SettingValue {

  private var valueHolder: Optional[String] = Optional.empty()

  def value(): Optional[String] = valueHolder

  def updateValue(v: JsonObject): Boolean = {
    Option(v.get(settingName)) match {
      case Some(value) =>
        val newValue = Optional.of(value.getAsString)
        if (!newValue.equals(valueHolder)) {
          valueHolder = newValue
          true
        } else {
          false
        }
      case None =>
        false
    }
  }

  def deepCopy(): DefaultOptionalStringSetting = {
    val clone = copy()
    clone.valueHolder = this.valueHolder
    clone
  }
}

object DefaultProjectSettings {
  def create(): ProjectSettings = {
    ProjectSettings()
  }
}
