package org.mule.weave.lsp.project.commands

import java.io.FileWriter
import java.nio.charset.StandardCharsets
import java.nio.file.Files
import java.nio.file.Path

class DefaultProjectCreator extends ProjectCreator {

  def create(projectInfo: ProjectCreationInfo): Path = {
    val projectPath: Path = projectInfo.pathToCreate.resolve(projectInfo.projectName)
    if (projectInfo.failIfPresent && projectPath.toFile.exists()) {
      throw new RuntimeException("Project already exists: " + projectPath);
    }
    Files.createDirectories(projectPath)
    val replacedPom = getResourceAsString("pom-default.xml")
      .replaceAll("@group-id@", projectInfo.groupId)
      .replaceAll("@artifact-id@", projectInfo.artifactId)
      .replaceAll("@version@", projectInfo.version)
      .replaceAll("@project-name@", projectInfo.projectName)

    writeFile(projectPath.resolve("pom.xml"), replacedPom)

    val sourcePath = projectPath.resolve("src")
    val mainPath = sourcePath.resolve("main")
    val mainDwPath = mainPath.resolve("dw")
    val dwModulePath = mainDwPath.resolve("MyModule.dwl")
    val mainResourcesPath = mainPath.resolve("resources")
    val testPath = sourcePath.resolve("test")
    val testDwPath = testPath.resolve("dw")
    val dwMappingPath = testDwPath.resolve("MyMapping.dwl")
    val testResourcesPath = testPath.resolve("resources")

    Files.createDirectories(mainDwPath)
    Files.createDirectories(mainResourcesPath)
    Files.createDirectories(testDwPath)
    Files.createDirectories(testResourcesPath)

    if (projectInfo.createExampleFiles) {
      val exampleDwModule = getResourceAsString("dw-template-project-module.dwl")
      val exampleDwMapping = getResourceAsString("dw-template-project-mapping.dwl")
      writeFile(dwModulePath, exampleDwModule)
      writeFile(dwMappingPath, exampleDwMapping)
    }
    projectPath
  }

  private def getResourceAsString(resource: String): String = {
    val source = this.getClass.getClassLoader.getResourceAsStream(resource)
    new String(source.readAllBytes(), StandardCharsets.UTF_8)
  }

  private def writeFile(p: Path, content: String): Unit = {
    val writer = new FileWriter(p.toFile)
    try {
      writer.write(content)
    } finally {
      writer.close()
    }
  }
}

trait ProjectCreator {

  def create(projectInfo: ProjectCreationInfo): Path
}

case class ProjectCreationInfo(var groupId: String = "", var artifactId: String = "", var version: String = "", var projectName: String = "", var pathToCreate: Path = null, createExampleFiles: Boolean = true, failIfPresent: Boolean = false)
