package org.mule.weave.v2.metadata.api

import org.mule.metadata.api.annotation.DescriptionAnnotation
import org.mule.metadata.api.annotation.ExampleAnnotation
import org.mule.metadata.api.annotation.LabelAnnotation
import org.mule.metadata.api.annotation.MetadataFormatPropertiesAnnotation
import org.mule.metadata.api.annotation.TypeAliasAnnotation
import org.mule.metadata.api.annotation.TypeAnnotation
import org.mule.metadata.api.annotation.TypeIdAnnotation
import org.mule.metadata.api.annotation.UniquesItemsAnnotation
import org.mule.metadata.api.builder.ArrayTypeBuilder
import org.mule.metadata.api.builder.BaseTypeBuilder
import org.mule.metadata.api.builder.IntersectionTypeBuilder
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder
import org.mule.metadata.api.builder.ObjectKeyBuilder
import org.mule.metadata.api.builder.TypeBuilder
import org.mule.metadata.api.builder.WithAnnotation
import org.mule.metadata.api.model.MetadataFormat
import org.mule.metadata.api.model.MetadataType
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation
import org.mule.weave.v2.metadata.api.MetadataFormatConstant.CSV_ID
import org.mule.weave.v2.metadata.api.MetadataFormatConstant.JAVA_ID
import org.mule.weave.v2.metadata.api.MetadataFormatConstant.JSON_ID
import org.mule.weave.v2.metadata.api.MetadataFormatConstant.WEAVE_DATA_FORMAT
import org.mule.weave.v2.metadata.api.MetadataFormatConstant.XML_ID
import org.mule.weave.v2.model.values.coercion.NumberUtils
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.BinaryType
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.DateTimeType
import org.mule.weave.v2.ts.DynamicReturnType
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.LocalDateTimeType
import org.mule.weave.v2.ts.LocalDateType
import org.mule.weave.v2.ts.LocalTimeType
import org.mule.weave.v2.ts.Metadata
import org.mule.weave.v2.ts.MetadataConstraint
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NamespaceType
import org.mule.weave.v2.ts.NothingType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.PeriodType
import org.mule.weave.v2.ts.RangeType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.RegexType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TimeType
import org.mule.weave.v2.ts.TimeZoneType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.TypeType
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.UriType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.{ SynchronizedIdentityHashMap, ThreadSafe }

import java.util
import javax.xml.namespace.QName
import scala.collection.JavaConverters._
import scala.collection.mutable

/**
  * This Class handles the translation between DataWeave Types to MetadataModel API.
  * Use the config to configure this converter.
  */
@ThreadSafe
class MetadataModelConverter(config: MetadataModelConverterConfig = MetadataModelConverterConfig()) {

  private val referenceBuilder: SynchronizedIdentityHashMap[WeaveType, TypeBuilder[_ <: MetadataType]] = SynchronizedIdentityHashMap()

  def toMuleType(
    wtype: WeaveType,
    format: MetadataFormat,
    sample: Option[String] = None,
    mimeTypeProperties: Map[String, String] = Map()): MetadataType = {
    val builder = transformToMuleType(wtype, format, mimeTypeProperties.isEmpty, sample.isEmpty)
    builder match {
      case t: WithAnnotation[_] => {
        if (sample.isDefined) {
          t.`with`(new ExampleAnnotation(sample.get))
        }
        if (mimeTypeProperties.nonEmpty) {
          t.`with`(new MetadataFormatPropertiesAnnotation(mimeTypeProperties.asJava))
        }
      }
      case _ =>
    }
    builder.build()
  }

  def toMuleType(wtype: WeaveType): MetadataType = {
    transformToMuleType(wtype, getFormat(wtype).getOrElse(WEAVE_DATA_FORMAT)).build()
  }

  def getFormat(wtype: WeaveType): Option[MetadataFormat] = {
    val maybeFormat = wtype
      .getMetadataConstraint(WeaveTypesConverter.METADATA_FORMAT_ANNOTATION)
      .map((annotation) =>
        annotation.value.toString match {
          case CSV_ID  => MetadataFormat.CSV
          case XML_ID  => MetadataFormat.XML
          case JSON_ID => MetadataFormat.JSON
          case JAVA_ID => MetadataFormat.JAVA
          case id      => new MetadataFormat(id.capitalize, id, s"application/${id.toLowerCase}")
        })
    maybeFormat
  }

  private def transformToMuleType(
    wtype: WeaveType,
    format: MetadataFormat,
    addReaderProperties: Boolean = true,
    addExampleAnnotation: Boolean = true,
    skipAnnotations: Boolean = false): TypeBuilder[_ <: MetadataType] = {
    //If it is already mapped returned
    val maybeBuilder: Option[TypeBuilder[_ <: MetadataType]] = referenceBuilder.get(wtype)
    if (maybeBuilder.isDefined) {
      //We need to return :( at this point to avoid modifying it with the annotations
      return maybeBuilder.get
    }

    val metaDataBuilder: BaseTypeBuilder = BaseTypeBuilder.create(format)
    val result: TypeBuilder[_ <: MetadataType] = wtype match {
      case ObjectType(properties, close, ordered) => {
        val objectBuilder = metaDataBuilder.objectType()
        referenceBuilder.put(wtype, objectBuilder)
        if (!close) {
          objectBuilder.open()
        }
        objectBuilder.ordered(ordered)
        properties.foreach((prop) => {
          prop.key match {
            case KeyType(name, attrs) => {
              name match {
                case NameType(Some(qName)) => {
                  val field: ObjectFieldTypeBuilder = objectBuilder.addField()
                  val key: ObjectKeyBuilder = field.key(new QName(qName.ns.orNull, qName.name))
                  processAnnotations(prop.key, field.withKeyAnnotation)
                  field.value(transformToMuleType(prop.value, getFormat(prop.value).getOrElse(format), addReaderProperties, addExampleAnnotation))
                  attrs.foreach((attr) => {
                    val attrName: WeaveType = attr.name
                    attrName match {
                      case NameType(Some(attrQName)) => {
                        val attrBuilder = key.addAttribute()
                        attrBuilder.name(new QName(attrQName.ns.orNull, attrQName.name))
                        attrBuilder.value(transformToMuleType(attr.value, format, addReaderProperties, addExampleAnnotation))
                        attrBuilder.required(!attr.optional)
                      }
                      case _ =>
                    }
                  })
                  field.required(!prop.optional)
                  field.repeated(prop.repeated)
                }
                case nt @ NameType(None) => {
                  val maybePatternConstraint = nt.getMetadataConstraint(WeaveTypesConverter.PATTERN_ANNOTATION)
                  if (maybePatternConstraint.isDefined) {
                    val pattern: java.util.regex.Pattern = java.util.regex.Pattern.compile(maybePatternConstraint.get.value.toString)
                    val fieldTypeBuilder = objectBuilder.addField()
                    fieldTypeBuilder.key(pattern)
                    fieldTypeBuilder.value(transformToMuleType(prop.value, format, addReaderProperties, addExampleAnnotation))
                  } else {
                    objectBuilder.openWith(transformToMuleType(prop.value, format, addReaderProperties, addExampleAnnotation))
                  }
                }
                case _ =>
              }
            }
            case _ =>
          }
        })
        objectBuilder
      }
      case rt: ReferenceType => {
        if (config.useReferenceTypes) {
          val builder = BaseTypeBuilder.create(format).referenceType()
          builder.withName(rt.referenceName())
          builder.withResolver(() => {
            val value = transformToMuleType(rt.resolveType(), format, addReaderProperties, addExampleAnnotation)
            value.build()
          })
        } else {
          val weaveType: WeaveType = rt.resolveType()
          val maybeBuilder: Option[TypeBuilder[_ <: MetadataType]] = referenceBuilder.get(weaveType)
          if (maybeBuilder.isDefined) {
            //We need to return :( at this point to avoid modifying it with the annotations
            return maybeBuilder.get
          } else {
            if (config.structuralIdentity) {
              val option = referenceBuilder
                .find((entry) => TypeHelper.areEqualStructurally(entry._1, weaveType))
                .map(_._2)
              if (option.isDefined) {
                //We need to return :( at this point to avoid modifying it with the annotations
                return option.get
              }
            }

            // Reference types maps its own annotations (constraints & metadata) so we skip annotations to avoid duplicates
            transformToMuleType(weaveType, format, addReaderProperties, addExampleAnnotation, skipAnnotations = true)
          }
        }
      }
      case ArrayType(of) => {
        val arrayTypeBuilder: ArrayTypeBuilder = metaDataBuilder.arrayType()
        referenceBuilder.put(wtype, arrayTypeBuilder)
        arrayTypeBuilder.of(transformToMuleType(of, format, addReaderProperties, addExampleAnnotation))
      }
      case UnionType(of) => {
        if (!config.skipUnionsAsEnums && of.forall(_.baseType().isInstanceOf[StringType])) {
          val builder = metaDataBuilder.stringType()
          val options = of.collect({
            case StringType(Some(value)) => value
          })
          builder.enumOf(options.toArray: _*)
          builder
        } else if (!config.skipUnionsAsEnums && of.forall(_.baseType().isInstanceOf[NumberType])) {
          val builder = metaDataBuilder.numberType()
          val options = of
            .collect({
              case NumberType(Some(value)) => {
                val number = NumberUtils.fromString(value).number
                if (number.isDefined) number.get.underlying() else None
              }
            })
            .filter((p) => p.isInstanceOf[Number])
            .asInstanceOf[Seq[Number]] //It traverse to the Seq two times.
          builder.enumOf(options.toArray: _*)
          builder
        } else {
          val unionBuilder = metaDataBuilder.unionType()
          referenceBuilder.put(wtype, unionBuilder)
          var alreadyAdded = new mutable.HashSet[WeaveType]()
          of.foreach((unionType) => {
            if (!alreadyAdded.contains(unionType.baseType())) {
              unionBuilder.of(transformToMuleType(unionType, format, addReaderProperties, addExampleAnnotation))
              alreadyAdded.+=(unionType.baseType())
            }
          })
          unionBuilder
        }
      }
      case StringType(value) => {
        val stringTypeBuilder = metaDataBuilder.stringType()
        if (!config.skipLiterals && value.nonEmpty) {
          stringTypeBuilder.enumOf(value.get)
        }
        stringTypeBuilder
      }
      case AnyType()         => metaDataBuilder.anyType()
      case BooleanType(_, _) => metaDataBuilder.booleanType()
      case NumberType(value) => {
        val numberTypeBuilder = metaDataBuilder.numberType()
        if (!config.skipLiterals && value.nonEmpty) {
          try {
            numberTypeBuilder.enumOf(value.get.toDouble)
          } catch {
            case _: NumberFormatException =>
          }
        }
        numberTypeBuilder
      }
      case DateTimeType()  => metaDataBuilder.dateTimeType()
      case LocalDateType() => metaDataBuilder.dateType()
      case TimeType()      => metaDataBuilder.timeType()
      case BinaryType()    => metaDataBuilder.binaryType()
      case NullType()      => metaDataBuilder.nullType()
      case NothingType()   => metaDataBuilder.nothingType()
      case RangeType() => {
        val arrayTypeBuilder = metaDataBuilder.arrayType()
        arrayTypeBuilder.of().numberType()
        arrayTypeBuilder
      }
      case KeyType(_, _)       => metaDataBuilder.stringType()
      case NamespaceType(_, _) => metaDataBuilder.stringType()
      case NameType(_)         => metaDataBuilder.stringType()
      case LocalDateTimeType() => metaDataBuilder.localDateTimeType()
      case LocalTimeType()     => metaDataBuilder.localTimeType()
      case TimeZoneType()      => metaDataBuilder.timeZoneType()
      case PeriodType()        => metaDataBuilder.periodType()
      case TypeType(_) => {
        metaDataBuilder.anyType()
      }
      case RegexType() => metaDataBuilder.regexType()
      case FunctionType(_, args, returnType, _, _, _) => {
        val functionType = metaDataBuilder.functionType()
        args.foreach((arg) => {
          if (arg.optional) {
            functionType.addOptionalParameterOf(arg.name, transformToMuleType(arg.wtype, format, addReaderProperties, addExampleAnnotation))
          } else {
            functionType.addParameterOf(arg.name, transformToMuleType(arg.wtype, format, addReaderProperties, addExampleAnnotation))
          }
        })

        functionType.returnType(transformToMuleType(returnType, format, addReaderProperties, addExampleAnnotation))
        functionType
      }
      //This types are not supported
      case UriType(_) => metaDataBuilder.anyType()
      case IntersectionType(of) => {
        val resolvedIntersection = if (config.simplifyIntersection) TypeHelper.resolveIntersection(of) else wtype
        resolvedIntersection match {
          case IntersectionType(of) => {
            val intersectionBuilder: IntersectionTypeBuilder = metaDataBuilder.intersectionType()
            referenceBuilder.put(wtype, intersectionBuilder)
            of.foreach((wt) => {
              intersectionBuilder.of(transformToMuleType(wt, format, addReaderProperties, addExampleAnnotation))
            })
            intersectionBuilder
          }
          case rt => transformToMuleType(rt, format, addReaderProperties, addExampleAnnotation)
        }
      }
      case TypeParameter(name, topType, bottomType, _, _) => {
        topType
          .map(transformToMuleType(_, format, addReaderProperties, addExampleAnnotation))
          .orElse(bottomType.map(transformToMuleType(_, format, addReaderProperties, addExampleAnnotation)))
          .getOrElse(metaDataBuilder.typeParameter(name))
      }
      case _: DynamicReturnType => metaDataBuilder.anyType()
    }
    result match {
      case builder: WithAnnotation[_] => {
        val annotator: TypeAnnotation => _ = builder.`with` _
        val readerProperties = new mutable.HashMap[String, String]()
        if (!skipAnnotations) {
          val typeAnnotations: Seq[TypeAnnotation] = wtype
            .metadataConstraints()
            .flatMap(constraint => {
              constraint.name match {
                case TypeIdAnnotation.NAME                          => Some(new TypeIdAnnotation(constraint.value.toString))
                case TypeAliasAnnotation.NAME                       => Some(new TypeAliasAnnotation(constraint.value.toString))
                case LabelAnnotation.NAME                           => Some(new LabelAnnotation(constraint.value.toString))
                case DescriptionAnnotation.NAME                     => Some(new DescriptionAnnotation(constraint.value.toString))
                case ExampleAnnotation.NAME                         => if (addExampleAnnotation) Some(new ExampleAnnotation(constraint.value.toString)) else None
                case UniquesItemsAnnotation.NAME                    => Some(new UniquesItemsAnnotation())
                //We can ignore this
                case WeaveTypesConverter.METADATA_FORMAT_ANNOTATION => None
                //We can ignore this
                case MetadataConstraint.CLASS_PROPERTY_NAME => {
                  Some(new ClassInformationAnnotation(constraint.value.toString, true, false, true, false, false, new util.ArrayList[String](), "java.lang.Object", new util.ArrayList[String](), false))
                }
                // This is required to ignore documentation metadata
                case Metadata.DOCUMENTATION_ANNOTATION => None
                case name => {
                  readerProperties.put(name, constraint.value.toString)
                  None
                }
              }
            })

          typeAnnotations.foreach(annotator)

          if (addReaderProperties && readerProperties.nonEmpty) {
            annotator(new MetadataFormatPropertiesAnnotation(readerProperties.asJava))
          }

          processAnnotations(wtype, annotator)
        }
      }
      case _ =>
    }
    result
  }

  private def processAnnotations(wtype: WeaveType, annotator: (TypeAnnotation) => _): Unit = {
    if (config.weaveTypeMetadataConvert.isDefined) {
      val convert = config.weaveTypeMetadataConvert.get
      wtype
        .metadata()
        .flatMap(metadata => {
          Option(convert.toTypeAnnotation(metadata))
        })
        .foreach(annotator)
    }
  }
}

case class MetadataModelConverterConfig(
  // If two types are structural equal return the same
  structuralIdentity: Boolean = false,
  // If true union of literals are not going to be transformed to Enums
  skipUnionsAsEnums: Boolean = false,
  // If true literal types are not going to be transformed to enum of a single value
  skipLiterals: Boolean = false,
  // If true Intersection are going to be simplified
  simplifyIntersection: Boolean = false,
  // If true metadata Reference Type is going to be used to represent any Reference to other type instead of inline
  useReferenceTypes: Boolean = true,
  // Callback for metadata convertions
  weaveTypeMetadataConvert: Option[WeaveTypeMetadataConvert] = None)

