%dw 2.0

import * from dw::extension::DataFormat
import readLinesWith, writeLinesWith from dw::core::Binaries
import try, fail from dw::Runtime


type SkipNullSettings = {
    /**
    * Skips null values on the specified data structure. By default it doesn't skip
    */
    skipNullOn?: String {possibleValues: "arrays,objects,objects"}
}

type WriteAttributesSettings = {
    /**
    * If a key has attributes, it will put them as children key-value pairs of the key that contains them. The attribute key name will start with @.
    */
    writeAttributes?: Boolean {defaultValue: false}
}

type NDJsonWriterSettings = EncodingSettings & SkipNullSettings & WriteAttributesSettings


type SkipInvalidSettings = {
    /**
    * Skips invalid records. If any record is not a valid json value it will ignore it.
    */
    skipInvalid? : Boolean {defaultValue: false}
}

type IgnoreEmptyLineSettings = {
    /**
    * Ignores any empty line.
    */
    ignoreEmptyLine? : Boolean {defaultValue: true}
}


fun readNDJson(content: Binary, charset: String, settings: SkipInvalidSettings & IgnoreEmptyLineSettings): Any = do {
   var lines = if(settings.ignoreEmptyLine default true)
                    content readLinesWith charset filter not isEmpty($)
               else
                    content readLinesWith charset
   ---
   if(settings.skipInvalid default false)
      lines
          map ((line, index) -> try(() -> read(line, "application/json")))
          filter ((tryResult, index) -> tryResult.success)
          map ((successResult, index) -> successResult.result!)
   else
     lines
        map ((line, index) -> do{
          var tryRead = try(() -> read(line, "application/json"))
          ---
          if(tryRead.success)
            tryRead.result!
          else
            fail("Exception while reading line\n\n$(index + 1)| $(line)\n\nReason:\n\n$(tryRead.error.message)")
        })
}

fun writeNDJson(content: Any, settings: NDJsonWriterSettings): Binary = do {

    var jsonWriterSettings = {
        (skipNullOn: settings.skipNullOn) if(settings.skipNullOn?),
        (writeAttributes: settings.writeAttributes) if(settings.writeAttributes?),
        indent: false
    }

    var jsonLines = content as Array<String>
                    map ((item, index) -> write(item , "application/json", jsonWriterSettings) as String)

    var encoding = (settings.encoding default "UTF-8")
    ---
    jsonLines writeLinesWith encoding
}

@DataFormatExtension
var ndjson = {
  acceptedMimeTypes: ["application/x-ndjson", "application/x-ldjson"],
  fileExtensions: [".ndjson", ".ldjson", ".ldj", ".jsonl"],
  reader: readNDJson,
  writer: writeNDJson
}

