package org.mule.weave.v2.parser.module

import org.mule.weave.v2.parser.location.UnknownLocation

import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

case class MimeType(mainType: String, subtype: String, parameters: Map[String, String] = Map.empty) {

  def getCharset(): Option[String] = {
    parameters.get("charset")
  }

  def withParameters(newParams: Map[String, String]) = MimeType(mainType, subtype, newParams)

  def isWildcardMainType: Boolean = MimeType.WILDCARD_TYPE == mainType

  def isWildcardSubtype: Boolean = MimeType.WILDCARD_TYPE == subtype || subtype.startsWith("*+")

  def isCompatibleWith(other: MimeType): Boolean = {
    if (other == null) {
      return false
    }
    if (isWildcardMainType || other.isWildcardMainType || mainType == other.mainType) {
      if (subtype == other.subtype) {
        return true
      }
      // wildcard with suffix? e.g. application/*+xml
      if (this.isWildcardSubtype || other.isWildcardSubtype) {
        val thisPlusIdx: Int = subtype.indexOf('+')
        val otherPlusIdx: Int = other.subtype.indexOf('+')
        if (thisPlusIdx == -1 && otherPlusIdx == -1) {
          return true
        } else if (thisPlusIdx != -1 && otherPlusIdx != -1) {
          val thisSubtypeNoSuffix: String = subtype.substring(0, thisPlusIdx)
          val otherSubtypeNoSuffix: String = other.subtype.substring(0, otherPlusIdx)
          val thisSubtypeSuffix: String = subtype.substring(thisPlusIdx + 1)
          val otherSubtypeSuffix: String = other.subtype.substring(otherPlusIdx + 1)
          if (thisSubtypeSuffix == otherSubtypeSuffix && (MimeType.WILDCARD_TYPE == thisSubtypeNoSuffix || MimeType.WILDCARD_TYPE == otherSubtypeNoSuffix)) {
            return true
          }
        }
      }
    }
    false
  }

  override def toString: String = {
    val value = toStringWithoutParameters
    if (parameters.nonEmpty) {
      value + parameters
        .map((keyValue) => {
          val entryValue = quote(keyValue._2)
          s";${keyValue._1}=${entryValue}"
        })
        .reduce(_ + _)
    } else {
      value
    }
  }

  def toStringWithoutParameters: String = {
    s"$mainType/$subtype"
  }
  private def isTokenChar(c: Char) = c > ' ' && c < 127 && "()<>@,;:/[]?=\\\"".indexOf(c) < 0

  private def quote(value: String) = {
    var needsQuotes = false
    val length = value.length
    var i = 0
    while (i < length && !needsQuotes) {
      needsQuotes = !isTokenChar(value.charAt(i))
      i += 1
    }
    if (!needsQuotes) value
    else {
      val buffer = new StringBuffer
      buffer.ensureCapacity((length.toDouble * 1.5D).toInt)
      buffer.append('"')
      var i = 0
      while (i < length) {
        val c = value.charAt(i)
        if (c == '\\' || c == '"') buffer.append('\\')
        buffer.append(c)
        i += 1
      }
      buffer.append('"')
      buffer.toString
    }
  }

  /**
    * Indicate whether this MediaType includes the given media type.
    *
    */
  def includes(other: MimeType): Boolean = {
    if (other == null) return false
    if (this.isWildcardMainType || mainType == other.mainType) {
      if (subtype == other.subtype) return true
      if (this.isWildcardSubtype) {
        // wildcard with suffix, e.g. application/*+xml
        val thisPlusIdx = subtype.indexOf('+')
        if (thisPlusIdx == -1) return true
        else {
          // application/*+xml includes application/soap+xml
          val otherPlusIdx = other.subtype.indexOf('+')
          if (otherPlusIdx != -1) {
            val thisSubtypeNoSuffix = subtype.substring(0, thisPlusIdx)
            val thisSubtypeSuffix = subtype.substring(thisPlusIdx + 1)
            val otherSubtypeSuffix = other.subtype.substring(otherPlusIdx + 1)
            if (thisSubtypeSuffix == otherSubtypeSuffix && MimeType.WILDCARD_TYPE == thisSubtypeNoSuffix) return true
          }
        }
      }
    }
    false
  }

}

object MimeType {

  val MULTIPART_FORM_DATA = new MimeType("multipart", "form-data")

  val WILDCARD_TYPE = "*"

  val CHARSET = "charset"

  val NEEDS_QUOTES = Seq('(', ')', '<', '>', '@', ',', ';', ':', '\\', '<', '>', '/', '[', ']', '?', '.', '=', '\n')

  val NEEDS_ESCAPE = Seq('"', '\\')

  val ANY_MIME_TYPE = "*/*"

  val JAVA_MIME_TYPE = "application/java"

  val INTERNAL_MIME_TYPE = "internal/internal"

  val APPLICATION_JAVA = new MimeType("application", "java")

  val OCTET_STREAM = new MimeType("application", "octet-stream")

  val APPLICATION_JSON = new MimeType("application", "json")

  val APPLICATION_XML = new MimeType("application", "xml")

  val ANY = new MimeType("*", "*")

  def fromSimpleString(mediaType: String): MimeType = {
    //OPTIMIZE FOR MOST COMMON MimeTypes
    if (mediaType.equals(ANY_MIME_TYPE)) {
      ANY
    } else if (mediaType.equals(JAVA_MIME_TYPE)) {
      APPLICATION_JAVA
    } else if (mediaType.equals("application/json")) {
      APPLICATION_JSON
    } else if (mediaType.equals("application/xml")) {
      APPLICATION_XML
    } else {
      val slashIndex: Int = mediaType.indexOf("/")
      val semIndex: Int = mediaType.indexOf(";")
      if (slashIndex < 0 && semIndex < 0) throw new InvalidMimeTypeExpression(mediaType, UnknownLocation)
      else if (slashIndex < 0 && semIndex >= 0) throw new InvalidMimeTypeExpression(mediaType, UnknownLocation)
      else {
        if (slashIndex >= 0 && semIndex < 0) {
          val primaryType = mediaType.substring(0, slashIndex).trim.toLowerCase()
          val subType = mediaType.substring(slashIndex + 1).trim.toLowerCase()
          new MimeType(primaryType, subType)
        } else {
          if (slashIndex >= semIndex) throw new InvalidMimeTypeExpression(mediaType, UnknownLocation)
          val primaryType = mediaType.substring(0, slashIndex).trim.toLowerCase
          val subType = mediaType.substring(slashIndex + 1, semIndex).trim.toLowerCase
          val propertiesString = mediaType.substring(semIndex)
          val properties = parseProperties(propertiesString)
          new MimeType(primaryType, subType, properties)
        }
      }
    }
  }

  def parseProperties(propertiesString: String): Map[String, String] = {
    var i = 0
    var insideQuote = false
    val content = new StringBuilder()
    var name: String = null
    var properties: mutable.ArrayBuffer[(String, String)] = ArrayBuffer()
    while (i < propertiesString.length) {
      val c = propertiesString.charAt(i)
      c match {
        case '"' => {
          insideQuote = !insideQuote
        }
        case _ if insideQuote => {
          content.append(c)
        }
        case '=' if name == null => {
          name = content.toString().trim
          content.clear()
        }
        case ';' => {
          if (name != null && content.nonEmpty) {
            properties.+=((name, content.toString()))
            name = null
            content.clear()
          }
        }
        case '\\' =>
        case _ => {
          content.append(c)
        }
      }
      i = i + 1
    }
    if (name != null && content.nonEmpty) {
      properties.+=((name, content.toString()))
    }
    properties.toMap
  }
}
