package org.mule.weave.v2.scope

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.ImportedElement
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.ts.WeaveTypeReferenceResolver
import org.mule.weave.v2.utils.IdentityHashMap

import scala.collection.mutable
import scala.collection.mutable.ListBuffer

class VariableScope(val parsingContext: ParsingContext, var parentScope: Option[VariableScope], val name: Option[String], val astNode: AstNode, val index: Int) {

  private val _children: ListBuffer[VariableScope] = ListBuffer()
  //References
  private val _references: ListBuffer[NameIdentifier] = ListBuffer()

  //Declarations
  private val _declarations: ListBuffer[NameIdentifier] = ListBuffer()
  private val _importedModules: ListBuffer[(ImportDirective, VariableScope)] = ListBuffer()

  private lazy val navigator = AstNavigator(astNode)

  private val localReferenceCache = mutable.HashMap[String, Option[Reference]]()

  private val referencesToCache = IdentityHashMap[NameIdentifier, Seq[Reference]]()

  //TODO this should be moved outside here as this is from the type phase and not the scope phase
  //But for now we keep it here as it has the same life span we should propagate it same as this one
  private lazy val _referenceResolver: ScopeGraphTypeReferenceResolver = WeaveTypeReferenceResolver(scopesNavigator())

  private lazy val _scopesNavigator: ScopesNavigator = new ScopesNavigator(this);

  /**
    * Returns the scope Navigator of this variable scope
    *
    * @return
    */
  def scopesNavigator(): ScopesNavigator = _scopesNavigator

  def referenceResolver(): ScopeGraphTypeReferenceResolver = _referenceResolver

  def collectVisibleVariables(collector: Reference => Boolean): Seq[Reference] = {
    val localAnnotations = collectLocalVariables(collector)
    val identifiers = _importedModules.flatMap((im) => {
      if (im._1.isImportStart()) {
        im._2.collectLocalVariables(collector).map(_.copy(moduleSource = Some(im._1.importedModule.elementName)))
      } else {
        //TODO we should also take into account the imported by name and alias
        Seq()
      }
    })
    localAnnotations ++ identifiers
  }

  def isRootScope(): Boolean = parentScope.isEmpty

  def collectLocalVariables(collector: Reference => Boolean): mutable.Seq[Reference] = {
    val references: mutable.Seq[Reference] = _declarations.map((ni) => Reference(ni, this)).filter(collector)

    val parentReferences = parentScope.map((ps) => ps.collectLocalVariables(collector)).getOrElse(Seq())

    references ++ parentReferences
  }

  def astNavigator(): AstNavigator = {
    rootScope().navigator
  }

  def moduleName: NameIdentifier = parsingContext.nameIdentifier

  def importedModules(): Seq[(ImportDirective, VariableScope)] = _importedModules

  def addImportedModule(id: ImportDirective, vs: VariableScope): VariableScope = {
    _importedModules += ((id, vs))
    this
  }

  def references(): Seq[NameIdentifier] = {
    _references
  }

  def declarations(): Seq[NameIdentifier] = {
    _declarations
  }

  def children(): Seq[VariableScope] = _children

  def createParent(): VariableScope = {
    val scope = VariableScope(parsingContext, astNode)
    scope._children += this
    parentScope = Some(scope)
    scope
  }

  def rootScope(): VariableScope = {
    parentScope.map(_.rootScope()).getOrElse(this)
  }

  def createParent(_declarations: Seq[NameIdentifier] = Seq(), _references: Seq[NameIdentifier] = Seq()): VariableScope = {
    val scope = createParent()
    scope._declarations ++= _declarations
    scope._references ++= _references
    scope
  }

  def addReference(ref: NameIdentifier): VariableScope = {
    _references.+=(ref)
    this
  }

  def addDeclaration(decl: NameIdentifier): VariableScope = {
    _declarations.+=(decl)
    this
  }

  def addDeclarations(decl: Seq[NameIdentifier]): VariableScope = {
    _declarations.++=(decl)
    this
  }

  def createChild(location: AstNode): VariableScope = {
    val child = VariableScope(parsingContext, this, location)
    _children += child
    child
  }

  def createChild(name: String, location: AstNode): VariableScope = {
    val child = VariableScope(parsingContext, this, name, location)
    _children += child
    child
  }

  def resolveLocalReferenceTo(nameIdentifier: NameIdentifier): Seq[Reference] = {
    def resolveRef(vr: NameIdentifier): Option[Reference] = {
      resolveContextLocalVariable(vr) match {
        case Some(ref) if ref.referencedNode eq nameIdentifier => {
          Some(Reference(vr, this))
        }
        case _ => None
      }
    }

    val childRefs = _children.flatMap(_.resolveReferenceTo(nameIdentifier))
    val result = references().flatMap((ref) => if (ref.name == nameIdentifier.name) resolveRef(ref) else None) ++ childRefs
    result
  }

  def shadowedVariables(): Seq[NameIdentifier] = {
    val identifiers = declarations()
    identifiers.flatMap((identifier) => {
      parentScope.flatMap(_.searchDeclaration(identifier))
    })
  }

  def searchDeclaration(identifier: NameIdentifier): Option[NameIdentifier] = {
    declarations()
      .find(_.name.equals(identifier.name))
      .orElse({
        parentScope.flatMap(_.searchDeclaration(identifier))
      })
  }

  def resolveReferenceTo(nameIdentifier: NameIdentifier): Seq[Reference] = {
    def resolveRef(vr: NameIdentifier): Option[Reference] = {
      resolveVariable(vr) match {
        case Some(ref) if ref.referencedNode eq nameIdentifier => {
          Some(Reference(vr, this))
        }
        case _ => None
      }
    }

    if (referencesToCache.contains(nameIdentifier)) {
      referencesToCache(nameIdentifier)
    } else {
      val localReferences: Seq[Reference] = references().flatMap(resolveRef)
      val childReferences: Seq[Reference] = _children.flatMap(_.resolveReferenceTo(nameIdentifier))
      val result: Seq[Reference] = localReferences ++ childReferences
      referencesToCache.update(nameIdentifier, result)
      result
    }

  }

  def resolveVariable(name: NameIdentifier): Option[Reference] = {
    //We search first in local scope
    //Then on parent scope
    //Last on import
    resolveContextLocalVariable(name)
      .orElse(resolveImportedVariable(name))
      .orElse(rootScope().resolveImportedVariable(name))
      .orElse(resolveFQNVariable(name))
  }

  /**
    * Returns true if that variable is defined in the parent scope
    *
    * @param name The name of the variable
    * @return True if it is defined
    */
  def isDefinedOnParentScope(name: NameIdentifier): Boolean = {
    parentScope.flatMap(_.resolveContextLocalVariable(name)).isDefined
  }

  private def resolveContextLocalVariable(name: NameIdentifier): Option[Reference] = {
    resolveLocalVariable(name)
      .orElse(parentScope.flatMap(_.resolveContextLocalVariable(name)))
  }

  def resolveLocalVariable(name: NameIdentifier): Option[Reference] = {
    localReferenceCache.getOrElseUpdate(name.name, {
      val resolvedLocalVariable: Option[NameIdentifier] = declarations().find(x => x.name == name.name)
      resolvedLocalVariable.map(Reference(_, this))
    })
  }

  def resolveFQNVariable(name: NameIdentifier): Option[Reference] = {
    if (!name.isLocalReference()) {
      val moduleIdentifier: NameIdentifier = name.parent().get
      if (moduleIdentifier.equals(parsingContext.nameIdentifier)) {
        //Referencing  to this module
        resolveContextLocalVariable(name.localName())
      } else {
        val module = parsingContext.getScopeGraphForModule(moduleIdentifier)
        if (module.hasResult()) {
          val scope: VariableScope = module.getResult().scope.rootScope
          val variable = scope.resolveVariable(name.localName())
          toAbsoluteReference(variable, moduleIdentifier)
        } else {
          None
        }
      }
    } else {
      None
    }
  }

  def resolveImportedVariable(name: NameIdentifier): Option[Reference] = {
    _importedModules.toStream
      .flatMap((importModule) => {
        val importDirective: ImportDirective = importModule._1
        val importScope: VariableScope = importModule._2
        val importedElements: Seq[AstNode] = importDirective.subElements.children()

        val maybeAbsoluteReference = if (name.isLocalReference() && importedElements.nonEmpty) {
          resolveLocalRefInImports(name, importDirective, importScope)
        } else if (importedElements.isEmpty) {
          resolveRefWithModuleName(name, importDirective, importScope)
        } else {
          None
        }
        maybeAbsoluteReference
      })
      .headOption
  }

  /**
    * Goes through all the elements in the imported module checking if one matches the name to resolve (star always matches) and returns an absolute reference
    *
    */
  private def resolveLocalRefInImports(nameToResolve: NameIdentifier, importDirective: ImportDirective, importScope: VariableScope): Option[Reference] = {
    val moduleIdentifier: NameIdentifier = importDirective.importedModule.elementName
    val importedElements: Seq[AstNode] = importDirective.subElements.children()

    importedElements.toStream
      .flatMap({
        case ImportedElement(NameIdentifier("*", None), None) => {
          val variable: Option[Reference] = importScope.resolveLocalVariable(nameToResolve)
          toAbsoluteReference(variable, moduleIdentifier)
        }
        case ImportedElement(elementName, alias) => {
          val matchesImportedElemName = alias.getOrElse(elementName).name.equals(nameToResolve.name)
          if (matchesImportedElemName) {
            toAbsoluteReference(importScope.resolveLocalVariable(elementName), moduleIdentifier)
          } else {
            None
          }
        }
      })
      .headOption
  }

  /**
    * This resolves references with the form moduleName::name
    */
  private def resolveRefWithModuleName(nameToResolve: NameIdentifier, importDirective: ImportDirective, importScope: VariableScope) = {
    val moduleIdentifier: NameIdentifier = importDirective.importedModule.elementName

    nameToResolve.parent() match {
      case Some(parentName) if parentName.parent().isEmpty => {
        val matchesImportedModuleName = getLocalModuleName(importDirective).equals(parentName.localName())
        if (matchesImportedModuleName) {
          val nameIdentifier = importScope.resolveLocalVariable(nameToResolve.localName())
          toAbsoluteReference(nameIdentifier, moduleIdentifier)
        } else {
          None
        }
      }
      case _ => None
    }
  }

  private def getLocalModuleName(importDirective: ImportDirective): NameIdentifier = {
    importDirective.importedModule.aliasOrLocalName
  }

  /**
    * This method returns a copy of the reference with the moduleSource added
    */
  private def toAbsoluteReference(maybeReference: Option[Reference], moduleIdentifier: NameIdentifier): Option[Reference] = {
    maybeReference.map((ref: Reference) => ref.copy(moduleSource = Some(moduleIdentifier)))
  }

  def localVariables: Seq[Reference] = declarations().map(Reference(_, this))

  def importedVariables: Seq[Reference] = {
    _importedModules.flatMap {
      case (importDirective, importScope) =>
        val moduleName = importDirective.importedModule.elementName
        importDirective.subElements.children().flatMap {
          case ImportedElement(NameIdentifier("*", None), None) =>
            importScope.visibleVariables.map(_.copy(moduleSource = Some(moduleName)))

          case importedElement: ImportedElement =>
            Seq(Reference(importedElement.aliasOrLocalName, importScope, Some(moduleName)))
        }
    }
  }

  def visibleVariables: Seq[Reference] = {
    val parentDeclarations = parentScope match {
      case Some(parent) => parent.visibleVariables
      case _            => Nil
    }
    val importedVars: Map[String, Reference] = importedVariables.map(x => x.referencedNode.name -> x).toMap
    val parentVarMap: Map[String, Reference] = parentDeclarations.map(x => x.referencedNode.name -> x).toMap
    val localVarMap: Map[String, Reference] = localVariables.map(x => x.referencedNode.name -> x).toMap
    val visibleVars: Seq[Reference] = (localVarMap ++ importedVars ++ parentVarMap).values.toSeq
    visibleVars
  }

}

object VariableScope {
  def apply(parsingContext: ParsingContext, astNode: AstNode): VariableScope = {
    new VariableScope(parsingContext, None, None, astNode, 0)
  }

  def apply(parsingContext: ParsingContext, name: String, astNode: AstNode): VariableScope = {
    new VariableScope(parsingContext, None, Some(name), astNode, 0)
  }

  def apply(parsingContext: ParsingContext, parent: VariableScope, astNode: AstNode): VariableScope = {
    new VariableScope(parsingContext, Some(parent), None, astNode, parent.index + 1)
  }

  def apply(parsingContext: ParsingContext, parent: VariableScope, name: String, astNode: AstNode): VariableScope = {
    new VariableScope(parsingContext, Some(parent), Some(name), astNode, parent.index + 1)
  }
}

case class Reference(referencedNode: NameIdentifier, scope: VariableScope, moduleSource: Option[NameIdentifier] = None) {

  /**
    * True if this reference a local element
    *
    * @return
    */
  def isLocalReference: Boolean = moduleSource.isEmpty

  /**
    * True if this reference is pointing to another module
    *
    * @return
    */
  def isCrossModule: Boolean = moduleSource.nonEmpty

  /**
    * Returns the fully qualified name of this reference. This will name will contain the module part and the local name
    *
    * @return The fully qualified name
    */
  def fqnReferenceName: NameIdentifier = {
    if (moduleSource.isDefined) {
      moduleSource.get.child(referencedNode.name)
    } else {
      scope.moduleName.child(referencedNode.name)
    }
  }
}
