package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.InvalidReferenceMessage
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.scope.ScopesNavigator

/**
  * This phase applies scope resolution
  *
  * @tparam T the ast node type
  */
class ScopeGraphPhase[T <: AstNode] extends CompilationPhase[AstNodeResultAware[T] with ParsingContentAware, ScopeGraphResult[T]] {

  override def doCall(node: AstNodeResultAware[T] with ParsingContentAware, ctx: ParsingContext): PhaseResult[ScopeGraphResult[T]] = {
    val navigator: ScopesNavigator = ScopesNavigator(node.astNode, ctx)
    val invalidRefs: Seq[AstNode] = navigator.invalidReferences()
    invalidRefs.foreach((refNode) => {
      ctx.messageCollector.error(InvalidReferenceMessage(refNode), refNode.location())
    })
    SuccessResult(new ScopeGraphResult(node.input, node.astNode, navigator), ctx)
  }

}

object ScopeGraphPhase {
  def apply[T <: AstNode](): CompilationPhase[AstNodeResultAware[T] with ParsingContentAware, ScopeGraphResult[T]] = {
    new ScopeGraphPhase[T]()
      .enrichWith(new ForwardReferenceValidationPhase[T]())
      .enrichWith(new TypeParameterCheckerPhase[T]())
  }
}

trait ScopeNavigatorResultAware {
  def scope: ScopesNavigator
}

class ScopeGraphResult[T <: AstNode](_input: ParsingContentInput, _astNode: T, val scope: ScopesNavigator) extends ParsingResult[T](_input, _astNode) with ScopeNavigatorResultAware

object ScopeGraphResult {
  def unapply[T <: AstNode](result: ScopeGraphResult[T]): Option[(ParsingContentInput, T, ScopesNavigator)] = {
    Some((result.input, result.astNode, result.scope))
  }
}