package org.mule.weave.v2.editor.quickfix

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.ImportedElement
import org.mule.weave.v2.parser.ast.header.directives.ImportedElements
import org.mule.weave.v2.parser.ast.header.directives.VersionDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

class InsertImportQuickFix(val editorSupport: WeaveEditorSupport, val variable: String, val module: NameIdentifier) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {
    val importedElement = ImportedElement(NameIdentifier(variable))
    editorSupport.astDocument() match {
      case Some(ast: DocumentNode) => {
        val directives = ast.header.directives
        runAddImport(directives, importedElement, document)
      }
      case Some(ast: ModuleNode) => {
        val directives = ast.directives
        runAddImport(directives, importedElement, document)
      }
      case None =>
    }
  }

  private def runAddImport(directives: Seq[DirectiveNode], importedElement: ImportedElement, document: WeaveTextDocument): Unit = {
    val existingImport: Option[DirectiveNode] = directives
      .find({
        case ImportDirective(importedModule, elements, _) => {
          importedModule.elementName.equals(module) && elements.elements.nonEmpty
        }
        case _ => false
      })
    existingImport match {
      case Some(importD: ImportDirective) => {
        val newImport = importD.copy(subElements = ImportedElements(importD.subElements.elements :+ importedElement))
        document.delete(importD.location().startPosition.index, importD.location().endPosition.index)
        document.insert(CodeGenerator.generate(newImport).trim + "\n", importD.location().startPosition.index)
      }
      case _ => {
        //Represents (true, position) start position
        //Represents (false, position) end position
        val startPositionPair: Option[(Boolean, Int)] = directives
          .flatMap({
            case id: ImportDirective if (!AstNodeHelper.isInjectedNode(id)) => {
              Some((true, id.location().startPosition.index))
            }
            case _ => None
          })
          .headOption
          .orElse({
            directives
              .flatMap({
                case id: VersionDirective => {
                  Some((false, id.location().endPosition.index))
                }
                case _ => None
              })
              .headOption
          })
        val newImport = ImportDirective(ImportedElement(module), ImportedElements(Seq(importedElement)))
        val noHeader = directives.isEmpty || directives.forall(AstNodeHelper.isInjectedNode(_))
        if (noHeader) {
          document.insert(s"%dw 2.0\n${CodeGenerator.generate(newImport).trim}\n---\n", 0)
        } else {
          if (startPositionPair.isEmpty) {
            document.insert(CodeGenerator.generate(newImport).trim + "\n", 0)
          } else {
            if (startPositionPair.get._1) {
              document.insert(CodeGenerator.generate(newImport).trim + "\n", startPositionPair.get._2)
            } else {
              document.insert("\n" + CodeGenerator.generate(newImport).trim, startPositionPair.get._2)
            }
          }
        }
      }
    }
  }
}
