package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.grammar.AsOpId
import org.mule.weave.v2.grammar.IsOpId
import org.mule.weave.v2.parser.annotation.MaterializeTypeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.patterns.TypePatternNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.WeaveType

/**
  * This phase will mark all usages of types that may need materialization.
  */
class MaterializeTypeMarkerPhase[R <: AstNode, T <: AstNodeResultAware[R] with ScopeNavigatorResultAware] extends CompilationPhase[T, T] {
  override def doCall(source: T, context: ParsingContext): PhaseResult[_ <: T] = {
    val astNavigator: AstNavigator = source.scope.astNavigator()
    val binaryOpNodes = astNavigator.allWithType(classOf[BinaryOpNode])
    val typeReferenceResolver = new ScopeGraphTypeReferenceResolver(source.scope)

    binaryOpNodes.foreach(op => {
      op.opId match {
        case AsOpId | IsOpId =>
          val needsMaterialize = typeNeedsMaterialize(op.rhs, typeReferenceResolver)
          op.rhs.annotate(MaterializeTypeAnnotation(needsMaterialize))
        case _ =>
      }
    })

    val typePatternNodes = astNavigator.allWithType(classOf[TypePatternNode])
    typePatternNodes.foreach(pattern => {
      val needsMaterialize = typeNeedsMaterialize(pattern.pattern, typeReferenceResolver)
      pattern.pattern.annotate(MaterializeTypeAnnotation(needsMaterialize))
    })

    val functionNodes = astNavigator.allWithType(classOf[FunctionNode])
    functionNodes.foreach(function => {
      function.params.paramList.foreach(param => {
        if (param.wtype.isDefined) {
          val wType = param.wtype.get
          val needsMaterialize = typeNeedsMaterialize(wType, typeReferenceResolver)
          wType.annotate(MaterializeTypeAnnotation(needsMaterialize))
        }
      })
    })

    SuccessResult(source, context)
  }

  private def typeNeedsMaterialize(typeNode: AstNode, typeReferenceResolver: ScopeGraphTypeReferenceResolver): Boolean = {
    //Only if it is a complex object needs to be consumed else is ok
    typeNode match {
      case weaveTypeNode: WeaveTypeNode =>
        TypeHelper.requiredMaterialize(WeaveType(weaveTypeNode, typeReferenceResolver))
      case _ => true
    }
  }
}
