package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.types.DynamicReturnTypeNode
import org.mule.weave.v2.scope.ScopesNavigator

object FunctionTypeHelper {

  val ANONYMOUS_FUNCTION_NAME = "AnonymousFunction"

  def createDynamicParameter(i: Int): WeaveType = {
    TypeParameter(s"${WeaveTypeResolver.UNSPECIFIED_PARAMETER_TYPE_PREFIX}${i}")
  }

  def isDynamicFunction(ft: FunctionType): Boolean = {
    ft.params.exists((param) => {
      val wtype = param.wtype
      isDynamicTypeParameter(wtype)
    })
  }

  def hasDynamicParameter(dynamicReturnType: DynamicReturnType): Boolean = {
    dynamicReturnType.typeParameters.map(_.wtype).exists(FunctionTypeHelper.isDynamicTypeParameter)
  }

  def collectDynamicFunctions(ft: FunctionType): Seq[FunctionType] = WeaveTypeTraverse.shallowCollectAll(ft, {
    case t: FunctionType if FunctionTypeHelper.isDynamicFunction(t) => Seq(t)
    case _ => Seq()
  })

  def isDynamicTypeParameter(wtype: WeaveType): Boolean = {
    wtype match {
      case tp: TypeParameter => tp.name.startsWith(WeaveTypeResolver.UNSPECIFIED_PARAMETER_TYPE_PREFIX)
      case _                 => false
    }
  }

  def isDynamicReturnType(weaveType: WeaveType): Boolean = {
    weaveType.isInstanceOf[DynamicReturnType]
  }

  def hasDynamicReturn(weaveType: WeaveType): Boolean = {
    WeaveTypeTraverse.exists(weaveType, {
      case _: DynamicReturnType => true
      case _                    => false
    })
  }

  def hasNothingType(weaveType: WeaveType): Boolean = {
    WeaveTypeTraverse.exists(weaveType, {
      case _: NothingType => true
      case _              => false
    })
  }

  def getFunctionSubGraphFor(drt: DynamicReturnType, ctx: WeaveTypeResolutionContext): Option[TypeNode] = {
    val DynamicReturnType(args, functionNode, _, _, _, _, _) = drt
    ctx.getFunctionSubGraph(functionNode, args.map(_.wtype)).flatMap(_.findLocalNode(functionNode))
  }

  def resolveReturnType(argTypes: Seq[WeaveType], returnType: Option[WeaveType], ctx: WeaveTypeResolutionContext, dynamicReturnType: DynamicReturnType, strict: Boolean, collector: MessageCollector = new MessageCollector()): Option[WeaveType] = {
    val functionNode: FunctionNode = dynamicReturnType.node
    val scopesNavigator: ScopesNavigator = dynamicReturnType.scope
    val resolver: ReferenceResolver = dynamicReturnType.resolver
    val typeGraph: TypeGraph = dynamicReturnType.typeGraph

    val subGraph: Option[TypeGraph] = ctx.getFunctionSubGraph(functionNode, argTypes)
    if (subGraph.isDefined) {
      subGraph.get.findLocalNode(functionNode).get.resultType()
    } else {
      val context = if (strict) {
        ctx.currentParsingContext
      } else {
        ctx.currentParsingContext.withMessageCollector(collector)
      }
      //If the expected output type was not specified select from the function node
      val declaredReturnType: Option[WeaveType] =
        returnType
          .orElse(functionNode.returnType.flatMap((rt) => {
            rt match {
              case _: DynamicReturnTypeNode => None
              case _                        => Some(WeaveType(rt, scopesNavigator.rootScope.referenceResolver()))
            }
          }))

      val dataGraph: TypeGraph = TypeGraph(context, typeGraph, scopesNavigator, functionNode, argTypes, declaredReturnType, resolver)
      ctx.addFunctionSubGraph(functionNode, argTypes, dataGraph)
      ctx.newExecutorWithContext(scopesNavigator, dataGraph, context).run()
      val value = dataGraph.findLocalNode(functionNode).get
      val maybeType = value.resultType()
      if (maybeType.isEmpty && !strict && collector.hasErrors()) {
        ctx.removeFunctionSubGraph(functionNode, argTypes)
      }
      maybeType
    }
  }

}
