package org.mule.weave.v2.editor.bat

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.ValidationMessage
import org.mule.weave.v2.editor.bat.BatUIModelResultFactory.fail
import org.mule.weave.v2.editor.bat.BatUIModelResultFactory.success
import org.mule.weave.v2.grammar.AsOpId
import org.mule.weave.v2.grammar.ValueSelectorOpId
import org.mule.weave.v2.parser.InvalidSyntaxMessage
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.selectors.NullSafeNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.KeyNode
import org.mule.weave.v2.parser.ast.structure.KeyValuePairNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NullNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.ObjectNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.utils.StringEscapeHelper.unescapeString

class BatUIModelParser {

  def parse(batScript: String, name: String, pc: ParsingContext): BatUIModelResult = {
    val value = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource(name, batScript), pc)
    if (value.hasErrors()) {
      val seq = value.errorMessages()
      val messages = toValidationMessages(seq)
      BatUIModelResult(messages, None)
    } else {
      val testBody = value.getResult().astNode.root
      handleInFunction(testBody)
    }
  }

  private def toValidationMessages(seq: Seq[(WeaveLocation, Message)]) = {
    seq.map((pair) => ValidationMessage(pair._1, pair._2))
  }

  def isValidAssertionSourceExpression(expression: String, pc: ParsingContext): BatUIModelValidationResult = {
    val value = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource("assertionSourceExpression", "$.response." + expression), pc)
    if (value.hasErrors()) {
      BatUIModelValidationResult(toValidationMessages(value.errorMessages()))
    } else {
      val result = handleAssertion(value.getResult().astNode.root, NullNode(), "equals")
      result match {
        case Left(v)  => BatUIModelValidationResult(toValidationMessages(Seq(v)))
        case Right(_) => BatUIModelValidationResult()
      }
    }
  }

  def isValidAssertionExpectedExpression(expression: String, pc: ParsingContext): BatUIModelValidationResult = {
    val value = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource("assertionSourceExpression", expression), pc)
    if (value.hasErrors()) {
      BatUIModelValidationResult(toValidationMessages(value.errorMessages()))
    } else {
      BatUIModelValidationResult()
    }
  }

  def isValidExpression(expression: String, pc: ParsingContext): BatUIModelValidationResult = {
    val value = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource("assertionSourceExpression", expression), pc)
    if (value.hasErrors()) {
      BatUIModelValidationResult(toValidationMessages(value.errorMessages()))
    } else {
      BatUIModelValidationResult()
    }
  }

  private def handleInFunction(testBody: AstNode): BatUIModelResult = {
    testBody match {
      case inFunctionCall @ FunctionCallNode(_, FunctionCallParametersNode(Seq(suiteFunctionCall: FunctionCallNode, tests: ArrayNode)), _, _) if (isINFunctionCall(inFunctionCall)) => {
        suiteFunctionCall match {
          case suiteFunctionCall @ FunctionCallNode(_, FunctionCallParametersNode(Seq(testName)), _, _) if (isCallingTo(suiteFunctionCall, "suite")) => {
            handleSuiteFunctionCall(testName, tests)
          }
          case suiteFunctionCall @ FunctionCallNode(_, FunctionCallParametersNode(Seq(ArrayNode(Seq(testName), _), ArrayNode(Seq(), _))), _, _) if (isCallingTo(suiteFunctionCall, "suite")) => {
            handleSuiteFunctionCall(testName, tests)
          }
          case _ => {
            fail((testBody.location(), InvalidSyntaxMessage("Expecting `suite` function call")))
          }
        }
      }
      case _ => {
        fail((testBody.location(), InvalidSyntaxMessage("Expecting `in` function call.")))
      }
    }
  }

  private def isINFunctionCall(inFunctionCall: FunctionCallNode) = {
    isCallingTo(inFunctionCall, "in")
  }

  private def handleSuiteFunctionCall(testName: AstNode, testsNode: ArrayNode): BatUIModelResult = {
    testName match {
      case sn: StringNode => {
        val suiteName = sn.value
        val tests = handleTestCases(testsNode, None, None)
        val errors = collectErrors(tests)
        if (errors.isEmpty) {
          success(BatUIModel(suiteName, tests.map((test) => test.right.get)))
        } else {
          fail(errors.head.left.get)
        }
      }
      case node => {
        fail((node.location(), InvalidSyntaxMessage(s"Expecting string literal value.")))
      }
    }
  }

  private def handleTestCases(testsNode: ArrayNode, testKind: Option[String], testCaseName: Option[String]): Seq[Either[(WeaveLocation, InvalidSyntaxMessage), BatTestUIModel]] = {
    val tests: Seq[Either[(WeaveLocation, InvalidSyntaxMessage), BatTestUIModel]] = testsNode.elements.flatMap({

      case inFunctionCall @ FunctionCallNode(_, FunctionCallParametersNode(Seq(testCase: FunctionCallNode, assertList: ArrayNode)), _, _) if (isINFunctionCall(inFunctionCall)) => {
        testCase match {
          case FunctionCallNode(VariableReferenceNode(testCaseKind: NameIdentifier, _), FunctionCallParametersNode(Seq(VariableReferenceNode(itName: NameIdentifier, _), StringNode(testName, _))), _, _) if (itName.name.equals("it")) => {
            handleTestCases(assertList, Some(testCaseKind.name), Some(testName))
          }
          case node => {
            Some(Left((node.location(), InvalidSyntaxMessage(s"Expecting function call such as: it must 'get a new deck of Cards' in []"))))
          }
        }
      }
      case assertFC @ FunctionCallNode(_, FunctionCallParametersNode(Seq(withFunction, assertList: ArrayNode)), _, _) if (isCallingTo(assertFC, "assert")) => {
        val withFunctionCall = handleWithFunctionCall(withFunction)
        withFunctionCall match {
          case Right(methodCall) => {
            val assertions = handleAssertions(assertList)
            assertions match {
              case Right(assertionResult) => {
                Some(Right(BatTestUIModel(methodCall, assertionResult, testCaseName, testKind)))
              }
              case Left(error) => Some(Left(error))
            }
          }
          case Left(error) => {
            Some(Left(error))
          }
        }
      }
      case node => {
        Some(Left((node.location(), InvalidSyntaxMessage(s"Expecting function call such as: GET `http://acme.com` with {} assert []"))))
      }
    })
    tests
  }

  private def handleAssertions(assertList: ArrayNode) = {
    val assertions = assertList.elements.map({
      case FunctionCallNode(vrn: VariableReferenceNode, FunctionCallParametersNode(Seq(source, target)), _, _) => {
        val assertionType = vrn.variable.name
        handleAssertion(source, target, assertionType)
      }
      case BinaryOpNode(opId, source: AstNode, target: AstNode, _) => {
        val assertionType = opId.name
        handleAssertion(source, target, assertionType)
      }
      case node => Left((node.location(), InvalidSyntaxMessage("Expecting assertion expression such as `$.response.headers.ServerName mustEqual 'Apache'`")))
    })

    val errors: Seq[Left[(WeaveLocation, InvalidSyntaxMessage), Nothing]] = collectErrors(assertions)
    if (errors.nonEmpty) {
      errors.head
    } else {
      Right({
        assertions.map((assertion) => {
          assertion.right.get
        })
      })
    }
  }

  private def handleAssertion(source: AstNode, target: AstNode, assertionType: String): Either[(WeaveLocation, InvalidSyntaxMessage), BatAssertionUIModel] = {
    source match {
      case NullSafeNode(node, _) => {
        val selectorPattern = AstNodeHelper.collectChildren(node, {
          case BinaryOpNode(ValueSelectorOpId, _: VariableReferenceNode, NameNode(selector: StringNode, None, _), _) if (selector.value == "response") => true
          case _ => false
        })
        if (selectorPattern.nonEmpty) {
          val rootExpression = CodeGenerator.generate(selectorPattern.head)
          val fullExpression = CodeGenerator.generate(node)
          val expected = target match {
            case sn: StringNode  => BatExpressionValue(BatExpressionType.STRING, unescapeString(sn.literalValue))
            case sn: NumberNode  => BatExpressionValue(BatExpressionType.NUMBER, sn.literalValue)
            case sn: BooleanNode => BatExpressionValue(BatExpressionType.BOOLEAN, sn.literalValue)
            case _               => BatExpressionValue(BatExpressionType.EXPRESSION, CodeGenerator.generate(target))
          }

          Right(BatAssertionUIModel(fullExpression.substring(rootExpression.length + 1), assertionType, expected))
        } else {
          Left((source.location(), InvalidSyntaxMessage("Expecting assertion expression such as `$.response.headers.ServerName mustEqual 'Apache'`")))
        }
      }
      case _ => {
        Left((source.location(), InvalidSyntaxMessage("Expecting assertion expression such as `$.response.headers.ServerName mustEqual 'Apache'`")))
      }
    }
  }

  private def handleWithFunctionCall(withFunction: AstNode): Either[(WeaveLocation, InvalidSyntaxMessage), BatRequestUIModel] = {
    withFunction match {
      case wfc: FunctionCallNode if (isCallingTo(wfc, "with")) => {
        wfc.args.args match {
          case Seq(httpMethodCall: FunctionCallNode, withParams: ObjectNode) => {
            val httpMethodCallResult = handleHttpMethodCall(httpMethodCall)
            httpMethodCallResult match {
              case Right(mc) => {
                val unsafeSSL = handleAllowUnsafeSSL(withParams)
                unsafeSSL match {
                  case Right(mayBeUnsafeSSL) => {
                    val header = handleHeaders(withParams)
                    header match {
                      case Right(headerMap) => {
                        val body = handleBody(withParams)
                        body match {
                          case Right(bodyString) => {
                            Right(BatRequestUIModel(mc, headerMap, bodyString, mayBeUnsafeSSL))
                          }
                          case Left(error) => Left(error)
                        }
                      }
                      case Left(error) => Left(error)
                    }
                  }
                  case Left(error) => Left(error)
                }

              }
              case Left(error) => Left(error)
            }
          }
          case _ => Left((withFunction.location(), InvalidSyntaxMessage(s"Expecting function call such as: GET `http://acme.com` with {}")))
        }
      }
    }
  }

  private def handleBody(withParams: ObjectNode): Either[(WeaveLocation, InvalidSyntaxMessage), Option[BatRequestUIBody]] = {
    val mayBeBody = AstNodeHelper.selectFirstFieldValue("body", withParams)
    mayBeBody match {
      case Some(BinaryOpNode(AsOpId, str: LiteralValueAstNode, rhs: WeaveTypeNode, _)) => Right(Some(BatRequestUIBody(str.literalValue, BatBodyExpressionType.LITERAL)))
      case Some(n) => Right(Some(BatRequestUIBody(CodeGenerator.generate(n), BatBodyExpressionType.EXPRESSION)))
      case None => Right(None)
    }
  }

  private def handleAllowUnsafeSSL(withParams: ObjectNode): Either[(WeaveLocation, InvalidSyntaxMessage), Option[Boolean]] = {
    val allowUnsafeSSLSelection = AstNodeHelper.selectFirstFieldValue("allowUnsafeSSL", withParams)
    allowUnsafeSSLSelection match {
      case Some(BooleanNode(literalValue, _)) => Right(Some(java.lang.Boolean.parseBoolean(literalValue)))
      case Some(n)                            => Left((n.location(), InvalidSyntaxMessage("Expecting `allowUnsafeSSL` field to be a Boolean Expression.")))
      case _                                  => Right(None)
    }
  }

  private def handleHeaders(withParams: ObjectNode): Either[(WeaveLocation, InvalidSyntaxMessage), Map[String, String]] = {
    val mayBeHeaders = AstNodeHelper.selectFirstFieldValue("headers", withParams)
    mayBeHeaders match {
      case Some(objectNode: ObjectNode) => {
        val mappedHeaders: Seq[Either[(WeaveLocation, InvalidSyntaxMessage), (String, String)]] = objectNode.elements.map({
          case KeyValuePairNode(KeyNode(str: StringNode, _, _, _), stringNode: LiteralValueAstNode, None) => {
            Right((str.value, stringNode.literalValue))
          }
          case n => Left((n.location(), InvalidSyntaxMessage("Expecting `headers` field to be an Object of Literal Keys and Values")))
        })
        val errors: Seq[scala.Left[(WeaveLocation, InvalidSyntaxMessage), Nothing]] = collectErrors(mappedHeaders)
        if (errors.nonEmpty) {
          errors.head
        } else {
          Right(mappedHeaders.map((tuple) => tuple.right.get).toMap)
        }
      }
      case Some(n) => Left((n.location(), InvalidSyntaxMessage("Expecting `headers` field to be an Object of Literal Keys and Values")))
      case None    => Right(Map[String, String]())
    }
  }

  private def collectErrors(mappedHeaders: Seq[Either[(WeaveLocation, InvalidSyntaxMessage), Any]]): Seq[Left[(WeaveLocation, InvalidSyntaxMessage), Nothing]] = {
    val errors = mappedHeaders.collect({
      case l: Left[(WeaveLocation, InvalidSyntaxMessage), Nothing] => l
    })
    errors
  }

  private def handleHttpMethodCall(httpMethodCall: FunctionCallNode): Either[(WeaveLocation, InvalidSyntaxMessage), BatEndpointUIModel] = {
    httpMethodCall.args.args match {
      case Seq(ArrayNode(Seq(url: LiteralValueAstNode), _), _) => {
        val urlString = url.literalValue
        httpMethodCall.function match {
          case vrn: VariableReferenceNode => {
            val methodName = vrn.variable.name
            Right(BatEndpointUIModel(methodName, urlString))
          }
          case node => Left((node.location(), InvalidSyntaxMessage(s"Expecting variable reference to either `GET`, `PUT`, `POST`, `PATCH`")))
        }
      }
      case _ => Left((httpMethodCall.location(), InvalidSyntaxMessage(s"Expecting `http method call` such as: GET `http://acme.com`")))
    }
  }

  private def isCallingTo(fcn: FunctionCallNode, name: String) = {
    fcn.function match {
      case vrn: VariableReferenceNode => vrn.variable.name == name
      case _                          => false
    }
  }
}

object BatUIModelResultFactory {
  def fail(errorMessage: (WeaveLocation, Message)): BatUIModelResult = {
    BatUIModelResult(Seq(ValidationMessage(errorMessage._1, errorMessage._2)), None)
  }

  def success(uiModel: BatUIModel): BatUIModelResult = {
    BatUIModelResult(Seq(), Some(uiModel))
  }
}

case class BatUIModelValidationResult(errorMessages: Seq[ValidationMessage] = Seq()) {
  def success(): Boolean = errorMessages.isEmpty
}

case class BatUIModelResult(errorMessages: Seq[ValidationMessage], result: Option[BatUIModel]) {
  def success(): Boolean = result.isDefined
}

case class BatUIModel(name: String, tests: Seq[BatTestUIModel])

case class BatTestUIModel(request: BatRequestUIModel, assertions: Seq[BatAssertionUIModel], name: Option[String], testKind: Option[String])

case class BatRequestUIModel(endpoint: BatEndpointUIModel, headers: Map[String, String], body: Option[BatRequestUIBody], allowUnsafeSSL: Option[Boolean])

case class BatRequestUIBody(value: String, expressionType: String)

case class BatEndpointUIModel(method: String, endpoint: String)

case class BatAssertionUIModel(source: String, expression: String, value: BatExpressionValue)

case class BatExpressionValue(valueType: String, value: String)

object BatExpressionType {
  val STRING = "String"
  val NUMBER = "Number"
  val BOOLEAN = "Boolean"
  val EXPRESSION = "Expression"
}

object BatBodyExpressionType {
  val LITERAL = "literal"
  val EXPRESSION = "expression"
}
