package org.mule.weave.v2.grammar

import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.MissingExpressionErrorAstNode
import org.mule.weave.v2.parser.annotation.CustomInterpolationAnnotation
import org.mule.weave.v2.parser.annotation.InjectedNodeAnnotation
import org.mule.weave.v2.parser.annotation.InterpolationExpressionAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper.markInjectedNode
import org.mule.weave.v2.parser.ast.functions._
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.EmptyStringNode
import org.mule.weave.v2.parser.ast.types.{ TypeParametersApplicationListNode, TypeParametersListNode, WeaveTypeNode }
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
import org.parboiled2._
import org.parboiled2.support.hlist.::
import org.parboiled2.support.hlist.HNil

trait Functions extends PositionTracking with Tokens with Variables with Annotations with TypeLiteral with VersionControl {
  this: Functions with Grammar =>

  def booleanOrExpr: Rule1[AstNode]

  def value: Rule1[AstNode]

  def filterSelectorLambda: Rule1[FunctionNode] = rule {
    pushPosition ~ push(None) ~ (push(FunctionParameters(List(FunctionParameter(NameIdentifier.$), FunctionParameter(NameIdentifier.$$)))) ~ push(None) ~ (booleanOrExpr | missingExpression("Missing filter selector expression.")) ~> createFunctionNode) ~ injectPosition
  }

  def lambdaLiteral: Rule1[FunctionNode] = rule {
    pushPosition ~ optional(typeParametersList) ~ ws ~ lambdaParameters ~ ws ~ optional(objFieldSep ~ !objFieldSep ~ typeExpression) ~ ws ~ lambdaMark ~!~ ws ~ (expr | missingExpression("Missing Lambda Expression.")) ~> createFunctionNode ~ injectPosition
  }

  def functionLiteral: Rule1[FunctionNode] = namedRule("Function Body Definition e.g (name: String) = name") {
    pushPosition ~
      optional(typeParametersList) ~
      ws ~
      functionParameters ~
      ws ~
      optional(objFieldSep ~ !objFieldSep ~ (typeExpression | dynamicReturnTypeExpression | missingTypeExpression)) ~
      ws ~ /*return type*/
      (assignment ~!~
        ws ~ /*Function Body*/
        (expr | missingExpression("Missing Function Body Expression. i.e fun a(name: String) = 'DW Rock $(a)'")) |
        missingToken("Missing Function Body Separator. i.e fun a(name: String) = 'DW Rock $(a)'", "=")) ~> createFunctionNode ~ injectPosition
  }

  def functionCallArgs: Rule1[FunctionCallParametersNode] = rule {
    (pushPosition ~ parenStart ~!~ zeroOrMore(expr).separatedBy(commaSep) ~ optional(commaSep ~ missingExpression("Missing Parameter Expression")) ~ (parenEnd | fail("')' for the function call.")) ~> createFunctionCallParametersNode) ~
      injectPosition
  }

  def functionCall = namedRule("Function Call") {
    optional(since("2.5") ~ typeParameterApplicationList ~ ws) ~ optional(functionCallArgs) ~> ((
      fn: AstNode,
      typeParameters: Option[TypeParametersApplicationListNode],
      arguments: Option[FunctionCallParametersNode]) => {
      arguments match {
        case Some(value)                      => push(resolveStartPosition(fn) :: FunctionCallNode(fn, value, typeParameters) :: HNil)
        case None if typeParameters.isDefined => push(resolveStartPosition(fn) :: FunctionCallNode(fn, createFunctionCallParametersNode(Seq(), Some(markInjectedNode(MissingExpressionErrorAstNode("missing function call")))), typeParameters) :: HNil)
        case _                                => MISMATCH
      }
    }) ~ injectPosition[FunctionCallNode]
  }

  var createFunctionCallParametersNode = (arguments: Seq[AstNode], missingExpression: Option[AstNode]) => {
    FunctionCallParametersNode(arguments ++ missingExpression)
  }

  def customInterpolation: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Custom String Interpolation") {
    (customStringInterpolation ~> createCustomInterpolationNode) ~ injectPosition
  }

  def customStringInterpolation: Rule1[AstNode] = rule {
    pushPosition ~ backTickString ~ injectPosition
  }

  val createCustomInterpolationNode = (fn: AstNode, interpolated: AstNode) => {

    var strings: Seq[AstNode] = Seq()
    var values: Seq[AstNode] = Seq()

    val usedElements = interpolated.children()

    if (usedElements.isEmpty) {
      strings :+= {
        val node = EmptyStringNode()
        node._location = Some(WeaveLocation(interpolated.location().endPosition, interpolated.location().endPosition, interpolated.location().resourceName))
        node
      }
    } else {
      var previousWasInterpolationLiteral = false

      usedElements.foreach(node => {
        val actualIsInterpolationLiteral = node.annotation(classOf[InterpolationExpressionAnnotation]).isEmpty

        if (!previousWasInterpolationLiteral && !actualIsInterpolationLiteral) {
          strings :+= {
            val node = EmptyStringNode()
            node._location = Some(WeaveLocation(interpolated.location().endPosition, interpolated.location().endPosition, interpolated.location().resourceName))
            node
          }
        }

        if (actualIsInterpolationLiteral) {
          strings :+= node
        } else {
          values :+= node
        }

        previousWasInterpolationLiteral = actualIsInterpolationLiteral

      })

      if (!previousWasInterpolationLiteral)
        strings :+= {
          val node = EmptyStringNode()
          node._location = Some(WeaveLocation(interpolated.location().endPosition, interpolated.location().endPosition, interpolated.location().resourceName))
          node
        }
    }

    val arrayStrings = ArrayNode(strings)
    arrayStrings.annotate(InjectedNodeAnnotation())
    //TODO don't uncomment this we need to create a CustomStringInterpolationNode and transform this into Function Call when compiling it. This breaks the AST TREE location
    //    arrayStrings._location = Some(WeaveLocation(strings.head.location().startPosition, strings.last.location().endPosition, interpolated.location().resourceName))
    val arrayValues = ArrayNode(values)
    arrayValues.annotate(InjectedNodeAnnotation())
    arrayValues._location = {
      if (values.isEmpty) {
        Some(WeaveLocation(interpolated.location().endPosition, interpolated.location().endPosition, interpolated.location().resourceName))
      } else {
        Some(WeaveLocation(values.head.location().startPosition, values.last.location().endPosition, interpolated.location().resourceName))
      }
    }
    val functionCallParametersNode = FunctionCallParametersNode(Seq(arrayStrings, arrayValues))
    functionCallParametersNode._location = Some(interpolated.location())
    val functionCallNode = FunctionCallNode(fn, functionCallParametersNode)
    functionCallNode.annotate(CustomInterpolationAnnotation())

    resolveStartPosition(fn) :: functionCallNode :: HNil
  }

  val createFunctionNode = (typeParameterList: Option[TypeParametersListNode], parameters: FunctionParameters, wtype: Option[WeaveTypeNode], ret: AstNode) => {
    FunctionNode(parameters, ret, wtype, typeParameterList)
  }

  val createClojureTwoParamsNode = (firstParam: FunctionParameter, secondParam: Option[FunctionParameter], ret: AstNode) => {
    FunctionNode(FunctionParameters(Seq(firstParam, secondParam.getOrElse(FunctionParameter(NameIdentifier.$$)))), ret)
  }

  val createSingleParameterFunctionNode = (parameter: FunctionParameter, ret: AstNode) => {
    FunctionNode(FunctionParameters(Seq(parameter, FunctionParameter(NameIdentifier.$$))), ret)
  }

  def functionParameters: Rule1[FunctionParameters] = rule {
    pushPosition ~ (parenStart ~!~ zeroOrMore(parameter).separatedBy(commaSep) ~ quiet(optional(commaSep ~ missingFunctionParameter())) ~!~ (parenEnd | fail("missing `)` for the function parameters. fun a() = 1")) ~> createFunctionParameters) ~ injectPosition
  }

  def lambdaParameters: Rule1[FunctionParameters] = rule {
    pushPosition ~ (parenStart ~ zeroOrMore(parameter).separatedBy(commaSep) ~ quiet(optional(commaSep ~ missingFunctionParameter())) ~ parenEnd ~> createFunctionParameters) ~ injectPosition
  }

  def parameter: Rule1[FunctionParameter] = rule {
    pushPosition ~ (annotations ~ ws ~ namedVariable ~ optional(ws ~ objFieldSep ~!~ (typeExpression | missingTypeExpression())) ~ optional(ws ~ variableInitialValue) ~> createFunctionParameter) ~ injectPosition
  }
}
