package org.mule.weave.v2.parser

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.api.tooling.impl.message.ToolingPhaseCategory
import org.mule.weave.v2.api.tooling.message.{ Message => ApiMessage }
import org.mule.weave.v2.api.tooling.message.{ MessageCollector => ApiMessageCollector }
import org.mule.weave.v2.api.tooling.message.{ ValidationMessage => ApiValidationMessage }
import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.QuickFix
import org.mule.weave.v2.inspector.EqualsBooleanInspectorFixAction
import org.mule.weave.v2.inspector.IfNotNullQuickFixAction
import org.mule.weave.v2.inspector.ReduceObjectToDynamicObjectAction
import org.mule.weave.v2.inspector.RemoveUnusedElement
import org.mule.weave.v2.inspector.RemoveUnusedImport
import org.mule.weave.v2.inspector.ReplaceDefaultWithDefaultValueFixAction
import org.mule.weave.v2.inspector.ReplaceDefaultWithValueFixAction
import org.mule.weave.v2.inspector.ReplaceUsingWithDoFixAction
import org.mule.weave.v2.inspector.SizeOfEqualsZeroFixAction
import org.mule.weave.v2.inspector.SizeOfGraterToZeroFixAction
import org.mule.weave.v2.inspector.TypeOfQuickFixAction
import org.mule.weave.v2.inspector.UnnecessaryDoubleNegationQuickFixAction
import org.mule.weave.v2.inspector.UnnecessaryIfQuickFixAction
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.conditional.DefaultNode
import org.mule.weave.v2.parser.ast.functions.UsingNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.FunctionTypeHelper
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.SelectionPath
import org.mule.weave.v2.ts.TypeCoercer
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.utils.StringHelper
import org.mule.weave.v2.utils.StringHelper.indent
import org.mule.weave.v2.utils.WeaveTypeEmitter

import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.collection.mutable.ListBuffer

/**
  * Collects error messages and warning messages
  */
class MessageCollector extends ApiMessageCollector {
  private val _errorMessages: ListBuffer[(WeaveLocation, Message)] = mutable.ListBuffer[(WeaveLocation, Message)]()
  private val _warningMessages: ListBuffer[(WeaveLocation, Message)] = mutable.ListBuffer[(WeaveLocation, Message)]()

  def errorMessages: Seq[(WeaveLocation, Message)] = _errorMessages

  def warningMessages: Seq[(WeaveLocation, Message)] = _warningMessages

  def mergeWith(messageCollector: MessageCollector): MessageCollector = {
    messageCollector._errorMessages.foreach(message => error(message._2, message._1))
    messageCollector._warningMessages.foreach(message => warning(message._2, message._1))
    this
  }

  def hasErrors(): Boolean = {
    _errorMessages.nonEmpty
  }

  def foreachErrorMessage(callback: Message => Unit): MessageCollector = {
    _errorMessages.foreach(pair => callback(pair._2))
    this
  }

  def error(message: Message, location: WeaveLocation): MessageCollector = {
    val exists = _errorMessages.exists(x => x._2 == message && x._1 == location)
    if (!exists) {
      _errorMessages.+=((location, message))
    }
    this
  }

  def warning(message: Message, location: WeaveLocation): MessageCollector = {
    val exists = _warningMessages.exists(x => x._2 == message && x._1 == location)
    if (!exists) {
      _warningMessages.+=((location, message))
    }
    this
  }

  def errorMessageString(): String = {
    _errorMessages
      .map(entry => {
        toMessageString(entry)
      })
      .mkString("\n")
  }

  def warningMessageString(): String = {
    _warningMessages
      .map(entry => {
        toMessageString(entry)
      })
      .mkString("\n")

  }

  private def toMessageString(entry: (WeaveLocation, Message)): String = {
    Message.toMessageString(entry._1, entry._2)
  }

  override def addErrorValidationMessage(validationMessage: ApiValidationMessage): ApiMessageCollector = {
    error(MessageAdapter(validationMessage.getMessage), validationMessage.getLocation.asInstanceOf[WeaveLocation])
  }

  override def addWarningValidationMessage(validationMessage: ApiValidationMessage): ApiMessageCollector = {
    warning(MessageAdapter(validationMessage.getMessage), validationMessage.getLocation.asInstanceOf[WeaveLocation])
  }

  override def getErrorValidationMessages: Array[ApiValidationMessage] = {
    _errorMessages.map(m => {
      new ApiValidationMessage(m._2, m._1)
    }).toArray
  }

  override def getWarningValidationMessages: Array[ApiValidationMessage] = {
    _warningMessages.map(m => {
      new ApiValidationMessage(m._2, m._1)
    }).toArray
  }
}

class MessageAdapter(delegate: ApiMessage) extends Message {

  override def kind: String = delegate.getKind

  override def message: String = delegate.getMessage

  override def category: MessageCategory = ToolingPhaseCategory

}

object MessageAdapter {
  def apply(delegate: ApiMessage) = new MessageAdapter(delegate)
}

object MessageCollector {
  def apply(): MessageCollector = new MessageCollector()
}

object Message {
  def apply(kind: String, message: String, category: MessageCategory): Message = new DefaultMessage(kind, message, category)

  def toMessageString(location: WeaveLocation, message: Message): String = {
    val result = new mutable.StringBuilder()
    result.append(message.message)
    result.append("\n")
    result.append(location.locationString)
    result.append("\nLocation:\n")
    result.append(location.resourceWithLocation())
    result.toString()
  }
}

class DefaultMessage(override val kind: String, override val message: String, override val category: MessageCategory) extends Message

trait TypeMessage extends Message {

  def expectedType: WeaveType

  def actualType: WeaveType

  private var trace: ArrayBuffer[() => String] = ArrayBuffer()

  protected var details: Seq[String] = Seq()

  def addTrace(wt: WeaveType): TypeMessage = {
    trace.+=(() => typeToString(wt))
    this
  }

  def addTrace(prefix: String, wt: WeaveType): TypeMessage = {
    trace.+=(() => prefix + typeToString(wt))
    this
  }

  def addDetail(detail: String): TypeMessage = {
    details = details :+ detail
    this
  }

  def message: String = {
    val origin =
      if (trace.isEmpty) ""
      else {
        "\n\t" + trace.zipWithIndex
          .map(pair => {
            "|" + ("--" * (pair._2 + 1)) + " From: " + pair._1()
          })
          .mkString("\n\t")
      }
    val detailsStr = if (details.nonEmpty) {
      " Hint: " ++ details.mkString("\n")
    } else ""

    errorMessage + detailsStr + origin
  }

  def errorMessage: String

}

object MessageKind {
  val UNUSED_IMPORT: String = "UnusedImport"
  val ACCESS_VIOLATION_MESSAGE_KIND: String = "AccessViolation"
  val ARRAY_FUNCTION_INJECTION_NOT_POSSIBLE_MESSAGE_KIND: String = "ArrayFunctionInjectionNotPossible"
  val CLOSE_DOES_NOT_ALLOW_EXTRA_PROPERTIES_MESSAGE_KIND: String = "CloseDoesNotAllowExtraProperties"
  val CLOSE_DOES_NOT_ALLOW_OPEN_MESSAGE_KIND: String = "CloseDoesNotAllowOpen"
  val COMPLEX_TYPE_MESSAGE_KIND: String = "ComplexType"
  val CYCLIC_WEAVE_IMPORT_MESSAGE_KIND: String = "CyclicWeaveImport"
  val DENIED_FUNCTION_USAGE_MESSAGE_KIND: String = "DeniedFunctionUsage"
  val DEPRECATED_FEATURE_MESSAGE_KIND: String = "Deprecated"
  val DUPLICATED_NAMESPACE_MESSAGE_KIND: String = "DuplicatedNamespace"
  val DUPLICATED_PARAMETER_MESSAGE_KIND: String = "DuplicatedParameter"
  val DUPLICATED_SYNTAX_DIRECTIVE_MESSAGE_KIND: String = "DuplicatedSyntaxDirective"
  val DUPLICATED_VARIABLE_MESSAGE_KIND: String = "DuplicatedVariable"
  val DYNAMIC_ACCESS_MESSAGE_KIND: String = "DynamicAccess"
  val DYNAMIC_FUNCTIONS_CAN_NOT_BE_CHECKED_MESSAGE_KIND: String = "DynamicFunctionsCanNotBeChecked"
  val ERROR_DIRECTIVE_MESSAGE_KIND: String = "ErrorDirective"
  val EXPERIMENTAL_FEATURE_MESSAGE_KIND: String = "ExperimentalFeature"
  val EXPRESSION_PATTERN_FORCE_MATERIALIZE_MESSAGE_KIND: String = "ExpressionPatternForceMaterialize"
  val FORWARD_REFERENCE_MESSAGE_KIND: String = "ForwardReference"
  val FUNCTION_INJECTION_NOT_POSSIBLE_MESSAGE_KIND: String = "FunctionInjectionNotPossible"
  val FUNCTION_INVALID_DEFAULT_VALUE_MESSAGE_KIND: String = "FunctionInvalidDefaultValue"
  val FUNCTION_IS_NOT_TAIL_REC_CAPABLE_MESSAGE_KIND: String = "FunctionIsNotTailRecCapable"
  val FUNCTION_NAME_CLASHED_WITH_VARIABLE_MESSAGE_KIND: String = "FunctionNameClashedWithVariable"
  val INCOMPATIBLE_RUNTIME_VERSION_MESSAGE_KIND: String = "IncompatibleRuntimeVersion"
  val INVALID_ARGUMENT_NAME_MESSAGE_KIND: String = "InvalidArgumentName"
  val INVALID_AMOUNT_OF_ANNOTATION_ARGUMENTS_MESSAGE_KIND: String = "InvalidAmountOfAnnotationArguments"
  val INVALID_AMOUNT_TYPE_PARAMETERS_MESSAGE_KIND: String = "InvalidAmountTypeParameters"
  val INVALID_ANNOTATION_REFERENCE_MESSAGE_KIND: String = "InvalidAnnotationReference"
  val INVALID_ANNOTATION_TARGET_MESSAGE_KIND: String = "InvalidAnnotationTarget"
  val INVALID_DIRECTIVE_IN_DO_BLOCK_MESSAGE_KIND: String = "InvalidDirectiveInDoBlock"
  val INVALID_DYNAMIC_RETURN_MESSAGE_KIND: String = "InvalidDynamicReturn"
  val INVALID_EQUAL_COMPARISON_MESSAGE_KIND: String = "InvalidEqualComparison"
  val INVALID_FIELD_NAME_IDENTIFIER_MESSAGE_KIND: String = "InvalidFieldNameIdentifier"
  val INVALID_INPUT_DIRECTIVE_IN_MODULE_MESSAGE_KIND: String = "InvalidInputDirectiveInModule"
  val INVALID_JSON_SCHEMA_MESSAGE_KIND: String = "InvalidJsonSchema"
  val INVALID_XML_SCHEMA_MESSAGE_KIND: String = "InvalidXmlSchema"
  val INVALID_AVRO_SCHEMA_MESSAGE_KIND: String = "InvalidAvroSchema"
  val INVALID_METHOD_TYPES_MESSAGE_KIND: String = "InvalidMethodTypes"
  val INVALID_NAME_IDENTIFIER_MESSAGE_KIND: String = "InvalidNameIdentifier"
  val INVALID_OUTPUT_DIRECTIVE_IN_MODULE_MESSAGE_KIND: String = "InvalidOutputDirectiveInModule"
  val INVALID_OVERLOADED_FUNCTION_TYPE_MESSAGE_KIND: String = "InvalidOverloadedFunctionType"
  val INVALID_REFERENCE_MESSAGE_KIND: String = "InvalidReference"
  val INVALID_REGEX_MESSAGE_KIND: String = "InvalidRegex"
  val INVALID_SYNTAX_DIRECTIVE_PLACEMENT_MESSAGE_KIND: String = "InvalidSyntaxDirectivePlacement"
  val INVALID_SYNTAX_MESSAGE_KIND: String = "InvalidSyntax"
  val INVALID_TYPE_DECLARATION_MESSAGE_KIND: String = "InvalidTypeDeclaration"
  val INVALID_TYPE_PARAMETER_CALL_MESSAGE_KIND: String = "InvalidTypeParameterCall"
  val INVALID_TYPE_REF_MESSAGE_KIND: String = "InvalidTypeRef"
  val INVALID_URI_MESSAGE_KIND: String = "InvalidUri"
  val INVALID_VARIABLE_NAME_IDENTIFIER_MESSAGE_KIND: String = "InvalidVariableNameIdentifier"
  val INVALID_WEAVE_DOC_SYNTAX_MESSAGE_KIND: String = "InvalidWeaveDocSyntax"
  val INVALID_WEAVE_VERSION_MESSAGE_KIND: String = "InvalidWeaveVersion"
  val LANGUAGE_FEATURE_NOT_AVAILABLE_MESSAGE_KIND: String = "LanguageFeatureNotAvailable"
  val MAX_TYPE_GRAPH_EXECUTION_MESSAGE_KIND: String = "MaxTypeGraphExecution"
  val METADATA_PARAMETER_SHOULD_BE_VALUE_MESSAGE_KIND: String = "MetadataParameterShouldBeValue"
  val METADATA_ANNOTATION_AND_INJECTOR_COLLISION_MESSAGE_KIND: String = "MetadataAnnotationInjectorCollision"
  val METADATA_ANNOTATION_DUPLICATED_KEY: String = "MetadataAnnotationDuplicatedKey"
  val METADATA_ANNOTATION_MISSING_VALUE: String = "MetadataAnnotationMissingValue"
  val MISSING_DATA_FORMAT_DEFINITION_MESSAGE_KIND: String = "MissingDataFormatDefinition"
  val MISSING_EXPRESSION_MESSAGE_KIND: String = "MissingExpression"
  val MISSING_FORMAT_DEFINITION_MESSAGE_KIND: String = "MissingFormatDefinition"
  val MISSING_FUNCTION_DEFINITION_MESSAGE_KIND: String = "MissingFunctionDefinition"
  val MISSING_FUNCTION_PARAMETER_MESSAGE_KIND: String = "MissingFunctionParameter"
  val MISSING_KEY_SELECTOR_MESSAGE_KIND: String = "MissingKeySelector"
  val MISSING_NAME_IDENTIFIER_MESSAGE_KIND: String = "MissingNameIdentifier"
  val MISSING_REQUIRED_ANNOTATION_ARGUMENT: String = "MissingRequiredAnnotationArgument"
  val MISSING_REQUIRED_PROPERTY_MESSAGE_KIND: String = "MissingRequiredProperty"
  val MISSING_SELECTOR_EXPRESSION_MESSAGE_KIND: String = "MissingSelectorExpression"
  val MISSING_SCHEMA_MESSAGE_KIND: String = "MissingSchema"
  val MISSING_TOKEN_MESSAGE_KIND: String = "MissingToken"
  val MISSING_VERSION_MAJOR_MESSAGE_KIND: String = "MissingVersionMajor"
  val MISSING_VERSION_MINOR_MESSAGE_KIND: String = "MissingVersionMinor"
  val MULTIPLE_VALID_FUNCTIONS_MESSAGE_KIND: String = "MultipleValidFunctions"
  val NATIVE_FUNCTION_PARAMETER_NOT_ANNOTATED_MESSAGE_KIND: String = "NativeFunctionParameterNotAnnotated"
  val NEGATIVE_INDEX_ACCESS_MESSAGE_KIND: String = "NegativeIndexAccess"
  val USING_UPSERT: String = "UsingUpsert"
  val NESTED_TYPE_MESSAGE_KIND: String = "NestedType"
  val NO_COERCION_AVAILABLE_MESSAGE_KIND: String = "NoCoercionAvailable"
  val NOT_ENOUGH_ARGUMENT_MESSAGE_KIND: String = "NotEnoughArgument"
  val NOT_STREAMABLE_OPERATOR_MESSAGE_KIND: String = "NotStreamableOperator"
  val NOT_STREAMABLE_PARAMETER_MESSAGE_KIND: String = "NonStreamableParameter"
  val ONLY_ONE_DEFAULT_PATTERN_MESSAGE_KIND: String = "OnlyOneDefaultPattern"
  val PARAMETER_IS_NOT_STREAM_CAPABLE_MESSAGE_KIND: String = "ParameterIsNotStreamCapable"
  val PROPERTY_NOT_DEFINED_MESSAGE_KIND: String = "PropertyNotDefined"
  val REDUCE_OBJECT_TO_DYNAMIC_OBJECT_MESSAGE_KIND: String = "ReduceObjectToDynamicObject"
  val REPEATED_FIELD_NOT_SUPPORTED_MESSAGE_KIND: String = "RepeatedFieldNotSupported"
  val REPLACE_USING_WITH_DO_MESSAGE_KIND: String = "ReplaceUsingWithDo"
  val REVERT_SELECTION_MESSAGE_KIND: String = "RevertSelection"
  val SELF_IMPORT_MODULE_MESSAGE_KIND: String = "SelfImportingModule"
  val SIMPLIFY_BOOLEAN_EQUALITY_MESSAGE_KIND: String = "SimplifyBooleanEquality"
  val TOO_MANY_ARGUMENT_MESSAGE_KIND: String = "TooManyArgument"
  val TYPE_COERCED_MESSAGE_KIND: String = "TypeCoerced"
  val TYPE_MISMATCH_MESSAGE_KIND: String = "TypeMismatch"
  val UNABLE_TO_RESOLVE_MODULE_MESSAGE_KIND: String = "UnableToResolveModule"
  val UNEXPECTED_EXCEPTION_MESSAGE_KIND: String = "UnexpectedException"
  val UNFULFILLED_CONSTRAINT_MESSAGE_KIND: String = "UnfulfilledConstraint"
  val UNNECESSARY_DOUBLE_NEGATION_MESSAGE_KIND: String = "UnnecessaryDoubleNegation"
  val UNNECESSARY_IF_EXPRESSION_MESSAGE_KIND: String = "UnnecessaryIfExpression"
  val USING_DEFAULT_WITH_LITERAL_VALUE_MESSAGE_KIND: String = "UsingDefaultWithLiteralValue"
  val USING_DEFAULT_WITH_NULL_VALUE_MESSAGE_KIND: String = "UsingDefaultWithNullValue"
  val USING_IF_NOT_NULL_INSTEAD_OF_DEFAULT_MESSAGE_KIND: String = "UsingIfNotNullInsteadOfDefault"
  val USING_IF_NULL_INSTEAD_OF_DEFAULT_MESSAGE_KIND: String = "UsingIfNullInsteadOfDefault"
  val USING_SIZE_OF_TO_COMPARE_EMPTY_MESSAGE_KIND: String = "UsingSizeOfToCompareEmpty"
  val USING_SIZE_OF_TO_COMPARE_NON_EMPTY_MESSAGE_KIND: String = "UsingSizeOfToCompareNonEmpty"
  val USING_TYPE_OF_TO_COMPARE_TYPES_MESSAGE_KIND: String = "UsingTypeOfToCompareTypes"
  val VARIABLE_MODULE_OPEN_ACCESS_MESSAGE_KIND: String = "VariableModuleOpenAccess"
  val VARIABLE_NAME_CLASHED_WITH_IMPLICIT_INPUT_MESSAGE_KIND: String = "VariableNameClashedWithImplicitInput"
  val VARIABLE_REFERENCED_MORE_THAN_ONCE_MESSAGE_KIND: String = "VariableReferencedMoreThanOnce"
  val VARIABLE_REFERENCED_IN_OTHER_SCOPE_MESSAGE_KIND: String = "VariableReferencedInOtherScope"
  val TYPE_SELECTOR_KEY_NOT_FOUND: String = "TypeSelectorKeyNotFound"
  val TYPE_SELECTOR_KEY_STRING_INTERPOLATION: String = "TypeSelectorKeyStringInterpolation"
  val TYPE_SELECTION_NOT_ALLOWED_ON_TYPE: String = "TypeSelectionNotAllowedOnType"
  val SHOULD_TYPE_DYNAMIC_GENERIC_FUNCTION: String = "ShouldTypeDynamicGenericFunction"
  val INVALID_KEY_TYPE: String = "InvalidKeyType"
  val INVALID_NAME_TYPE: String = "InvalidNameType"
}

@WeaveApi(Seq("Tooling"))
trait Message extends ApiMessage {

  val DEFAULT_CODE_LENGTH = 20

  /**
    * The kind message.
    * It's used by mule to classify the messages so it is considered as API
    *
    * @return the message kind
    */
  @WeaveApi(Seq("Tooling"))
  def kind: String

  /**
    * The text message
    *
    * @return The text
    */
  def message: String

  /**
    * The category of this message
    *
    * @return
    */
  def category: MessageCategory

  /**
    * Generates the String of a given Type. It will try to make it pretty print for messages.
    *
    * @param theType The type to emit
    * @return The text of the type
    */
  def typeToString(theType: WeaveType): String = {
    typeToString(theType, "`", "`")
  }

  def typeToString(theType: WeaveType, prefix: String, suffix: String): String = {
    WeaveTypeEmitter.toPresentableString(theType, prefix, suffix)
  }

  override def getKind: String = {
    kind
  }

  override def getMessage: String = {
    message
  }
}

/**
  * A message that knows how to fix itself
  */
trait QuickFixAwareMessage extends Message {

  /**
    * Returns the quick fixes available for this message
    *
    * @return The list of all possible fixes
    */
  def quickFixes(): Array[QuickFix]
}

object MessageCategoryHelper {
  def isSyntaxMessage(message: Message): Boolean = {
    message.category eq ParsingPhaseCategory
  }

  def isCanonicalMessage(message: Message): Boolean = {
    message.category eq CanonicalPhaseCategory
  }

  def isScopeMessage(message: Message): Boolean = {
    message.category eq ScopePhaseCategory
  }

  def isTypeCheckMessage(message: Message): Boolean = {
    message.category eq TypePhaseCategory
  }
}

trait MessageCategory {
  def name: String
}

case class UnusedImportModule(id: ImportDirective) extends QuickFixAwareMessage {

  override def quickFixes(): Array[QuickFix] = {
    Array(QuickFix("Remove unused import directive.", s"Remove unused import statement: `${id.importedModule.elementName}`.", new RemoveUnusedImport(id)))
  }

  override def kind: String = MessageKind.UNUSED_IMPORT

  override def message: String = {
    s"Import directive `${id.importedModule.elementName}` is not being used."
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class UnusedImportElement(id: ImportDirective, element: NameIdentifier) extends QuickFixAwareMessage {

  override def quickFixes(): Array[QuickFix] = {
    Array(QuickFix(s"Remove unused element `${element}` from import directive.", s"Remove unused element: `${element}` from import statement: `${id.importedModule.elementName}`.", new RemoveUnusedElement(id, element)))
  }

  override def kind: String = MessageKind.UNUSED_IMPORT

  override def message: String = {
    s"Element `${element}` is not being used."
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class DynamicFunctionsCanNotBeChecked() extends Message {
  override def kind: String = MessageKind.DYNAMIC_FUNCTIONS_CAN_NOT_BE_CHECKED_MESSAGE_KIND

  override def message: String = "Invoking dynamic functions can not be validated."

  override def category: MessageCategory = ScopePhaseCategory
}

case class ReduceObjectToDynamicObject(expression: AstNode, containerNode: AstNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.REDUCE_OBJECT_TO_DYNAMIC_OBJECT_MESSAGE_KIND

  override def message: String = {
    "Merging object using reduce can be simplified with using a `Dynamic Elements` expression."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace reduce Objects with `Dynamic Elements` expression.",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(containerNode), DEFAULT_CODE_LENGTH)}` with `Dynamic Elements`.",
        new ReduceObjectToDynamicObjectAction(expression, containerNode)))
  }
}

case class ReplaceUsingWithDo(usingNode: UsingNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.REPLACE_USING_WITH_DO_MESSAGE_KIND

  override def message: String = {
    "using() is deprecated, replace with do-block"
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace `using()` with `do-block`",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(usingNode), DEFAULT_CODE_LENGTH)}` with do-block",
        new ReplaceUsingWithDoFixAction(usingNode)))
  }
}

case class UsingDefaultWithLiteralValue(defaultNode: DefaultNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.USING_DEFAULT_WITH_LITERAL_VALUE_MESSAGE_KIND

  override def message: String = {
    "The `default` Expression value will never be null"
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace `default` Expression with value",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(defaultNode), DEFAULT_CODE_LENGTH)}` with `${StringHelper.abbreviate(CodeGenerator.generate(defaultNode.lhs), DEFAULT_CODE_LENGTH)}`",
        new ReplaceDefaultWithValueFixAction(defaultNode)))
  }
}

case class UsingDefaultWithNullValue(defaultNode: DefaultNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.USING_DEFAULT_WITH_NULL_VALUE_MESSAGE_KIND

  override def message: String = {
    "The `default` Expression value will always be null"
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace `default` Expression with default value",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(defaultNode), DEFAULT_CODE_LENGTH)}` with `${StringHelper.abbreviate(CodeGenerator.generate(defaultNode.rhs), DEFAULT_CODE_LENGTH)}`",
        new ReplaceDefaultWithDefaultValueFixAction(defaultNode)))
  }
}

case class UnnecessaryIfExpression(ifNode: AstNode, ifCondition: AstNode, negate: Boolean) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.UNNECESSARY_IF_EXPRESSION_MESSAGE_KIND

  override def message: String = {
    s"If expression is not necessary and can be simplified."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Simplify if expression.",
        s"Simplify `${StringHelper.abbreviate(CodeGenerator.generate(ifNode), DEFAULT_CODE_LENGTH)}`.",
        new UnnecessaryIfQuickFixAction(ifNode, ifCondition, negate)))
  }
}

case class UnnecessaryDoubleNegation(negation: AstNode, expression: AstNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.UNNECESSARY_DOUBLE_NEGATION_MESSAGE_KIND

  override def message: String = {
    s"Unnecessary double negation can be simplified."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Simplify double negation expression.",
        s"Simplify `${StringHelper.abbreviate(CodeGenerator.generate(negation), DEFAULT_CODE_LENGTH)}`.",
        new UnnecessaryDoubleNegationQuickFixAction(negation, expression)))
  }
}

case class UsingSizeOfToCompareNonEmpty(expression: AstNode, containerNode: AstNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.USING_SIZE_OF_TO_COMPARE_NON_EMPTY_MESSAGE_KIND

  override def message: String = {
    s"Checking `non emptiness` with `sizeOf` is not performant use `!isEmpty`."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace With `!isEmpty` Function Call.",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(containerNode), DEFAULT_CODE_LENGTH)}` with `!isEmpty` Function Call.",
        new SizeOfGraterToZeroFixAction(expression, containerNode)))
  }
}

case class UsingSizeOfToCompareEmpty(expression: AstNode, containerNode: AstNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.USING_SIZE_OF_TO_COMPARE_EMPTY_MESSAGE_KIND

  override def message: String = {
    s"Checking `emptiness` with `sizeOf` is not performant use `isEmpty`."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace With `isEmpty` Function Call.",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(containerNode), DEFAULT_CODE_LENGTH)}` with `isEmpty` Function Call.",
        new SizeOfEqualsZeroFixAction(expression, containerNode)))
  }
}

case class SimplifyBooleanEquality(expression: BinaryOpNode, simplification: String) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.SIMPLIFY_BOOLEAN_EQUALITY_MESSAGE_KIND

  override def message: String = {
    s"Simplify `${CodeGenerator.generate(expression)}`. $simplification"
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(QuickFix(message, s"Simplify `${CodeGenerator.generate(expression)}`.", new EqualsBooleanInspectorFixAction(expression)))
  }
}

case class UsingTypeOfToCompareTypes(expression: AstNode, typeName: String, containerNode: AstNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.USING_TYPE_OF_TO_COMPARE_TYPES_MESSAGE_KIND

  override def message: String = {
    s"Comparing types by using type names is not correct. Use `${CodeGenerator.generate(expression)} is $typeName`  operator to compare types."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace with `is` Expression",
        s"Replace `${CodeGenerator.generate(containerNode)}` with `is` Expression.",
        new TypeOfQuickFixAction(expression, typeName, containerNode)))
  }
}

case class UsingIfNotNullInsteadOfDefault(expression: AstNode, defaultValue: AstNode, ifExpression: AstNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.USING_IF_NOT_NULL_INSTEAD_OF_DEFAULT_MESSAGE_KIND

  override def message: String = {
    s"Using if(expr != null) when it can be replaced with `default` expression."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace with `default` Expression",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(ifExpression), 10)}` with `default` Expression.",
        new IfNotNullQuickFixAction(expression, defaultValue, ifExpression)))
  }
}

case class UsingIfNullInsteadOfDefault(expression: AstNode, defaultValue: AstNode, ifExpression: AstNode) extends Message with QuickFixAwareMessage {
  override def kind: String = MessageKind.USING_IF_NULL_INSTEAD_OF_DEFAULT_MESSAGE_KIND

  override def message: String = {
    s"Using if(expr == null) when it can be replaced with `default` expression."
  }

  override def category: MessageCategory = InspectorPhaseCategory

  override def quickFixes(): Array[QuickFix] = {
    Array(
      QuickFix(
        "Replace with `default` Expression",
        s"Replace `${StringHelper.abbreviate(CodeGenerator.generate(ifExpression), DEFAULT_CODE_LENGTH)}` with `default` Expression.",
        new IfNotNullQuickFixAction(expression, defaultValue, ifExpression)))
  }
}

case class InvalidEqualComparison(leftType: WeaveType, rightType: WeaveType, wtrc: WeaveTypeResolutionContext) extends Message {
  override def kind: String = MessageKind.INVALID_EQUAL_COMPARISON_MESSAGE_KIND

  override def message: String = {
    val topMessage = s"Comparing ${typeToString(leftType)} with ${typeToString(rightType)} always returns false."
    if (TypeCoercer.coerce(leftType, rightType, wtrc).isDefined || TypeCoercer.coerce(rightType, leftType, wtrc).isDefined) {
      topMessage + s"\nTIP: Try using `~=` for comparing types that can be coerced."
    } else {
      topMessage
    }
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class DeniedFunctionUsage(function: String) extends Message {
  override def kind: String = MessageKind.DENIED_FUNCTION_USAGE_MESSAGE_KIND

  override def message: String = s"Function `$function` is not allowed."

  override def category: MessageCategory = ScopePhaseCategory
}

case class AccessViolation(name: NameIdentifier, currentLocation: String, allowedPatterns: Seq[String]) extends Message {
  override def kind: String = MessageKind.ACCESS_VIOLATION_MESSAGE_KIND

  override def message: String = s"`${name.name}` can not be access from `$currentLocation` as it is constrained to: ${allowedPatterns.map(n => s"`$n`").mkString(" or ")} subdomain."

  override def category: MessageCategory = ScopePhaseCategory
}

case class ExperimentalFeature(name: NameIdentifier) extends Message {
  override def kind: String = MessageKind.EXPERIMENTAL_FEATURE_MESSAGE_KIND

  override def message: String = s"`${name.name}` is experimental this means that it may change or even be removed in the future. Use it at your own risk."

  override def category: MessageCategory = ScopePhaseCategory
}

case class DeprecatedFeature(name: NameIdentifier, since: String, maybeReplacement: Option[String]) extends Message {
  override def kind: String = MessageKind.DEPRECATED_FEATURE_MESSAGE_KIND

  override def message: String = {
    var suffix = ""
    if (maybeReplacement.isDefined) {
      val replacement = maybeReplacement.get
      if (replacement.trim.nonEmpty) {
        suffix = s" and has been replaced with `$replacement`. We strongly advice to use new functionality."
      }
    }
    s"`${name.name}` has been deprecated since `$since`$suffix"
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class IncompatibleRuntimeVersion(currentVersion: String, requiredVersion: String, identifier: NameIdentifier) extends Message {
  override def kind: String = MessageKind.INCOMPATIBLE_RUNTIME_VERSION_MESSAGE_KIND

  override def message: String = s"`${identifier.name}` requires runtime version `$requiredVersion`, but your version level is `$currentVersion`."

  override def category: MessageCategory = ScopePhaseCategory
}

case class LanguageFeatureNotAvailable(currentVersion: String, requiredVersion: String, featureName: String) extends Message {
  override def kind: String = MessageKind.LANGUAGE_FEATURE_NOT_AVAILABLE_MESSAGE_KIND

  override def message: String = s"`$featureName` was introduced on version `$requiredVersion`, but your version level is `$currentVersion`.`"

  override def category: MessageCategory = ScopePhaseCategory
}

case class NonStreamableParameter() extends Message {
  override def kind: String = MessageKind.NOT_STREAMABLE_PARAMETER_MESSAGE_KIND

  override def message: String = "Invoking a function whose parameter is not StreamCapable."

  override def category: MessageCategory = ScopePhaseCategory
}

case class NativeFunctionParameterNotAnnotated() extends Message {
  override def kind: String = MessageKind.NATIVE_FUNCTION_PARAMETER_NOT_ANNOTATED_MESSAGE_KIND

  override def message: String = "Native function parameter is not annotated as StreamCapable"

  override def category: MessageCategory = ScopePhaseCategory
}

case class VariableReferencedMoreThanOnce(variable: NameIdentifier, locations: Seq[WeaveLocation]) extends Message {
  override def kind: String = MessageKind.VARIABLE_REFERENCED_MORE_THAN_ONCE_MESSAGE_KIND

  override def message: String = s"Variable ${variable.name} is referenced more than once. Locations: \n---------------------------\n${locations.map(_.locationString).mkString("\n---------------------------\n")}"

  override def category: MessageCategory = ScopePhaseCategory
}

case class MaxTypeGraphExecution(executionMode: String, maxAmount: Int) extends Message {
  override def kind: String = MessageKind.MAX_TYPE_GRAPH_EXECUTION_MESSAGE_KIND

  override def message: String = s"Type $executionMode graph execution was not able to reach a stable point after: `$maxAmount` attempts"

  override def category: MessageCategory = TypePhaseCategory
}

case class VariableReferencedInOtherScope(variable: NameIdentifier, locations: Seq[WeaveLocation]) extends Message {
  override def kind: String = MessageKind.VARIABLE_REFERENCED_IN_OTHER_SCOPE_MESSAGE_KIND

  override def message: String = s"Variable ${variable.name} is referenced in a different scope from where it was defined. Locations: \n---------------------------\n${locations.map(_.locationString).mkString("\n---------------------------\n")}"

  override def category: MessageCategory = ScopePhaseCategory
}

case class VariableModuleOpenAccess(variable: NameIdentifier) extends Message {
  override def kind: String = MessageKind.VARIABLE_MODULE_OPEN_ACCESS_MESSAGE_KIND

  override def message: String = s"Module variable ${variable.name} is publicly accessible so there is no way to verify its access."

  override def category: MessageCategory = ScopePhaseCategory
}

case class NegativeIndexAccess(index: String) extends Message {
  override def kind: String = MessageKind.NEGATIVE_INDEX_ACCESS_MESSAGE_KIND

  override def message: String = s"Negative index '$index' access."

  override def category: MessageCategory = ScopePhaseCategory
}

case class UsingUpsert() extends Message {
  override def kind: String = MessageKind.USING_UPSERT

  override def message: String = s"Using upsert expression."

  override def category: MessageCategory = ScopePhaseCategory
}

case class RevertSelection() extends Message {
  override def kind: String = MessageKind.REVERT_SELECTION_MESSAGE_KIND

  override def message: String = s"Range selector is inverted (from bigger than to) so it breaks streaming."

  override def category: MessageCategory = ScopePhaseCategory
}

case class NotStreamableOperator(opId: String) extends Message {
  override def kind: String = MessageKind.NOT_STREAMABLE_OPERATOR_MESSAGE_KIND

  override def message: String = s"Operator `$opId` is not stream capable."

  override def category: MessageCategory = ScopePhaseCategory
}

case class DynamicAccess() extends Message {
  override def kind: String = MessageKind.DYNAMIC_ACCESS_MESSAGE_KIND

  override def message: String = s"Dynamic access."

  override def category: MessageCategory = ScopePhaseCategory
}

case class ComplexType() extends Message {
  override def kind: String = MessageKind.COMPLEX_TYPE_MESSAGE_KIND

  override def message: String = s"Complex type operation requires value to be materialized."

  override def category: MessageCategory = ScopePhaseCategory
}

case class ExpressionPatternForceMaterialize() extends Message {
  override def kind: String = MessageKind.EXPRESSION_PATTERN_FORCE_MATERIALIZE_MESSAGE_KIND

  override def message: String = s"Expression pattern node requires value to be materialized."

  override def category: MessageCategory = ScopePhaseCategory
}

object ParsingPhaseCategory extends MessageCategory {
  override def name: String = "Syntax"
}

object InspectorPhaseCategory extends MessageCategory {
  override def name: String = "Inspector"
}

object CanonicalPhaseCategory extends MessageCategory {
  override def name: String = "Canonical"
}

object ScopePhaseCategory extends MessageCategory {
  override def name: String = "Scope"
}

object TypePhaseCategory extends MessageCategory {
  override def name: String = "Type"
}

object WeaveDocParsingPhaseCategory extends MessageCategory {
  override def name: String = "WeaveDoc"
}

case class InvalidTypeParameterCall(typeName: String, passedAmount: Int, expectedAmount: Int) extends Message {
  override def kind: String = MessageKind.INVALID_TYPE_PARAMETER_CALL_MESSAGE_KIND

  override def message: String = {
    if (passedAmount > expectedAmount) {
      s"Too many type parameters specified for `$typeName` expecting $expectedAmount but was $passedAmount."
    } else {
      s"Not enough type parameters specified for `$typeName` expecting $expectedAmount but was $passedAmount."
    }
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class ParameterIsNotStreamCapable(nameIdentifier: NameIdentifier, reasons: Seq[(WeaveLocation, Message)]) extends Message {
  override def kind: String = MessageKind.PARAMETER_IS_NOT_STREAM_CAPABLE_MESSAGE_KIND

  override def message: String = {
    val reasonsText =
      if (reasons.nonEmpty)
        s"\nReasons:\n - ${reasons.map(m => m._2.message + " at " + m._1.locationString).mkString("\n - ")}"
      else
        ""
    s"Parameter `${nameIdentifier.name}` is not stream capable.$reasonsText"
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class FunctionIsNotTailRecCapable(nameIdentifier: NameIdentifier) extends Message {
  override def kind: String = MessageKind.FUNCTION_IS_NOT_TAIL_REC_CAPABLE_MESSAGE_KIND

  override def message: String = {
    s"Function `${nameIdentifier.name}` is not tail recursive."
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class InvalidAnnotationTarget(validTarget: Array[String]) extends Message {
  override def kind: String = MessageKind.INVALID_ANNOTATION_TARGET_MESSAGE_KIND

  override def message: String = {
    s"Annotation is only valid on: ${validTarget.map("`" + _ + "`").mkString(", ")} targets."
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class CloseDoesNotAllowExtraProperties(propertyPath: SelectionPath, additionalProperties: KeyValuePairType, expectedType: WeaveType, actualType: WeaveType) extends TypeMessage {
  override def kind: String = MessageKind.CLOSE_DOES_NOT_ALLOW_EXTRA_PROPERTIES_MESSAGE_KIND

  override def errorMessage: String = {
    if (propertyPath.isEmpty) {
      s"Property: `${additionalProperties.toString(prettyPrint = true, namesOnly = true)}` was not defined in: ${typeToString(expectedType)}. No additional properties is allowed."
    } else {
      s"Property: `$propertyPath` was not defined in: ${typeToString(expectedType)}. No additional properties are allowed."
    }
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class CloseDoesNotAllowOpen(expectedType: WeaveType, actualType: WeaveType) extends TypeMessage {
  override def kind: String = MessageKind.CLOSE_DOES_NOT_ALLOW_OPEN_MESSAGE_KIND

  override def errorMessage: String = {
    s"Closed object (`${typeToString(expectedType)}`) does not allow assignment of open object (`${typeToString(actualType)}`)"
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class MultipleValidFunctions(args: Seq[WeaveType], options: Seq[FunctionType]) extends Message {
  override def kind: String = MessageKind.MULTIPLE_VALID_FUNCTIONS_MESSAGE_KIND

  override def message: String = {
    s"Multiple valid coercion functions: \n${options.mkString("\n\t-")}\nArguments: ${args.mkString(", ")}.\nHINT: To avoid this warning please coerce the argument to match the one wanted"
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class ArrayFunctionInjectionNotPossible(functionName: String) extends Message {
  override def kind: String = MessageKind.ARRAY_FUNCTION_INJECTION_NOT_POSSIBLE_MESSAGE_KIND

  override def message: String = {
    s"Unable to inject function as there is an overload of the function `$functionName` that its second argument is Array but not Array<Function>."
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class FunctionInjectionNotPossible(functionName: String) extends Message {
  override def kind: String = MessageKind.FUNCTION_INJECTION_NOT_POSSIBLE_MESSAGE_KIND

  override def message: String = {
    s"Unable to inject function as there is an overload of the function `$functionName` that is not of type Function."
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class MissingRequiredProperty(propertyPath: SelectionPath, expectedType: WeaveType, actualType: WeaveType) extends TypeMessage {
  override def kind: String = MessageKind.MISSING_REQUIRED_PROPERTY_MESSAGE_KIND

  override def errorMessage: String = s"Missing required property: `$propertyPath` required by ${typeToString(expectedType)}."

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidDynamicReturnMessage(reason: Message, messageLocation: WeaveLocation) extends Message {
  override def kind: String = MessageKind.INVALID_DYNAMIC_RETURN_MESSAGE_KIND

  override def message: String = {
    messageLocation.locationString + "\n\t- " + reason.message
  }

  override def category: MessageCategory = TypePhaseCategory

}

case class RepeatedFieldNotSupported(propertyPath: SelectionPath, expectedType: WeaveType, actualType: WeaveType) extends TypeMessage {
  override def kind: String = MessageKind.REPEATED_FIELD_NOT_SUPPORTED_MESSAGE_KIND

  override def errorMessage: String = s"Repeated field: `$propertyPath` is not supported by: ${typeToString(expectedType)}."

  override def category: MessageCategory = TypePhaseCategory
}

case class PropertyNotDefined(name: String, options: Seq[String], weaveType: WeaveType) extends Message {
  override def kind: String = MessageKind.PROPERTY_NOT_DEFINED_MESSAGE_KIND

  override def message: String = {
    if (options.isEmpty) {
      s"Property: `$name` was not found in: ${typeToString(weaveType)}."
    } else {
      val firstOptions = options.map(option => (StringHelper.levenshtein(name, option), option)).sortBy(_._1).take(5).map(_._2)
      s"Property: `$name` was not found in: ${typeToString(weaveType)}. Similar options are: \n\t- ${firstOptions.mkString("\n\t- ")}"
    }
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidWeaveVersion(expectedVersion: String, actualVersion: String) extends Message {
  override def kind: String = MessageKind.INVALID_WEAVE_VERSION_MESSAGE_KIND

  override def message: String = s"Invalid weave version `$actualVersion`. Supported version is `$expectedVersion``"

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class InvalidDirectiveInDoBlock(expectedDirectives: Seq[String], actualDirective: String) extends Message {
  override def kind: String = MessageKind.INVALID_DIRECTIVE_IN_DO_BLOCK_MESSAGE_KIND

  override def message: String = {
    s"Invalid `$actualDirective` directive in do block. Expected directives are: `${expectedDirectives.mkString(", ")}``"
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class ForwardReference(nameIdentifier: NameIdentifier) extends Message {
  override def kind: String = MessageKind.FORWARD_REFERENCE_MESSAGE_KIND

  override def message: String = s"Variable: `${nameIdentifier.name}` is not yet initialized."

  override def category: MessageCategory = ScopePhaseCategory
}

case class CyclicWeaveImport(nameIdentifier: NameIdentifier, path: Seq[NameIdentifier]) extends Message {
  override def kind: String = MessageKind.CYCLIC_WEAVE_IMPORT_MESSAGE_KIND

  override def message: String = s"ImportError: Cyclic import `${nameIdentifier.name}`.\nPath is:\n- ${path.map(_.name).mkString(" is imported by: \n- ")}"

  override def category: MessageCategory = ScopePhaseCategory
}

case class SelfImportingModule(nameIdentifier: NameIdentifier) extends Message {
  override def kind: String = MessageKind.SELF_IMPORT_MODULE_MESSAGE_KIND

  override def message: String = s"ImportError: Module `${nameIdentifier.name}` imports itself"

  override def category: MessageCategory = ScopePhaseCategory
}

case class InvalidArgumentName(name: NameIdentifier, options: Seq[String]) extends Message {
  override def kind: String = MessageKind.INVALID_ARGUMENT_NAME_MESSAGE_KIND

  override def message: String = {
    val firstOptions = options
      .map(option => (StringHelper.levenshtein(name.name, option), option))
      .sortBy(_._1)
      .take(5)
      .map(_._2)
    s"Invalid argument name: `${name.name}` similar options: \n- ${firstOptions.mkString("\n- ")}"
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidAmountOfAnnotationArguments(expected: Int, actual: Int) extends Message {
  override def kind: String = MessageKind.INVALID_AMOUNT_OF_ANNOTATION_ARGUMENTS_MESSAGE_KIND

  override def message: String = {
    s"Invalid amount of arguments expecting `$expected` but got `$actual`."
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidAnnotationReference(nameIdentifier: NameIdentifier) extends Message {
  override def kind: String = MessageKind.INVALID_ANNOTATION_REFERENCE_MESSAGE_KIND

  override def message: String = s"Reference `${nameIdentifier.name}` is not of type annotation."

  override def category: MessageCategory = TypePhaseCategory
}

case class TypeMismatch(expectedType: WeaveType, actualType: WeaveType, path: SelectionPath = SelectionPath()) extends TypeMessage {
  override def kind: String = MessageKind.TYPE_MISMATCH_MESSAGE_KIND

  def isNullable(weaveType: WeaveType): Boolean = {
    weaveType match {
      case UnionType(of) =>
        of.exists(TypeHelper.isNullType)
      case wt if TypeHelper.isNullType(wt) => true
      case _                               => false
    }
  }

  override def errorMessage: String = {
    val message =
      if (!path.isEmpty) {
        s"Expecting Type: ${typeToString(expectedType)}, but got: ${typeToString(actualType)} at `${path.toString}`."
      } else {
        s"Expecting Type: ${typeToString(expectedType)}, but got: ${typeToString(actualType)}."
      }
    message
  }

  def isNullMismatch: Boolean = {
    isNullable(actualType) && !isNullable(expectedType)
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class UnfulfilledConstraint(reason: TypeMessage) extends TypeMessage {
  override def kind: String = MessageKind.UNFULFILLED_CONSTRAINT_MESSAGE_KIND

  override def expectedType: WeaveType = reason.expectedType

  override def actualType: WeaveType = reason.actualType

  override def errorMessage: String = {
    s"${typeToString(actualType)} does not satisfy ${typeToString(expectedType)} constraint. " ++ indent(s"\n- ${reason.message}")
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class NestedTypeMessage(typeParam: TypeParameter, actualType: WeaveType) extends Message {
  override def kind: String = MessageKind.NESTED_TYPE_MESSAGE_KIND

  override def message: String = s"Type: ${typeParam.name}. Is nested inside: ${actualType.toString}"

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidTypeRef(ref: TypeReferenceNode, tn: AstNode) extends Message {
  override def kind: String = MessageKind.INVALID_TYPE_REF_MESSAGE_KIND

  override def message: String = {
    s"Type Reference `${ref.variable.name}` should reference a Type Directive but got: `${tn.getClass.getSimpleName}`."
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidReferenceMessage(reference: AstNode) extends Message {
  override def kind: String = MessageKind.INVALID_REFERENCE_MESSAGE_KIND

  override def message: String = {
    val name = reference match {
      case VariableReferenceNode(NameIdentifier(varName, _), _) => varName
      case TypeReferenceNode(NameIdentifier(varName, _), _, _, _, _) => varName
      case NamespaceNode(NameIdentifier(varName, _)) => varName
      case NameIdentifier(varName, _) => varName
      case _ => reference.toString
    }
    s"Unable to resolve reference of: `$name`."
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class DuplicatedNamespaceMessage(prefix: String) extends Message {
  override def kind: String = MessageKind.DUPLICATED_NAMESPACE_MESSAGE_KIND

  override def message: String = s"Duplicated namespace `$prefix` definition."

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class DuplicatedVariableMessage(variableName: String) extends Message {
  override def kind: String = MessageKind.DUPLICATED_VARIABLE_MESSAGE_KIND

  override def message: String = {
    s"Duplicated variable: `$variableName`."
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class DuplicatedParameterMessage(paramName: String) extends Message {
  override def kind: String = MessageKind.DUPLICATED_PARAMETER_MESSAGE_KIND

  override def message: String = {
    s"Duplicated parameter `$paramName` name."
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class VariableNameClashedWithImplicitInputMessage(name: String) extends Message {
  override def kind: String = MessageKind.VARIABLE_NAME_CLASHED_WITH_IMPLICIT_INPUT_MESSAGE_KIND

  override def message: String = {
    s"Variable`$name` is being used as a variable and implicit input. Implicit input will be ignored in favor of variable `$name` definition."
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class FunctionNameClashedWithVariableMessage(functionName: String) extends Message {
  override def kind: String = MessageKind.FUNCTION_NAME_CLASHED_WITH_VARIABLE_MESSAGE_KIND

  override def message: String = {
    s"Function `$functionName` is already used by a variable."
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class TooManyArgumentMessage(expectedAmount: Seq[WeaveType], actualAmount: Seq[WeaveType], expectedType: WeaveType, actualType: WeaveType) extends TypeMessage {
  override def kind: String = MessageKind.TOO_MANY_ARGUMENT_MESSAGE_KIND

  override def errorMessage: String = {
    s"Expects `${expectedAmount.size}` argument but got `${actualAmount.size}`. Expecting: ${if (expectedAmount.nonEmpty) s"(${expectedAmount.map(typeToString(_, "", "")).mkString(", ")})" else ""}, " +
      s"but got: (${actualAmount.map(typeToString(_, "", "")).mkString(", ")})."
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class NotEnoughArgumentMessage(expectedAmount: Seq[WeaveType], actualAmount: Seq[WeaveType], expectedType: WeaveType, actualType: WeaveType) extends TypeMessage {
  override def kind: String = MessageKind.NOT_ENOUGH_ARGUMENT_MESSAGE_KIND

  override def errorMessage: String = {
    s"Expects `${expectedAmount.size}` argument but got `${actualAmount.size}`. Expecting: ${if (expectedAmount.nonEmpty) s"(${expectedAmount.map(typeToString(_, "", "")).mkString(", ")})" else ""}, " +
      s"but got: (${actualAmount.map(typeToString(_, "", "")).mkString(", ")})."
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidAmountTypeParametersMessage(expectedAmount: Int, actualAmount: Int, expectedType: WeaveType, actualType: WeaveType) extends TypeMessage {
  override def kind: String = MessageKind.INVALID_AMOUNT_TYPE_PARAMETERS_MESSAGE_KIND

  override def errorMessage: String = {
    s"Function expects $expectedAmount type parameters but got $actualAmount"
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class UnableToResolveModule(nameIdentifier: NameIdentifier) extends Message {
  override def kind: String = MessageKind.UNABLE_TO_RESOLVE_MODULE_MESSAGE_KIND

  override def message: String = {
    s"Unable to resolve module with identifier ${nameIdentifier.name}."
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class InvalidMethodTypesMessage(opName: String, actualTypes: Seq[WeaveType], problems: Seq[(FunctionType, Seq[(WeaveLocation, Message)])]) extends TypeMessage {
  override def kind: String = MessageKind.INVALID_METHOD_TYPES_MESSAGE_KIND

  override def expectedType: WeaveType = NullType()

  override def actualType: WeaveType = NullType()

  override def errorMessage: String = {
    if (problems.size == 1) {
      val theFunction: (FunctionType, Seq[(WeaveLocation, Message)]) = problems.head
      val errorMessages: Seq[(WeaveLocation, Message)] = theFunction._2
      val reason = if (errorMessages.size == 1) {
        errorMessages.head._2.message
      } else {
        "- " + errorMessages.map(_._2.message).mkString("\n- ")
      }
      reason

    } else {
      val callArgumentsString = s"(${actualTypes.map(typeToString).mkString(", ")})"
      val problemsSortedByUnmatched = problems.sortBy(_._2.size)
      val problemsToShow = if (problemsSortedByUnmatched.size > 2) {
        problemsSortedByUnmatched.slice(0, 2)
      } else {
        problemsSortedByUnmatched
      }
      val footer = if (problemsSortedByUnmatched.size > problemsToShow.size) {
        s"\n- ${problemsSortedByUnmatched.size - problemsToShow.size} more options ..."
      } else {
        ""
      }
      s"Unable to call: `$opName` with arguments: $callArgumentsString.\n " + (if (problems.nonEmpty) "Reasons:\n" else "") +
        indent({
          problemsToShow //Sort based on the one with less issues
            .map(ftProblemsPair => {
              val functionType: FunctionType = ftProblemsPair._1
              val functionDef: String = opName + typeToString(functionType, "", "")
              val rootCauses: String = "- " + ftProblemsPair._2.map(_._2.message).mkString("\n- ")
              rootCauses +
                indent("\n|- From: " + functionDef) +
                indent(indent(ftProblemsPair._2.head._1.locationString))
            })
            .mkString("\n")
        } + footer)

    }
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class TypeCoercedMessage(expectedType: WeaveType, actualType: WeaveType) extends Message {
  override def kind: String = MessageKind.TYPE_COERCED_MESSAGE_KIND

  override def message: String = {
    s"Auto-Coercing type from: ${typeToString(actualType)} to: ${typeToString(expectedType)}."
  }

  override def category: MessageCategory = ScopePhaseCategory
}

case class NoCoercionAvailableMessage(actualType: WeaveType, expectedType: WeaveType) extends Message {
  override def kind: String = MessageKind.NO_COERCION_AVAILABLE_MESSAGE_KIND

  override def message: String = s"No coercion available from: ${typeToString(actualType)} to: ${typeToString(expectedType)}."

  override def category: MessageCategory = TypePhaseCategory
}

case class InvalidSyntaxMessage(message: String) extends Message {
  override def kind: String = MessageKind.INVALID_SYNTAX_MESSAGE_KIND

  override def category: MessageCategory = ParsingPhaseCategory
}

case class InvalidWeaveDocSyntaxMessage(message: String) extends Message {
  override def kind: String = MessageKind.INVALID_WEAVE_DOC_SYNTAX_MESSAGE_KIND

  override def category: MessageCategory = WeaveDocParsingPhaseCategory
}

case class FunctionInvalidDefaultValueMessage() extends Message {
  override def kind: String = MessageKind.FUNCTION_INVALID_DEFAULT_VALUE_MESSAGE_KIND

  override def message: String = "Functions can only have head or tail default function values."

  override def category: MessageCategory = ScopePhaseCategory
}

case class OnlyOneDefaultPattern() extends Message {
  override def kind: String = MessageKind.ONLY_ONE_DEFAULT_PATTERN_MESSAGE_KIND

  override def message: String = "Default can be only one."

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class DuplicatedSyntaxDirective() extends Message {
  override def kind: String = MessageKind.DUPLICATED_SYNTAX_DIRECTIVE_MESSAGE_KIND

  override def message: String = {
    s"Duplicated syntax directive"
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class InvalidSyntaxDirectivePlacement() extends Message {
  override def kind: String = MessageKind.INVALID_SYNTAX_DIRECTIVE_PLACEMENT_MESSAGE_KIND

  override def message: String = {
    s"Syntax directive must be at the top of the script"
  }

  override def category: MessageCategory = CanonicalPhaseCategory
}

case class TypeSelectorKeyNotFound(selectorName: String, weaveType: WeaveType) extends Message {
  override def kind: String = MessageKind.TYPE_SELECTOR_KEY_NOT_FOUND

  override def message: String = {
    s"Key with name `${selectorName}` was not found on type: ${typeToString(weaveType)}."
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class InterpolationInTypeSelector() extends Message {
  override def kind: String = MessageKind.TYPE_SELECTOR_KEY_STRING_INTERPOLATION

  override def message: String = {
    s"String interpolation is not supported for type selectors."
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class IncorrectTypeForTypeSelection(weaveType: WeaveType) extends Message {
  override def kind: String = MessageKind.TYPE_SELECTION_NOT_ALLOWED_ON_TYPE

  override def message: String = {
    s"Type selection is not allowed on type: ${typeToString(weaveType)}."
  }

  override def category: MessageCategory = TypePhaseCategory
}

case class ShouldTypeDynamicGenericFunction(functionNode: FunctionType) extends Message {
  override def kind: String = MessageKind.SHOULD_TYPE_DYNAMIC_GENERIC_FUNCTION

  override def category: MessageCategory = TypePhaseCategory

  private val dynamicFunctionLocationsString: String = {
    val funcs = FunctionTypeHelper.collectDynamicFunctions(functionNode)
    val funcMsgs = funcs.map(ft => ft.location().noErrorMarkerLocationString)
    if (funcMsgs.nonEmpty) s"\n At: ${funcMsgs.mkString("\n At: ")}" else ""
  }

  override def message: String = {
    s"An untyped lambda is being returned. This may generate errors when using this function, please consider either inserting the return type of the function or typing the lambda.$dynamicFunctionLocationsString"
  }
}

case class InvalidKeyType(wt: WeaveType) extends Message {
  override def kind: String = MessageKind.INVALID_KEY_TYPE

  override def category: MessageCategory = TypePhaseCategory

  override def message: String = s"Type ${typeToString(wt)} cannot be used as Key"
}

case class InvalidNameType(wt: WeaveType) extends Message {
  override def kind: String = MessageKind.INVALID_NAME_TYPE

  override def category: MessageCategory = TypePhaseCategory

  override def message: String = s"Type ${typeToString(wt)} cannot be used as Name"
}

case class MetadataAnnotationShouldHaveValue(nameIdentifier: Option[NameIdentifier], expectedSize: Option[Int]) extends Message {
  override def kind: String = MessageKind.METADATA_PARAMETER_SHOULD_BE_VALUE_MESSAGE_KIND

  override def message: String = {
    s"Annotation parameters should be only one${expectedSize.map(actualSize => " but found " + actualSize).getOrElse("")} with name `value`${nameIdentifier.map(id => " but a parameter with name `" + id.name + "` was found").getOrElse("")} for metadata annotations."
  }

  override def category: MessageCategory = ParsingPhaseCategory
}

case class IgnoredMetadataAnnotation(annotationNode: AnnotationNode) extends Message {
  override def kind: String = MessageKind.METADATA_ANNOTATION_AND_INJECTOR_COLLISION_MESSAGE_KIND

  override def message: String = {
    s"Metadata annotation ${annotationNode.name} will be ignored since an injector operation () was found."
  }

  override def category: MessageCategory = ParsingPhaseCategory
}

case class DuplicatedMetadataAnnotation(annotationNode: AnnotationNode) extends Message {
  override def kind: String = MessageKind.METADATA_ANNOTATION_DUPLICATED_KEY

  override def message: String = {
    s"Metadata annotation ${annotationNode.name} will be ignored since it is duplicated in the same node."
  }

  override def category: MessageCategory = ParsingPhaseCategory
}

case class NoMetadataValueSpecified(annotationNode: AnnotationNode) extends Message {
  override def kind: String = MessageKind.METADATA_ANNOTATION_MISSING_VALUE

  override def message: String = {
    s"Metadata annotation ${annotationNode.name} will be ignored since no value was specified."
  }

  override def category: MessageCategory = ParsingPhaseCategory
}

case class MissingRequiredAnnotationArgument(argument: String) extends Message {

  override def kind: String = MessageKind.MISSING_REQUIRED_ANNOTATION_ARGUMENT

  override def message: String = {
    s"Missing required annotation argument `$argument`."
  }

  override def category: MessageCategory = ScopePhaseCategory
}
