package org.mule.weave.v2.utils

import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.locks.ReentrantLock
import scala.collection.JavaConverters._
import scala.util.Random

class CacheBuilder[KeyType, ValueType](resolver: (KeyType) => ValueType) {

  private var maxElements: Int = 10000
  private var cleanupRation: Double = 0.5
  private var disposer: (ValueType) => Unit =
    //Default disposer function
    (valueType) => {
      valueType match {
        case d: Disposable => d.dispose()
        case _             =>
      }
    }

  def disposer(disposer: (ValueType) => Unit): CacheBuilder[KeyType, ValueType] = {
    this.disposer = disposer
    this
  }

  def maximumSize(maxElements: Int): CacheBuilder[KeyType, ValueType] = {
    this.maxElements = maxElements
    this
  }

  def cleanupRatio(cleanupRation: Double): CacheBuilder[KeyType, ValueType] = {
    if (cleanupRation > 1 || cleanupRation < 0) {
      throw new IllegalArgumentException("Cleanup ratio needs to be between 0 and 1")
    }
    this.cleanupRation = cleanupRation
    this
  }

  def build(): BasicCache[KeyType, ValueType] = {
    new JVMSimpleCache[KeyType, ValueType](resolver, maxElements, cleanupRation, disposer)
  }
}

object CacheBuilder {
  def apply[KeyType, ValueType](resolver: (KeyType) => ValueType): CacheBuilder[KeyType, ValueType] = new CacheBuilder[KeyType, ValueType](resolver)

  def apply[KeyType, ValueType](): CacheBuilder[KeyType, ValueType] = new CacheBuilder[KeyType, ValueType](null)
}

/**
  * A very simple and stupid cache implementation that uses ConcurrentHashMap and size limit only
  */
class JVMSimpleCache[KeyType, ValueType](resolver: (KeyType) => ValueType, maxElements: Int, cleanupRation: Double, disposer: (ValueType) => Unit) extends BasicCache[KeyType, ValueType] {

  private val elements = new ConcurrentHashMap[KeyType, ValueType]()

  val lock = new ReentrantLock

  def remove(path: KeyType): Option[ValueType] = {
    val result = Option(elements.remove(path))
    result.foreach(disposer(_))
    result
  }

  def get(key: KeyType): ValueType = {
    get(key, resolver)
  }

  def clear(): Unit = {
    elements.clear()
  }

  def getIfPresent(key: KeyType): Option[ValueType] = {
    Option(elements.get(key))
  }

  def get(key: KeyType, resolver: (KeyType) => ValueType): ValueType = {
    if (elements.size > maxElements) {
      lock.lock()
      try {
        if (elements.size > maxElements) {
          //Shuffle to avoid removing same elements
          val types = Random.shuffle(elements.keySet.asScala).iterator
          val d = maxElements * cleanupRation
          var i = 0
          while (i < d && types.hasNext) {
            val valueType = Option(elements.remove(types.next()))
            valueType.foreach(disposer(_))
            i = i + 1
          }
        }
      } finally {
        lock.unlock()
      }
    }
    var valueType = elements.get(key)
    if (valueType == null) {
      valueType = elements.computeIfAbsent(key, (key) => resolver(key))
    }
    valueType
  }

  override def values(): Iterator[ValueType] = {
    elements.values().iterator().asScala
  }

  override def keys(): Iterator[KeyType] = {
    elements.keys().asScala
  }
}
