package org.mule.weave.v2.utils

import java.util.concurrent.ConcurrentHashMap

/**
  * An implementation of an IdentityHashMap that uses ConcurrentHashMap. The difference of this with SynchronizedHashMap
  * is that this one doesn't support re-entrant getOrElseUpdate.
  */
final class ConcurrentIdentityHashMap[K <: AnyRef, V]() {
  private val internalMap = new ConcurrentHashMap[IdentityKey, V]()

  def get(key: K): Option[V] = {
    Option(internalMap.get(new IdentityKey(key)))
  }

  def put(key: K, value: V): Option[V] = {
    Option(internalMap.put(new IdentityKey(key), value))
  }

  def getOrElseUpdate(key: K, value: => V): V = {
    internalMap.computeIfAbsent(new IdentityKey(key), (_) => value)
  }

  def find(predicate: ((K, V)) => Boolean): Option[(K, V)] = {
    val maybeValue = internalMap.entrySet().stream().filter((entry) => predicate((entry.getKey.key, entry.getValue))).findFirst()
    if (maybeValue.isPresent) {
      val entry = maybeValue.get()
      Some((entry.getKey.key, entry.getValue))
    } else {
      None
    }
  }

  def contains(key: K): Boolean = {
    internalMap.contains(new IdentityKey(key))
  }

  private class IdentityKey(val key: K) {
    override def hashCode(): Int = System.identityHashCode(key)

    override def equals(obj: Any): Boolean = obj.asInstanceOf[IdentityKey].key eq key
  }
}

object ConcurrentIdentityHashMap {
  def apply[A <: AnyRef, B]() = new ConcurrentIdentityHashMap[A, B]()
}
