package org.mule.weave.v2.ts

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.api.tooling.location.ResourceIdentifier
import org.mule.weave.v2.api.tooling.location.{ Location => ApiLocation }
import org.mule.weave.v2.api.tooling.ts
import org.mule.weave.v2.api.tooling.ts.DWMetadata
import org.mule.weave.v2.api.tooling.ts.DWMetadataConstraint
import org.mule.weave.v2.api.tooling.ts.DWMetadataValue
import org.mule.weave.v2.api.tooling.ts.DWType
import org.mule.weave.v2.api.tooling.ts.DWTypeVisitor
import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.DateTimeNode
import org.mule.weave.v2.parser.ast.structure.LocalDateNode
import org.mule.weave.v2.parser.ast.structure.LocalDateTimeNode
import org.mule.weave.v2.parser.ast.structure.LocalTimeNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.NullNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.PeriodNode
import org.mule.weave.v2.parser.ast.structure.RegexNode
import org.mule.weave.v2.parser.ast.structure.StringInterpolationNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.TimeNode
import org.mule.weave.v2.parser.ast.structure.TimeZoneNode
import org.mule.weave.v2.parser.ast.structure.UriNode
import org.mule.weave.v2.parser.ast.types._
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.LocationCapable
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.ts.resolvers.NameTypeResolver
import org.mule.weave.v2.utils.AnnotationSchemaNode
import org.mule.weave.v2.utils.ExplicitSchemaNode
import org.mule.weave.v2.utils.Optionals.toJavaOptional
import org.mule.weave.v2.utils.SchemaNodeOrigin
import org.mule.weave.v2.utils.StringEscapeHelper
import org.mule.weave.v2.utils.WeaveTypeEmitter
import org.mule.weave.v2.utils.WeaveTypeEmitterConfig
import org.mule.weave.v2.utils.WeaveTypeMetadataEmitter

import java.util.Optional
import scala.collection.mutable

@WeaveApi(Seq("data-weave-agent"))
sealed trait WeaveType extends DWType with LocationCapable {

  protected var _metadataConstraints: Option[mutable.ArrayBuffer[MetadataConstraint]] = None
  protected var _metadata: Option[mutable.ArrayBuffer[Metadata]] = None

  private var _location: WeaveLocation = UnknownLocation

  private var _label: Option[String] = None

  /**
    * Marks this type as optional. This is used for selection if a key value pair is optional
    */
  private var optionalType: Boolean = false

  /**
    * The parent key if any of this value
    */
  var parentKey: Option[KeyType] = None

  def label(): Option[String] = _label.orElse(defaultLabel())

  override def getLabel(): Optional[String] = label().asJava

  def label(label: String): WeaveType = {
    _label = Some(label)
    this
  }

  def label(label: Option[String]): WeaveType = {
    _label = label
    this
  }

  def defaultLabel(): Option[String] = None

  def withMetadataConstraint(metadataConstraint: MetadataConstraint): WeaveType = {
    if (this._metadataConstraints.isEmpty) {
      this._metadataConstraints = Some(new mutable.ArrayBuffer[MetadataConstraint]())
    }
    this._metadataConstraints.get.+=(metadataConstraint)
    this
  }

  def markWithEmptyMetadataConstraint(): Unit = {
    this._metadataConstraints = Some(new mutable.ArrayBuffer[MetadataConstraint]())
  }

  def metadataConstraints(): Seq[MetadataConstraint] = {
    _metadataConstraints.getOrElse(mutable.ArrayBuffer.empty)
  }

  override def getTypeMetadataConstraints: Array[DWMetadataConstraint] = {
    metadataConstraints().toArray
  }

  def copyMetadataConstraintsTo(to: WeaveType): Unit = {
    if (this._metadataConstraints.nonEmpty) {
      this._metadataConstraints.get.foreach(c => to.withMetadataConstraint(c))
    }
  }

  def withLocation(weaveLocation: WeaveLocation): WeaveType = {
    this._location = weaveLocation
    this
  }

  override def location(): WeaveLocation = {
    _location
  }

  def getMetadataConstraint(name: String): Option[MetadataConstraint] = {
    this._metadataConstraints.flatMap(_.find(m => m.name == name))
  }

  override def getTypeMetadataConstraint(name: String): Optional[DWMetadataConstraint] = {
    val maybeMetadataConstraint: Option[DWMetadataConstraint] = getMetadataConstraint(name)
    maybeMetadataConstraint.asJava
  }

  def getDocumentation(): Option[String] = getMetadata(Metadata.DOCUMENTATION_ANNOTATION).map(metadata => {
    metadata.value.asInstanceOf[LiteralMetadataValue].value
  })

  override def getTypeDocumentation(): Optional[String] = getDocumentation().asJava

  def withDocumentation(doc: Option[String], location: WeaveLocation): WeaveType = {
    doc.foreach((doc) => withMetadata(Metadata(Metadata.DOCUMENTATION_ANNOTATION, LiteralMetadataValue(doc, StringType(), location))))
    this
  }

  def withDocumentation(doc: String, location: WeaveLocation): WeaveType = {
    withMetadata(Metadata(Metadata.DOCUMENTATION_ANNOTATION, LiteralMetadataValue(doc, StringType(), location)))
    this
  }

  def withMetadata(metadata: Metadata): WeaveType = {
    if (this._metadata.isEmpty) {
      this._metadata = Some(new mutable.ArrayBuffer[Metadata]())
    }
    this._metadata.get.+=(metadata)
    this
  }

  def markWithEmptyMetadata(): Unit = {
    this._metadata = Some(new mutable.ArrayBuffer[Metadata]())
  }

  def copyMetadataTo(to: WeaveType): Unit = {
    if (this._metadata.nonEmpty) {
      this._metadata.get.foreach(m => to.withMetadata(m))
    }
  }

  def metadata(): Seq[Metadata] = {
    _metadata.getOrElse(mutable.ArrayBuffer.empty)
  }

  override def getTypeMetadata(): Array[DWMetadata] = metadata().toArray

  def getMetadata(name: String): Option[Metadata] = {
    this._metadata.flatMap(_.find(m => m.name == name))
  }

  override def getTypeMetadata(name: String): Optional[DWMetadata] = {
    val metadata: Option[DWMetadata] = getMetadata(name)
    metadata.asJava
  }

  def markOptional(): this.type = {
    optionalType = true
    this
  }

  def withOptional(optional: Boolean): this.type = {
    optionalType = optional
    this
  }

  def isOptional() = optionalType

  /**
    * Prints the type using the weave type system
    *
    * @return
    */
  override def toString: String = {
    val emitter = new WeaveTypeEmitter(WeaveTypeEmitterConfig(prettyPrint = true, maxDepth = Some(10), printTypeBounds = true, skipWeaveTypeMetadata = true))
    emitter.toString(this)
  }

  /**
    * Prints the type
    *
    * @param prettyPrint Use pretty printing adding new lines
    * @param namesOnly   If a type Id is present then we only print that
    * @return The String representation
    */
  def toString(prettyPrint: Boolean, namesOnly: Boolean): String = WeaveTypeEmitter.toString(this, prettyPrint = prettyPrint, nameOnly = namesOnly)

  /**
    * Prints the type
    *
    * @param prettyPrint Use pretty printing adding new lines
    * @param namesOnly   If a type Id is present then we only print that
    * @param useLiterals If it should use literal types or just print the parent type
    * @return The String representation
    */
  def toString(prettyPrint: Boolean, namesOnly: Boolean, useLiterals: Boolean): String = {
    new WeaveTypeEmitter(WeaveTypeEmitterConfig(prettyPrint = prettyPrint, nameOnly = namesOnly, useLiteralType = useLiterals)).toString(this)
  }

  /**
    * Prints this type using a given config
    *
    * @param config
    * @return
    */
  def toString(config: WeaveTypeEmitterConfig) = new WeaveTypeEmitter(config).toString(this)

  /**
    * Shallow copy of the type
    *
    * @return A new instance of this type
    */
  def cloneType(): WeaveType

  /**
    * Base type of literal types.
    *
    * @return The base type if it is a literal type, otherwise the same type.
    */
  def baseType(): WeaveType = this

  override def getBaseType(): DWType = this

  override def getLocation: ApiLocation = {
    this.location()
  }
}

case class MetadataConstraint(name: String, value: Any) extends DWMetadataConstraint {
  override def getName: String = {
    name
  }

  override def getValue: Any = {
    value
  }
}

case class Metadata(name: String, value: MetadataValue, fromAnnotation: SchemaNodeOrigin = ExplicitSchemaNode) extends DWMetadata {

  override def toString: String = {
    val emitter = WeaveTypeMetadataEmitter()
    emitter.toString(this)
  }

  override def getName: String = name

  override def getValue: DWMetadataValue = value
}

trait MetadataValue extends DWMetadataValue with LocationCapable

case class LiteralMetadataValue(value: String, weaveType: WeaveType, location: WeaveLocation) extends MetadataValue with ts.LiteralMetadataValue {
  override def getValue: String = value

  override def getType: DWType = weaveType

  override def getLocation: ApiLocation = {
    location
  }
}

case class ArrayMetadataValue(elements: Seq[MetadataValue], location: WeaveLocation) extends MetadataValue with ts.ArrayMetadataValue {
  override def getElements: Array[DWMetadataValue] = elements.toArray

  override def getLocation: ApiLocation = {
    location
  }
}

case class ObjectMetadataValue(properties: Seq[KeyValuePairMetadataValue], location: WeaveLocation) extends MetadataValue with ts.ObjectMetadataValue {
  override def getProperties: Array[ts.KeyValuePairMetadataValue] = properties.toArray

  override def getLocation: ApiLocation = {
    location
  }
}

case class KeyValuePairMetadataValue(key: String, value: MetadataValue, location: WeaveLocation) extends MetadataValue with ts.KeyValuePairMetadataValue {
  override def getKey: String = key

  override def getValue: DWMetadataValue = value

  override def getLocation: ApiLocation = {
    location
  }
}

case class VariableReferenceMetadataValue(fullQualifiedName: String, location: WeaveLocation) extends MetadataValue with ts.VariableReferenceMetadataValue {
  override def getFullyQualifiedName: String = fullQualifiedName

  override def getLocation: ApiLocation = {
    location
  }
}

case class UnknownMetadataValue(location: WeaveLocation) extends MetadataValue with ts.UnknownMetadataValue {
  override def getLocation: ApiLocation = {
    location
  }
}

object Metadata {
  val DOCUMENTATION_ANNOTATION = "Documentation"
}

object MetadataConstraint {

  val CLASS_PROPERTY_NAME = "class"
  val CDATA_PROPERTY_NAME = "cdata"
  val ITERATOR_PROPERTY_NAME = "iterator"
  val ROUND_MODE_PROPERTY_NAME = "roundMode"
  val FORMAT_PROPERTY_NAME = "format"
  val UNIT_PROPERTY_NAME = "unit"
  val LOCALE_PROPERTY_NAME = "locale"
  val MODE_PROPERTY_NAME = "mode"
  val ENCODING_PROPERTY_NAME = "encoding"
  val BASE_PROPERTY_NAME = "base"
  val MIME_TYPE_PROPERTY_NAME = "mimeType"
  val CONTENT_LENGTH_PROPERTY_NAME = "contentLength"
  val MEDIA_TYPE_PROPERTY_NAME = "mediaType"
  val RAW_PROPERTY_NAME = "raw"

  val NS_URI_PROPERTY_NAME = "nsUri"
  val NS_PREFIX_PROPERTY_NAME = "nsPrefix"

}

case class ObjectType(properties: Seq[KeyValuePairType] = Seq(), close: Boolean = false, ordered: Boolean = false) extends ts.ObjectType with WeaveType {
  override def cloneType(): WeaveType = ObjectType(properties, close, ordered)

  def isOpen: Boolean = !close

  override def baseType(): WeaveType = {
    ObjectType(properties.map(_.baseType()), close, ordered)
  }

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitObjectType(this)

  override def getProperties: Array[ts.KeyValuePairType] = properties.toArray

  override def isClosed: Boolean = close

  override def isOrdered: Boolean = ordered

}

object SystemWeaveTypes {
  val simpleTypeMap: Map[String, WeaveType] =
    Map(
      TypeLiteral.STRING_TYPE_NAME -> StringType(),
      TypeLiteral.BOOLEAN_TYPE_NAME -> BooleanType(),
      TypeLiteral.NUMBER_TYPE_NAME -> NumberType(),
      TypeLiteral.RANGE_TYPE_NAME -> RangeType(),
      TypeLiteral.NAME_SPACE_TYPE_NAME -> NamespaceType(),
      TypeLiteral.URI_TYPE_NAME -> UriType(),
      TypeLiteral.DATETIME_TYPE_NAME -> DateTimeType(),
      TypeLiteral.LOCALDATETIME_TYPE_NAME -> LocalDateTimeType(),
      TypeLiteral.DATE_TYPE_NAME -> LocalDateType(),
      TypeLiteral.LOCALTIME_TYPE_NAME -> LocalTimeType(),
      TypeLiteral.TIME_TYPE_NAME -> TimeType(),
      TypeLiteral.TIMEZONE_TYPE_NAME -> TimeZoneType(),
      TypeLiteral.PERIOD_TYPE_NAME -> PeriodType(),
      TypeLiteral.BINARY_TYPE_NAME -> BinaryType(),
      TypeLiteral.NULL_TYPE_NAME -> NullType(),
      TypeLiteral.ANY_TYPE_NAME -> AnyType(),
      TypeLiteral.REGEX_TYPE_NAME -> RegexType(),
      TypeLiteral.NOTHING_TYPE_NAME -> NothingType(),
      TypeLiteral.ARRAY_TYPE_NAME -> ArrayType(NothingType()),
      TypeLiteral.OBJECT_TYPE_NAME -> ObjectType(Seq()),
      TypeLiteral.TYPE_TYPE_NAME -> TypeType(AnyType()),
      TypeLiteral.KEY_TYPE_NAME -> KeyType(NameType()))

  private lazy val literalTypesByNodes: Map[Class[_], WeaveType] =
    Map(
      classOf[BooleanNode] -> BooleanType(),
      classOf[DateTimeNode] -> DateTimeType(),
      classOf[LocalDateNode] -> LocalDateType(),
      classOf[LocalDateTimeNode] -> LocalDateTimeType(),
      classOf[LocalTimeNode] -> LocalTimeType(),
      classOf[NullNode] -> NullType(),
      classOf[NumberNode] -> NumberType(),
      classOf[PeriodNode] -> PeriodType(),
      classOf[RegexNode] -> RegexType(),
      classOf[StringNode] -> StringType(),
      classOf[TimeNode] -> TimeType(),
      classOf[TimeZoneNode] -> TimeZoneType(),
      classOf[UriNode] -> UriType())

  def getLiteralTypeByNode[T <: AstNode](classType: Class[T]): Option[WeaveType] = {
    literalTypesByNodes.get(classType)
  }
}

object WeaveType {

  def getSimpleType(typeId: String): Option[WeaveType] = SystemWeaveTypes.simpleTypeMap.get(typeId)

  def apply(node: WeaveTypeNode, typeSystemScope: WeaveTypeReferenceResolver): WeaveType = {
    val result = node match {
      case ObjectTypeNode(properties, _, _, close, ordered, _) => {
        ObjectType(properties.map(apply(_, typeSystemScope).asInstanceOf[KeyValuePairType]), close, ordered)
      }
      case NameTypeNode(localName, ns, _, _) => {
        localName match {
          case Some(name) => NameType(Some(QName(name, ns.flatMap(typeSystemScope.resolveNamespace))))
          case None       => NameType()
        }
      }
      case KeyValueTypeNode(key, value, repeated, optional) => {
        KeyValuePairType(WeaveType(key, typeSystemScope), WeaveType(value, typeSystemScope), optional, repeated)
      }
      case KeyTypeNode(nameWithPrefix, attrs, _, _, _) => {
        KeyType(
          apply(nameWithPrefix, typeSystemScope),
          attrs.map({
            case NameValueTypeNode(attrNameWithPrefix, value, optional) => {
              NameValuePairType(apply(attrNameWithPrefix, typeSystemScope), WeaveType(value, typeSystemScope), optional)
            }
          }))
      }
      case NameValueTypeNode(attrNameWithPrefix, value, optional) => {
        NameValuePairType(apply(attrNameWithPrefix, typeSystemScope), WeaveType(value, typeSystemScope), optional)
      }
      case TypeParameterNode(name, base) => TypeParameter(name.name, base.map(WeaveType(_, typeSystemScope)))
      case FunctionTypeNode(args, returnType, _, _, _) =>
        val arguments: Seq[FunctionTypeParameter] = args.zipWithIndex.map((indexedType) => {
          val argName: String = indexedType._1.name.map(_.name).getOrElse("$" * (indexedType._2 + 1))
          val argType: WeaveType = WeaveType(indexedType._1.valueType, typeSystemScope)
          FunctionTypeParameter(argName, argType, indexedType._1.optional)
        })
        FunctionType(Seq(), arguments, WeaveType(returnType, typeSystemScope))
      case UnionTypeNode(elems, _, _, _) => {
        val types = elems.map(WeaveType(_, typeSystemScope))
        UnionType(types)
      }
      case IntersectionTypeNode(elems, _, _, _) => {
        val types = elems.map(WeaveType(_, typeSystemScope))
        IntersectionType(types)
      }
      case NativeTypeNode(typeId, _) => {
        getSimpleType(typeId).getOrElse(NothingType()).cloneType()
      }
      case trsn: TypeSelectorNode => {
        val weaveType = WeaveType(trsn.weaveTypeNode, typeSystemScope)
        val (selectorName, nsPrefix) = trsn.selector match {
          case NameNode(StringNode(name, _), ns, _) => {
            ns match {
              case Some(namespaceNode: NamespaceNode) => (name, Some(namespaceNode))
              case None                               => (name, None)
            }
          }
          case NameNode(StringInterpolationNode(_), _, _) => throw new RuntimeException(s"String interpolation is not supported on Type Selectors.")
        }
        weaveType match {
          case typeReferenceType: ReferenceType => TypeSelectorType(nsPrefix, selectorName, typeReferenceType, new TypeSelectorReference(nsPrefix, selectorName, typeReferenceType, trsn.selector.location(), typeSystemScope))
          case _                                => AnyType()
        }
      }
      case trn: TypeReferenceNode => {
        trn.variable.name match {
          case TypeLiteral.ARRAY_TYPE_NAME =>
            val of =
              if (trn.typeArguments.isDefined) {
                trn.typeArguments.get
                  .map(x => {
                    WeaveType(x, typeSystemScope)
                  })
                  .headOption
              } else {
                Some(AnyType())
              }
            ArrayType(of.getOrElse(AnyType()))
          case TypeLiteral.TYPE_TYPE_NAME =>
            val of =
              if (trn.typeArguments.isDefined) {
                trn.typeArguments.get
                  .map(x => {
                    WeaveType(x, typeSystemScope)
                  })
                  .headOption
              } else {
                Some(AnyType())
              }
            TypeType(of.getOrElse(AnyType()))
          case simpleType if getSimpleType(simpleType).isDefined => getSimpleType(simpleType).get.cloneType()
          case NameIdentifier.INSERTED_FAKE_VARIABLE_NAME        => AnyType()
          case _ => {
            val maybeTypeArguments = trn.typeArguments.map(_.map(WeaveType(_, typeSystemScope)))
            SimpleReferenceType(trn.variable, maybeTypeArguments.map(_.toSeq), new DefaultTypeReference(trn.variable, maybeTypeArguments, typeSystemScope))
          }
        }
      }
      case functionParameterTypeNode: FunctionParameterTypeNode => WeaveType(functionParameterTypeNode.valueType, typeSystemScope)
      case LiteralTypeNode(value, _, _, _) => {
        value match {
          case n: NumberNode  => NumberType(Some(n.literalValue))
          case s: StringNode  => StringType(Some(s.literalValue))
          case b: BooleanNode => BooleanType(Some(b.literalValue == "true"))
          case _              => throw new RuntimeException(s"Type Node `${value}` is not supported as Literal Type. Please report this as a bug with the script.")
        }
      }
      case _ => throw new RuntimeException(s"Type Node `${node}` is not supported. Please report this as a bug with the script.")
    }

    node match {
      case withSchema: WeaveTypeNodeWithSchema => {
        withSchema.asSchema match {
          case Some(schema) => {
            // Has constraints but it was set to empty
            if (schema.properties.isEmpty) {
              result.markWithEmptyMetadataConstraint()
            } else {
              schema.properties.foreach((property) => {
                property.name match {
                  case nameNode: StringNode if property.value.isInstanceOf[LiteralValueAstNode] =>
                    val name = nameNode.value
                    val value = property.value match {
                      case str: StringNode => {
                        StringEscapeHelper.unescapeString(str.literalValue, str.quotedBy().getOrElse('"'), str.location())
                      }
                      case ln: LiteralValueAstNode => ln.literalValue
                    }
                    result.withMetadataConstraint(MetadataConstraint(name, value))
                  case _ =>
                }
              })
            }

          }
          case None =>
        }
        withSchema.asTypeSchema match {
          case Some(schema) =>
            val annotator = new WeaveTypeMetadataAnnotator(schema, result, typeSystemScope)
            annotator.annotate()
          case _ =>
        }

      }
      case _ => // Nothing to do
    }
    result.withDocumentation(node.weaveDoc.map(_.literalValue), node.weaveDoc.map(_.location()).getOrElse(UnknownLocation))
    result.withLocation(node.location())
  }
}

case class KeyValuePairType(var key: WeaveType, var value: WeaveType, var optional: Boolean = false, var repeated: Boolean = false) extends ts.KeyValuePairType with WeaveType {
  setParentKeyOfValue()

  private def setParentKeyOfValue(): Unit = {
    value.parentKey = key match {
      case x: KeyType => Some(x)
      case _          => None
    }
  }

  def getKeyName: String = {
    key match {
      case KeyType(NameType(Some(qName)), _) =>
        qName.getName
      case _ =>
        val asNameType = NameTypeResolver.toNameType(key, None)
        asNameType match {
          case Some(NameType(Some(qName))) =>
            qName.getName
          case _ =>
            // Printing as code by default
            new WeaveTypeEmitter().toString(key, insideKey = true)
        }
    }
  }

  def setKey(key: WeaveType): Unit = {
    this.key = key
    setParentKeyOfValue()
  }

  override def cloneType(): WeaveType = KeyValuePairType(key, value, optional, repeated)

  override def baseType(): KeyValuePairType = KeyValuePairType(key, value.baseType(), optional, repeated)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitKeyValuePairType(this)

  override def getKey(): DWType = key

  override def getValue(): DWType = value

  override def isOptionalKey(): Boolean = optional
}

trait ReferenceType extends ts.ReferenceType with WeaveType {
  /**
    * The Type expression of the referenced type
    *
    * @return
    */
  def referenceTypeName(): String

  override def getReferenceTypeName(): String = referenceTypeName()

  /**
    * Returns the resolved type
    *
    * @return
    */
  def resolveType(): WeaveType

  /**
    * Returns the type reference as defined in the type declaration and the sequence of type parameters in the declared order.
    *
    * @return
    */
  def referencedTypeDef(): Option[(NameIdentifier, WeaveType, Option[Seq[TypeParameter]])]

  /**
    * The FQN of the referenced Type expression
    *
    * @return
    */
  def referenceName(): String

  override def getReferenceFQName(): String = referenceName()

  /**
    * The name Identifier
    *
    * @return
    */
  def nameIdentifier(): NameIdentifier

  override def metadataConstraints(): Seq[MetadataConstraint] = {
    if (this._metadataConstraints.isEmpty) {
      resolveType().metadataConstraints()
    } else {
      super.metadataConstraints()
    }
  }

  override def copyMetadataConstraintsTo(to: WeaveType): Unit = {
    to match {
      case _: ReferenceType =>
        if (this._metadataConstraints.nonEmpty) {
          this._metadataConstraints.get.foreach(m => to.withMetadataConstraint(m))
        }
      case _ =>
        if (this._metadataConstraints.isEmpty) {
          this.resolveType().copyMetadataConstraintsTo(to)
        } else {
          super.copyMetadataConstraintsTo(to)
        }
    }
  }

  override def metadata(): Seq[Metadata] = {
    if (this._metadata.isEmpty) {
      resolveType().metadata()
    } else {
      val metadataFromAnnotation = this._metadata.get.filter(metadata => metadata.fromAnnotation match {
        case AnnotationSchemaNode => true
        case _                    => false
      })
      if (metadataFromAnnotation.nonEmpty) {
        metadataFromAnnotation ++ resolveType().metadata()
      } else {
        super.metadata()
      }
    }
  }

  override def copyMetadataTo(to: WeaveType): Unit = {
    to match {
      case _: ReferenceType =>
        if (this._metadata.nonEmpty) {
          this._metadata.get.foreach(m => to.withMetadata(m))
        }
      case _ =>
        if (this._metadata.isEmpty) {
          this.resolveType().copyMetadataTo(to)
        } else {
          val metadataFromAnnotation = this._metadata.get.filter(metadata => metadata.fromAnnotation match {
            case AnnotationSchemaNode => true
            case _                    => false
          })
          if (metadataFromAnnotation.nonEmpty) {
            resolveType().copyMetadataTo(to)
            metadataFromAnnotation.map(metadata => to.withMetadata(metadata))
          } else {
            super.copyMetadataTo(to)
          }
        }
    }
  }

  override def getMetadataConstraint(name: String): Option[MetadataConstraint] = {
    metadataConstraints().find(m => m.name == name)
  }

  override def getMetadata(name: String): Option[Metadata] = {
    metadata().find(m => m.name == name)
  }
}

case class TypeSelectorType(refPrefix: Option[NamespaceNode], refName: String, referencedType: ReferenceType, referenceResolver: TypeRef) extends ts.TypeSelectorType with ReferenceType {

  private val referencedTypeName: String = referencedType.referenceTypeName() + "." + refPrefix.map(nsNode => nsNode.prefix.name + "#").getOrElse("") + refName

  label(referencedTypeName)

  override def cloneType(): WeaveType = TypeSelectorType(refPrefix, refName, referencedType, referenceResolver)

  override def referenceName(): String = referencedType.referenceName() + "." + refPrefix.map(nsNode => nsNode.prefix.name + "#").getOrElse("") + refName

  override def referenceTypeName(): String = referencedTypeName

  override def nameIdentifier(): NameIdentifier = {
    NameIdentifier.fromFQN(referenceName())
  }

  override def resolveType(): WeaveType = {
    referenceResolver.ref()
  }

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitTypeSelectorType(this)

  override def getReferencePrefix: Optional[ResourceIdentifier] = {
    refPrefix.map(rp => {
      val resource: ResourceIdentifier = rp.prefix
      resource
    }).asJava
  }

  override def referencedTypeDef(): Option[(NameIdentifier, WeaveType, Option[Seq[TypeParameter]])] = None
}

case class SimpleReferenceType(refName: NameIdentifier, typeParams: Option[Seq[WeaveType]], typeRef: TypeRef) extends ReferenceType with ts.SimpleReferenceType {

  override def defaultLabel(): Option[String] = {
    Some(referenceTypeName())
  }

  def referenceTypeName(): String = {
    refName + (if (typeParams.isDefined) "<" + typeParams.get.map((wt) => {
      val emitter = new WeaveTypeEmitter(WeaveTypeEmitterConfig(prettyPrint = true, maxDepth = Some(10), nameOnly = true, printTypeBounds = true, skipWeaveTypeMetadata = true))
      emitter.toString(wt)
    }).mkString(", ") + ">"
    else "")
  }

  def resolveType(): WeaveType = {
    typeRef.ref()
  }

  override def getReferenceFQName(): String = referencedTypeDef()
    .map(typeDef => typeDef._1.toString)
    .getOrElse(super.getReferenceFQName())

  def referencedTypeDef(): Option[(NameIdentifier, WeaveType, Option[Seq[TypeParameter]])] = {
    typeRef.refResolver().flatMap(resolver => {
      resolver.referenceTypeDef(refName)
    })
  }

  override def cloneType(): WeaveType = SimpleReferenceType(refName, typeParams, typeRef)

  override def referenceName(): String = refName.loader.map(loader => loader + "!").getOrElse("") + referenceTypeName

  override def nameIdentifier(): NameIdentifier = refName

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitSimpleReferenceType(this)

  override def getTypeParameters: Optional[Array[DWType]] = typeParams.map(_.toArray[DWType]).asJava
}

case class KeyType(var name: WeaveType, var attrs: Seq[NameValuePairType] = Seq()) extends ts.KeyType with WeaveType {

  override def cloneType(): WeaveType = KeyType(name, attrs)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitKeyType(this)

  override def getAttributes: Array[ts.NameValuePairType] = attrs.toArray

  override def getName: DWType = name
}

case class NameValuePairType(name: WeaveType, value: WeaveType, optional: Boolean = false) extends ts.NameValuePairType with WeaveType {
  override def cloneType(): WeaveType = NameValuePairType(name, value, optional)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitNameValuePairType(this)

  override def getName: WeaveType = name

  override def getValue: WeaveType = value

  override def isOptionalPairType: Boolean = optional
}

case class NameType(value: Option[QName] = None) extends ts.NameType with WeaveType {
  override def cloneType(): WeaveType = NameType(value)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitNameType(this)

  override def getValue: Optional[ts.QName] = {
    val maybeQName: Option[ts.QName] = value
    maybeQName.asJava
  }
}

case class ArrayType(var of: WeaveType) extends ts.ArrayType with WeaveType {
  override def cloneType(): WeaveType = ArrayType(of)

  override def baseType(): WeaveType = ArrayType(of.baseType())

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitArrayType(this)

  override def arrayOf: DWType = of
}

case class UnionType(var of: Seq[WeaveType]) extends ts.UnionType with WeaveType {
  override def cloneType(): WeaveType = UnionType(of)

  override def baseType(): WeaveType = TypeHelper.unify(of.map((w: WeaveType) => w.baseType()))

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitUnionType(this)

  override def unionOf: Array[DWType] = of.toArray
}

case class IntersectionType(var of: Seq[WeaveType]) extends ts.IntersectionType with WeaveType {
  override def cloneType(): WeaveType = IntersectionType(of)

  override def baseType(): WeaveType = IntersectionType(of.map((w: WeaveType) => w.baseType()))

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitIntersectionType(this)

  override def intersectionOf: Array[DWType] = of.toArray
}

@WeaveApi(Seq("data-weave-agent"))
case class StringType(value: Option[String] = None) extends ts.StringType with WeaveType {
  override def cloneType(): WeaveType = StringType(value)

  override def baseType(): WeaveType = StringType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitStringType(this)

  override def getValue: Optional[String] = value.asJava
}

case class NamespaceType(prefix: Option[String] = None, namespace: Option[UriType] = None) extends ts.NamespaceType with WeaveType {
  override def cloneType(): WeaveType = NamespaceType(prefix, namespace)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitNamespaceType(this)

  override def getPrefix: Optional[String] = prefix.asJava

  override def getNamespace: Optional[ts.UriType] = {
    val apiNamespace: Option[ts.UriType] = namespace
    apiNamespace.asJava
  }
}

@WeaveApi(Seq("data-weave-agent"))
case class AnyType() extends ts.AnyType with WeaveType {
  override def cloneType(): WeaveType = AnyType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitAnyType(this)
}

@WeaveApi(Seq("data-weave-agent"))
case class BooleanType(value: Option[Boolean] = None, constraints: VariableConstraints = VariableConstraints.emptyConstraints()) extends ts.BooleanType with WeaveType {
  override def cloneType(): WeaveType = BooleanType(value, constraints)

  override def equals(obj: scala.Any): Boolean = obj.isInstanceOf[BooleanType]

  override def baseType(): WeaveType = BooleanType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitBooleanType(this)

  override def getValue: Optional[Boolean] = value.asJava
}

@WeaveApi(Seq("data-weave-agent"))
case class NumberType(value: Option[String] = None) extends ts.NumberType with WeaveType {
  override def cloneType(): WeaveType = NumberType(value)

  override def baseType(): WeaveType = NumberType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitNumberType(this)

  override def getValue: Optional[String] = value.asJava
}

case class RangeType() extends ts.RangeType with WeaveType {
  override def cloneType(): WeaveType = RangeType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitRangeType(this)
}

case class UriType(value: Option[String] = None) extends ts.UriType with WeaveType {

  override def cloneType(): WeaveType = UriType(value)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitUriType(this)

  override def getValue: Optional[String] = value.asJava
}

case class DateTimeType() extends ts.DateTimeType with WeaveType {
  override def cloneType(): WeaveType = DateTimeType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitDateTimeType(this)
}

//
case class LocalDateTimeType() extends ts.LocalDateTimeType with WeaveType {
  override def cloneType(): WeaveType = LocalDateTimeType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitLocalDateTimeType(this)
}

case class LocalDateType() extends ts.LocalDateType with WeaveType {
  override def cloneType(): WeaveType = LocalDateType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitLocalDateType(this)
}

case class LocalTimeType() extends ts.LocalTimeType with WeaveType {
  override def cloneType(): WeaveType = LocalTimeType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitLocalTimeType(this)
}

case class TimeType() extends ts.TimeType with WeaveType {
  override def cloneType(): WeaveType = TimeType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitTimeType(this)
}

case class TimeZoneType() extends ts.TimeZoneType with WeaveType {
  override def cloneType(): WeaveType = TimeZoneType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitTimeZoneType(this)
}

case class PeriodType() extends ts.PeriodType with WeaveType {
  override def cloneType(): WeaveType = PeriodType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitPeriodType(this)
}

case class BinaryType() extends ts.BinaryType with WeaveType {
  override def cloneType(): WeaveType = BinaryType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitBinaryType(this)
}

case class TypeType(t: WeaveType) extends ts.TypeType with WeaveType {
  override def cloneType(): WeaveType = TypeType(t)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitTypeType(this)

  override def getType: WeaveType = t
}

//
case class RegexType() extends ts.RegexType with WeaveType {
  override def cloneType(): WeaveType = RegexType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitRegexType(this)
}

case class NullType() extends ts.NullType with WeaveType {

  private var selectionMiss: Boolean = false

  def markSelectionMissed(): this.type = {
    selectionMiss = true
    this
  }

  def isSelectionMissed(): Boolean = {
    selectionMiss
  }

  override def cloneType(): WeaveType = NullType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitNullType(this)
}

case class NothingType() extends ts.NothingType with WeaveType {
  override def cloneType(): WeaveType = NothingType()

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitNothingType(this)
}

case class TypeParameter(name: String, top: Option[WeaveType] = None, bottom: Option[WeaveType] = None, instanceId: Option[Number] = None, var noImplicitBounds: Boolean = false) extends ts.TypeParameter with WeaveType {

  def isConcrete(): Boolean = instanceId.isDefined

  override def isAbstract(): Boolean = instanceId.isEmpty

  //Type Parameters can not be cloned
  override def cloneType(): WeaveType = this

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitTypeParameter(this)

  override def getTopType: Optional[DWType] = {
    val apiTopType: Option[DWType] = top
    apiTopType.asJava
  }

  override def getBottomType: Optional[DWType] = {
    val apiTopType: Option[DWType] = bottom
    apiTopType.asJava
  }

}

case class FunctionTypeParameter(name: String, wtype: WeaveType, optional: Boolean = false, defaultValueType: Option[WeaveType] = None) extends ts.FunctionTypeParameter {

  override def getType: WeaveType = wtype

  override def getDefaultValueType: Optional[DWType] = {
    val defaultApiValueType: Option[DWType] = defaultValueType
    defaultApiValueType.asJava
  }

  override def getName: String = name

  override def isOptional: Boolean = optional
}

case class FunctionType(var typeParams: Seq[TypeParameter] = Seq(), var params: Seq[FunctionTypeParameter], var returnType: WeaveType, var overloads: Seq[FunctionType] = Seq(), var name: Option[String] = None, customReturnTypeResolver: Option[CustomTypeResolver] = None) extends ts.FunctionType with WeaveType {
  override def cloneType(): WeaveType = FunctionType(typeParams, params, returnType, overloads, name, customReturnTypeResolver)

  override def isOverloaded(): Boolean = overloads.nonEmpty

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitFunctionType(this)

  override def getTypeParameters: Array[ts.TypeParameter] = typeParams.toArray

  override def getParameters: Array[ts.FunctionTypeParameter] = params.toArray

  override def getOverloads: Array[ts.FunctionType] = overloads.toArray

  override def getName: Optional[String] = name.asJava

  override def getReturnType: DWType = returnType
}

case class DynamicReturnType(typeParameters: Seq[FunctionTypeParameter], node: FunctionNode, typeGraph: TypeGraph, scope: ScopesNavigator, name: Option[String], expectedReturnType: Option[WeaveType], resolver: ReferenceResolver) extends ts.DynamicReturnType with WeaveType {
  override def cloneType(): WeaveType = DynamicReturnType(typeParameters, node, typeGraph, scope, name, expectedReturnType, resolver)

  override def accept(visitor: DWTypeVisitor): Unit = visitor.visitDynamicReturnType(this)

  override def getTypeParameters: Array[ts.FunctionTypeParameter] = typeParameters.toArray

  override def getName: Optional[String] = name.asJava

  override def getExpectedReturnType: Optional[DWType] = {
    val expectedReturnApiType: Option[DWType] = expectedReturnType
    expectedReturnApiType.asJava
  }
}
