package org.mule.weave.v2.editor.composer

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.grammar.AttributeValueSelectorOpId
import org.mule.weave.v2.grammar.MultiAttributeValueSelectorOpId
import org.mule.weave.v2.grammar.MultiValueSelectorOpId
import org.mule.weave.v2.grammar.ValueSelectorOpId
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.AstNodeHelper.collectChildrenWith
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.header.directives
import org.mule.weave.v2.parser.ast.header.directives.DirectiveOption
import org.mule.weave.v2.parser.ast.header.directives.DirectiveOptionName
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.header.directives.OutputDirective
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.AttributesNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.KeyNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NameValuePairNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.UriNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.CompilationException
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.ScopeGraphResult
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResource

import java.net.URI
import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.util.Try

/**
  * This class will handle the translation between Composer script and valid DW scripts.
  * Composer uses some convention in order to simplify UI. So this class will handle that composer semantic and translate it into valid scripts.
  */
object ComposerExpressionHelper extends ComposerExpressionValidator {

  val ATTRIBUTE_PREFIX = "{_@}"

  private lazy val parser = new ComposerExpressionParser()

  private def createParsingContext(implicitInputs: Array[String]): ParsingContext = {
    val context = ParsingContextFactory.createParsingContext(NameIdentifier.ANONYMOUS_NAME)
    for (input <- implicitInputs) {
      context.addImplicitInput(input, None)
    }
    context
  }

  /**
    * Parses a DW script to an expression node
    * @param script The DW script to parse
    * @param name A name it is going to be used for debugging
    * @return The resulting ComposerExpressionNode or null if the script could not be parse
    */
  def parseExpression(script: String, name: String): ComposerExpressionNode = {
    val maybeExpressionNode = parser.parseExpression(script, name, createParsingContext(Array.empty))
    maybeExpressionNode.orNull
  }

  /**
    * Check if the DW script use a set of allowed functions.
    *
    * @param script The DW script to be validated
    * @param allowedModules The set of allowed modules by the DW script
    * @return the resulting validation messages
    * @throws CompilationException if script is invalid
    */
  def validateAgainstModules(script: String, allowedModules: Array[String]): ComposerValidationMessages = {
    validate(script, Array.empty, allowedModules, Array.empty)
  }

  /**
    * Check if the DW script use a set of allowed functions.
    *
    * @param script The DW script to be validated
    * @param allowedModules The set of allowed modules by the DW script
    * @param implicitInputs The set of implicit input used in the DW script
    * @return the resulting validation messages
    * @throws CompilationException if script is invalid
    */
  def validateAgainstModules(script: String, allowedModules: Array[String], implicitInputs: Array[String]): ComposerValidationMessages = {
    validate(script, Array.empty, allowedModules, implicitInputs)
  }

  /**
    * Check if the DW script use a set of allowed functions.
    *
    * @param script The DW script to be validated
    * @param allowedFunctions The set of allowed functions by the DW script
    * @return the resulting validation messages
    * @throws CompilationException if script is invalid
    */
  def validate(script: String, allowedFunctions: Array[String]): ComposerValidationMessages = {
    validate(script, allowedFunctions, Array.empty, Array.empty)
  }

  /**
    * Check if the DW script use a set of allowed functions.
    *
    * @param script The DW script to be validated
    * @param allowedFunctions The set of allowed functions by the DW script
    * @param implicitInputs The set of implicit input used in the DW script
    * @return the resulting validation messages
    * @throws CompilationException if script is invalid
    */
  def validate(script: String, allowedFunctions: Array[String], implicitInputs: Array[String]): ComposerValidationMessages = {
    validate(script, allowedFunctions, Array.empty, implicitInputs)
  }

  /**
    * Check if the DW script use a set of allowed functions.
    *
    * @param script The DW script to be validated
    * @param allowedFunctions The set of allowed functions by the DW script
    * @param allowedModules The set of allowed modules by the DW script
    * @param implicitInputs The set of implicit input used in the DW script
    * @return the resulting validation messages
    * @throws CompilationException if script is invalid
    */
  def validate(script: String, allowedFunctions: Array[String], allowedModules: Array[String], implicitInputs: Array[String]): ComposerValidationMessages = {
    val parsingContext = createParsingContext(implicitInputs)
    val value: PhaseResult[ScopeGraphResult[DocumentNode]] = MappingParser.parse(MappingParser.scopePhase(), WeaveResource.anonymous(script), parsingContext)
    value.getResult()
    validateComposerExpression(value, parsingContext, allowedFunctions, allowedModules)
  }

  /**
    * Check if reference directly any function that requires dw::core::Core::RuntimePrivileges
    * @param composerScript The script to be validated
    * @return If true then the function reference directly a function that requires privileges.
    */
  def requiresRuntimePrivileges(composerScript: String): Boolean = {
    requiresRuntimePrivileges(composerScript, Array.empty)
  }

  /**
    * Check if reference directly any function that requires dw::core::Core::RuntimePrivileges
    * @param composerScript The script to be validated
    * @param implicitInputs The set of implicit input used in the DW script
    * @return If true then the function reference directly a function that requires privileges.
    */
  def requiresRuntimePrivileges(composerScript: String, implicitInputs: Array[String]): Boolean = {
    val context = createParsingContext(implicitInputs)
    val value: PhaseResult[ScopeGraphResult[DocumentNode]] = MappingParser.parse(MappingParser.scopePhase(), WeaveResource.anonymous(composerScript), context)
    val result = value.getResult()
    val nodes = AstNodeHelper.collectChildrenWith(result.astNode, classOf[VariableReferenceNode])
    nodes.exists((vrn) => {
      val maybeReference = result.scope.resolveVariable(vrn.variable)
      maybeReference match {
        case Some(functionRef) => {
          val node = functionRef.referencedNode
          val scope = functionRef.scope
          val navigator = scope.astNavigator()
          navigator.parentOf(node) match {
            case Some(node: FunctionDirectiveNode) => {
              val annotations = node.codeAnnotations
              annotations.exists((ann) => {
                scope.resolveVariable(ann.name) match {
                  case Some(annotationRef) => {
                    annotationRef.fqnReferenceName == NameIdentifier.RUNTIME_PRIVILEGE_ANNOTATION
                  }
                  case None => false
                }

              })
            }
            case _ => false
          }
        }
        case None => false
      }
    })

  }

  /**
    * This method will handle the translation between composer semantics and valid DW semantics
    *
    * @param composerScript The Composer script to be translated.
    * @param expectedOutputMimeType The MimeType to be set as output
    * @param writerProperties The Set of writer properties to be set to the output
    * @return The new DW code
    * @throws CompilationException if script is invalid
    */
  def transpile(composerScript: String, expectedOutputMimeType: String, writerProperties: java.util.Map[String, Object]): String = {
    val uriToPrefix: mutable.Map[String, String] = new mutable.HashMap[String, String]()
    val parsingContext = createParsingContext(Array.empty)
    val value: PhaseResult[ParsingResult[DocumentNode]] = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource.anonymous(composerScript), parsingContext)

    val documentNode: DocumentNode = value.getResult().astNode
    val keyNodes: Seq[KeyNode] = collectChildrenWith(documentNode, classOf[KeyNode])
    keyNodes.foreach((kn) => {
      transpileKeyName(kn, uriToPrefix)
      transpileAttributes(kn, uriToPrefix)
    })

    val binaryOpNodes = collectChildrenWith(documentNode, classOf[BinaryOpNode])
      .filter((bon) => {
        bon.rhs.isInstanceOf[NameNode] &&
          isSelectorOperator(bon)
      })

    binaryOpNodes.foreach((bop) => {
      bop.rhs match {
        case nn: NameNode => {
          nn.keyName match {
            case sn: StringNode => {
              var literalValue: String = sn.literalValue
              var switchToAttribute: Boolean = false
              if (literalValue.startsWith(ATTRIBUTE_PREFIX)) {
                literalValue = literalValue.substring(ATTRIBUTE_PREFIX.length)
                switchToAttribute = true
              }
              extractNamespace(literalValue) match {
                case Some(qname) => {
                  val name = qname.name
                  val newName = StringNode(name).withQuotation('"')
                  nn.keyName = newName
                  nn.ns = qname.ns.map((ns) => {
                    val namespacePrefix = uriToPrefix.getOrElseUpdate(ns, s"ns${uriToPrefix.size}")
                    NamespaceNode(NameIdentifier(namespacePrefix))
                  })
                }
                case _ if (switchToAttribute) => {
                  //The new name without the {_@}
                  val name: String = literalValue
                  val newName = StringNode(name).withQuotation('"')
                  nn.keyName = newName
                }
                case _ =>
              }

              if (switchToAttribute) {
                bop.binaryOpId = bop.opId match {
                  case MultiValueSelectorOpId => {
                    MultiAttributeValueSelectorOpId
                  }
                  case ValueSelectorOpId => {
                    AttributeValueSelectorOpId
                  }
                  case opId => opId
                }
              }
            }
            case _ =>
          }
        }
        case _ =>
      }
    })

    val namespaceDirectives = uriToPrefix.map((pair) => NamespaceDirective(NameIdentifier(pair._2), UriNode(pair._1)))
    documentNode.header.directives = documentNode.header.directives ++ namespaceDirectives

    if (expectedOutputMimeType != null && expectedOutputMimeType.trim.nonEmpty) {
      val hasOutputDirective = documentNode.header.directives.exists((dn) => dn.isInstanceOf[OutputDirective])
      if (!hasOutputDirective) {
        //Add output directive
        val options = writerProperties.asScala
          .map((pair) => {
            val value = pair._2 match {
              case s: String            => StringNode(s).withQuotation('"')
              case i: java.lang.Number  => NumberNode(i.toString)
              case b: java.lang.Boolean => BooleanNode(b.toString)
              case v                    => StringNode(v.toString)
            }
            DirectiveOption(DirectiveOptionName(pair._1), value)
          })
          .toSeq
        val outputDirective = new OutputDirective(None, Some(directives.ContentType(expectedOutputMimeType)), Some(options), None, Seq())
        documentNode.header.directives = documentNode.header.directives :+ outputDirective
      }
    }

    CodeGenerator.generate(documentNode)
  }

  private def isSelectorOperator(bon: BinaryOpNode) = {
    bon.opId == MultiValueSelectorOpId || bon.opId == ValueSelectorOpId
  }

  private def transpileKeyName(kn: KeyNode, uriToPrefix: mutable.Map[String, String]): Unit = {
    kn.keyName match {
      case sn: StringNode => {
        extractNamespace(sn.literalValue) match {
          case Some(qname) => {
            val name = qname.name
            val newName = StringNode(name).withQuotation('"')
            kn.keyName = newName
            kn.ns = qname.ns.map((ns) => {
              val namespacePrefix = uriToPrefix.getOrElseUpdate(ns, s"ns${uriToPrefix.size}")
              NamespaceNode(NameIdentifier(namespacePrefix))
            })
          }
          case None =>
        }

      }
      case _ =>
    }
  }

  private def transpileAttributes(kn: KeyNode, uriToPrefix: mutable.Map[String, String]): Unit = {
    kn.attr match {
      case Some(an: AttributesNode) => {
        an.attrs.foreach({
          case nameValuePairNode: NameValuePairNode =>
            val nameNode: AstNode = nameValuePairNode.key
            nameNode match {
              case nn: NameNode => {
                nn.keyName match {
                  case sn: StringNode => {
                    extractNamespace(sn.literalValue) match {
                      case Some(qname) => {
                        val name = qname.name
                        val newName = StringNode(name).withQuotation('"')
                        nn.keyName = newName
                        nn.ns = qname.ns.map((ns) => {
                          val namespacePrefix = uriToPrefix.getOrElseUpdate(ns, s"ns${uriToPrefix.size}")
                          NamespaceNode(NameIdentifier(namespacePrefix))
                        })
                      }
                      case _ =>
                    }
                  }
                  case _ =>
                }
              }
              case _ =>
            }
          case _ =>
        })
      }
      case _ =>
    }
  }

  private def extractNamespace(literalValue: String): Option[QName] = {
    if (literalValue.startsWith("{") && literalValue.contains("}")) {
      val indexOfCloseCurly = literalValue.indexOf("}")
      val uriString = literalValue.substring(1, indexOfCloseCurly)
      Try(new URI(uriString))
        .map((_) => {
          QName(literalValue.substring(indexOfCloseCurly + 1), Some(uriString))
        })
        .toOption
    } else {
      None
    }
  }
}
