package org.mule.weave.v2.sdk

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.SimpleParserPosition
import org.mule.weave.v2.parser.location.UnknownPosition

import java.nio.charset.StandardCharsets

/**
  * Represents a resource in a weave runtime
  */
@WeaveApi(Seq("data-weave-agent"))
trait WeaveResource {

  /**
    * The url of this resource
    *
    * @return
    */
  def url(): String

  /**
    * Returns the offset of a given line column
    *
    * @param line   the line number zero based
    * @param column The column number
    */
  def offsetOf(line: Int, column: Int): Int = {
    val contentStr: Iterator[String] = content().linesWithSeparators
    var i = 0
    var offset = 0
    while (i < line && contentStr.hasNext) {
      offset = offset + contentStr.next().length //Line number
      i = i + 1
    }
    offset + column
  }

  /**
    * Returns the Position of a given offset
    *
    * @param offset The offset to search the position
    * @return
    */
  def positionOf(offset: Int): Position = {
    val contentStr = content()
    if (offset < 0 || contentStr.length < offset) {
      UnknownPosition
    } else {

      var currentIndex: Int = 0
      var currentLine: Int = 1
      var currentColumn: Int = 1

      while (currentIndex < offset) {
        val currentChar: Char = contentStr.charAt(currentIndex)
        currentIndex = currentIndex + 1
        if (currentChar == '\r') {
          if (contentStr.length > currentIndex && contentStr.charAt(currentIndex) == '\n') {
            currentIndex = currentIndex + 1
          }
          currentLine = currentLine + 1
          currentColumn = 1
        } else if (currentChar == '\n') {
          currentLine = currentLine + 1
          currentColumn = 1
        } else {
          currentColumn = currentColumn + 1
        }
      }
      SimpleParserPosition(offset, currentLine, currentColumn, () => {
        content().linesIterator.toSeq(currentLine)
      })
    }
  }

  /**
    * The content of the resource
    *
    * @return
    */
  def content(): String

  /**
    * Returns the line number of the specified index zero based
    *
    * @param index The index to search
    * @return The line number if the index is out of range return -1
    */
  def lineNumberOf(index: Int): Int = {
    positionOf(index) match {
      case UnknownPosition => -1
      case position        => position.line - 1
    }
  }

  def previousNonWSChar(index: Int): Option[Char] = {
    var i = index - 1
    var result: Char = if (i > 0 && i < content().length) content().charAt(i) else ' '
    while (i > 0 && Character.isWhitespace(result)) {
      result = content().charAt(i)
      i = i - 1
    }
    if (Character.isWhitespace(result)) {
      None
    } else {
      Some(result)
    }
  }

}

object WeaveResource {
  def apply(url: String, content: String) = DefaultWeaveResource(url, content)

  def anonymous(content: String) = DefaultWeaveResource("anonymous", content)

  def apply(virtualFile: VirtualFile) = DefaultWeaveResource(virtualFile.url(), virtualFile.read())
}

case class BinaryWeaveResource(binaryContent: Array[Byte], sourceWeaveResource: WeaveResource) extends WeaveResource {

  override def content(): String = sourceWeaveResource.content()

  override def url(): String = sourceWeaveResource.url()
}

case class DefaultWeaveResource(url: String, content: String) extends WeaveResource
