package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.KeyNode
import org.mule.weave.v2.parser.ast.structure.KeyValuePairNode
import org.mule.weave.v2.parser.ast.structure.ObjectNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.utils.SchemaNodeOrigin
import org.mule.weave.v2.utils.StringEscapeHelper

class WeaveTypeMetadataAnnotator(schema: SchemaNode, result: WeaveType, typeReferenceResolver: WeaveTypeReferenceResolver) {

  def annotate(): Unit = {
    if (schema.properties.isEmpty) {
      result.markWithEmptyMetadata()
    } else {
      schema.properties.foreach(property => {
        property.name match {
          case nameNode: StringNode =>
            val key = nameNode.value
            val metadataValue = generateMetadataValue(property.value)
            result.withMetadata(Metadata(key, metadataValue, SchemaNodeOrigin(schema)))
          case _ =>
          // Nothing to do
        }
      })
    }
  }

  private def generateMetadataValue(node: AstNode): MetadataValue = {
    node match {
      case str: StringNode =>
        val value = StringEscapeHelper.unescapeString(str.literalValue, str.quotedBy().getOrElse('"'), str.location())
        val weaveType = SystemWeaveTypes.getLiteralTypeByNode(str.getClass).orNull
        LiteralMetadataValue(value, weaveType, str.location())

      case ln: LiteralValueAstNode =>
        val value = ln.literalValue
        val weaveType = SystemWeaveTypes.getLiteralTypeByNode(ln.getClass).orNull
        LiteralMetadataValue(value, weaveType, ln.location())

      case an: ArrayNode =>
        val elements = an.elements.map(generateMetadataValue)
        ArrayMetadataValue(elements, an.location())

      case on: ObjectNode =>
        val properties = on.elements.map(element => {
          val generate = generateMetadataValue(element)
          generate.asInstanceOf[KeyValuePairMetadataValue]
        })
        ObjectMetadataValue(properties, on.location())

      case kvp: KeyValuePairNode =>
        val keyName = getKeyName(kvp.key)
        val value = generateMetadataValue(kvp.value)
        KeyValuePairMetadataValue(keyName, value, kvp.location())

      case vrf: VariableReferenceNode =>
        val maybeResolved = typeReferenceResolver.variableResolver().flatMap(_.resolveVariable(vrf.variable))
        if (maybeResolved.isDefined) {
          VariableReferenceMetadataValue(maybeResolved.get.fqnReferenceName.fullQualifiedName(), vrf.location())
        } else {
          UnknownMetadataValue(vrf.location())
        }

      case other =>
        UnknownMetadataValue(other.location())
    }
  }

  private def getKeyName(key: AstNode): String = {
    key match {
      case KeyNode(StringNode(keyValue, _), _, _, _) =>
        keyValue
      case _ =>
        "unknown"
    }
  }
}

object WeaveTypeMetadataAnnotator {
  def apply(schema: SchemaNode, result: WeaveType, typeReferenceResolver: WeaveTypeReferenceResolver): WeaveTypeMetadataAnnotator = new WeaveTypeMetadataAnnotator(schema, result, typeReferenceResolver)
}
