package org.mule.weave.v2.scope

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.ImportedElement
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.utils.LazyValRef

/**
  * This class represent the variable scope that we can build when we are parsing a bdwl. This is a much simpler version
  * than org.mule.weave.v2.scope.VariableScope as we don't traverse through child scopes.
  */
class BinaryModuleVariableScope(
  val parsingContext: ParsingContext,
  val name: Option[String],
  val astNode: ModuleNode,
  val index: Int,
  val parentScope: Option[VariableScope],
  val declarations: Seq[NameIdentifier],
  var importedModules: Seq[(ImportDirective, LazyValRef[BaseVariableScope])]) extends BaseVariableScope {

  override val references: Seq[NameIdentifier] = Seq.empty

  def rootScope(): BaseVariableScope = parentScope.map(_.rootScope()).getOrElse(this)

  override def resolveVariable(name: NameIdentifier): Option[Reference] = {
    resolveLocalVariable(name)
      .orElse(resolveImportedVariable(name))
      .orElse(parentScope.flatMap(_.resolveVariable(name)))
  }

  override def resolveLocalVariable(name: NameIdentifier): Option[Reference] = {
    if (!name.isLocalReference()) {
      val moduleIdentifier: NameIdentifier = name.parent().get
      if (moduleIdentifier.equals(parsingContext.nameIdentifier)) {
        //Referencing  to this module
        return resolveLocalVariable(name.localName())
      } else {
        return None
      }
    }

    val localVariable = declarations.find(x => x.name == name.name)
    localVariable.map(Reference(_, this))
  }

  def collectLocalVariables(collector: Reference => Boolean): Seq[Reference] = {
    declarations.map(Reference(_, this)).filter(collector)
  }

  def localVariables: Seq[Reference] = declarations.map(Reference(_, this))

  override def asVariableScope: VariableScope = throw UnexpectedBinaryModuleVariableScope()

  override def scopesNavigator(): ScopesNavigator = throw UnexpectedBinaryModuleVariableScope()

  override def referenceResolver(): ScopeGraphTypeReferenceResolver = new ScopeGraphTypeReferenceResolver(new BinaryModuleVariableResolver(this))

  override def isRootScope(): Boolean = parentScope.isEmpty

  override def children(): Seq[VariableScope] = Seq()

  override def resolveLocalReferenceTo(nameIdentifier: NameIdentifier): Seq[Reference] = throw UnexpectedBinaryModuleVariableScope()

  override def resolveReferenceTo(nameIdentifier: NameIdentifier): Seq[Reference] = throw UnexpectedBinaryModuleVariableScope()

  override def shadowedVariables(): Seq[NameIdentifier] = throw UnexpectedBinaryModuleVariableScope()

  override def visibleVariables: Seq[Reference] = throw UnexpectedBinaryModuleVariableScope()

  private def resolveImportedVariable(name: NameIdentifier): Option[Reference] = {
    importedModules
      .toStream
      .flatMap(module => {
        val importDirective: ImportDirective = module._1
        val importScope = module._2
        val importedElements: Seq[AstNode] = importDirective.subElements.children()

        val maybeAbsoluteReference = if (name.isLocalReference() && importedElements.nonEmpty) {
          resolveLocalRefInImports(name, importDirective, importScope.get())
        } else if (importedElements.isEmpty) {
          resolveRefWithModuleName(name, importDirective, importScope.get())
        } else {
          None
        }
        maybeAbsoluteReference
      }).headOption
  }

  private def resolveLocalRefInImports(nameToResolve: NameIdentifier, importDirective: ImportDirective, importScope: BaseVariableScope): Option[Reference] = {
    val moduleIdentifier: NameIdentifier = importDirective.importedModule.elementName
    val importedElements: Seq[AstNode] = importDirective.subElements.children()

    importedElements
      .toStream
      .flatMap({
        case ImportedElement(NameIdentifier("*", None), None) =>
          val variable: Option[Reference] = importScope.resolveLocalVariable(nameToResolve)
          toAbsoluteReference(variable, moduleIdentifier)
        case ImportedElement(elementName, alias) =>
          val matchesImportedElemName = alias.getOrElse(elementName).name.equals(nameToResolve.name)
          if (matchesImportedElemName) {
            toAbsoluteReference(importScope.resolveLocalVariable(elementName), moduleIdentifier)
          } else {
            None
          }
      })
      .headOption
  }

  /**
    * This resolves references with the form moduleName::name
    */
  protected def resolveRefWithModuleName(nameToResolve: NameIdentifier, importDirective: ImportDirective, importScope: BaseVariableScope) = {
    val moduleIdentifier: NameIdentifier = importDirective.importedModule.elementName

    nameToResolve.parent() match {
      case Some(parentName) if parentName.parent().isEmpty =>
        val matchesImportedModuleName = importDirective.importedModule.aliasOrLocalName.equals(parentName.localName())
        if (matchesImportedModuleName) {
          val nameIdentifier = importScope.resolveLocalVariable(nameToResolve.localName())
          toAbsoluteReference(nameIdentifier, moduleIdentifier)
        } else {
          None
        }
      case _ => None
    }
  }
}

private case class UnexpectedBinaryModuleVariableScope() extends RuntimeException("Encountered BinaryModuleVariableScope in unexpected place")
