package org.mule.weave.v2.parser

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.imports.UnusedImportsPhase
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.phase._
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.versioncheck.LanguageLevelVersionCheckPhase
import org.mule.weave.v2.visibility.AccessCheckPhase
import org.parboiled2.ParseError
import org.parboiled2.Parser.DeliveryScheme.Either

/**
  * Parses a DataWeave Mapping File.
  */
object MappingParser {

  val parsingValidations: Array[AstNodeVerifier] = Array(
    new ErrorsNodeValidation(),
    new LiteralValidation(),
    new NamespaceValidator(),
    new DoBlockDirectivesValidation(),
    new UniqueNameVariablesDeclarationValidation(),
    new UniqueNameFunctionParametersValidation(),
    new FunctionDefaultValueLocationValidation(),
    new PatternMatcherWithMoreThanOneDefaultValidation(),
    new TypeDeclValidation(),
    new SyntaxDirectiveValidation())

  private val compilationTransformations = Array(
    new ImplicitOutputTransformer(),
    new ImplicitInputsTransformer(),
    new ImplicitImportsTransformer())

  def parsingPhase(): CompilationPhase[ParsingContentInput, ParsingResult[DocumentNode]] = {
    ParsingPhase[DocumentNode](parse)
      .chainWith(new AstNodeVerificationPhase[DocumentNode, ParsingResult[DocumentNode]](parsingValidations))
      .chainWith(new ParsingAnnotationProcessorPhase[DocumentNode, ParsingResult[DocumentNode]]())
  }

  def scopePhase(): CompilationPhase[ParsingContentInput, ScopeGraphResult[DocumentNode]] = {
    canonicalAstPhase().chainWith(scopePhasePhases())
  }

  def scopePhasePhases(): CompilationPhase[AstNodeResultAware[DocumentNode] with ParsingContentAware, ScopeGraphResult[DocumentNode]] = {
    ScopeGraphPhase[DocumentNode]()
      .chainWith(new MaterializeVariableMarkerPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new MaterializeTypeMarkerPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new StreamingCapableVariableMarkerPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new TailRecursiveMarkerPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new AnnotationArgumentNodeDefaultValuePhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new ScopeAnnotationProcessingPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new MetadataKeyNodeInjectorPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new MetadataTypeNodeInjectorPhase[DocumentNode, ScopeGraphResult[DocumentNode]])
      .chainWith(new LanguageLevelVersionCheckPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new AccessCheckPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new UnusedImportsPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new ImportsValidation[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new TypeSelectorValidation[DocumentNode, ScopeGraphResult[DocumentNode]]())

  }

  def canonicalAstPhase(): CompilationPhase[ParsingContentInput, ParsingResult[DocumentNode]] = {
    parsingPhase()
      .chainWith(canonicalPhasePhases())
  }

  def canonicalPhasePhases(): CompilationPhase[ParsingResult[DocumentNode], ParsingResult[DocumentNode]] = {
    new VersionCheckerPhase[DocumentNode]()
      .chainWith(new FunctionAggregationPhase[DocumentNode]())
      .chainWith(new AstNodeTransformationPhase[DocumentNode, ParsingResult[DocumentNode]](compilationTransformations))
      .chainWith(new DummyScopesNavigatorPhases[DocumentNode])
      .chainWith(new CanonicalAnnotationProcessingPhase[DocumentNode, ParsingResultWithDummyScopeNavigator[DocumentNode]]())
      .chainWith(new ImplicitFunctionTransformer[DocumentNode, ParsingResultWithDummyScopeNavigator[DocumentNode]]())
  }

  @WeaveApi(Seq("data-weave-agent"))
  def typeCheckPhase(): CompilationPhase[ParsingContentInput, TypeCheckingResult[DocumentNode]] = {
    scopePhase().chainWith(typeCheckPhasePhases())
  }

  def scopePhasesAndTypeChecking(): CompilationPhase[ParsingResult[DocumentNode], TypeCheckingResult[DocumentNode]] = {
    canonicalPhasePhases().chainWith(scopePhasePhases().chainWith(typeCheckPhasePhases()))
  }

  def typeCheckPhasePhases(): CompilationPhase[ScopeGraphResult[DocumentNode], TypeCheckingResult[DocumentNode]] = {
    new TypeCheckingPhase[DocumentNode]()
      .chainWith(new EqualityTypeChecker[DocumentNode, TypeCheckingResult[DocumentNode]]())
      .chainWith(new AnnotationParameterValidation[DocumentNode, TypeCheckingResult[DocumentNode]]())
      .chainWith(new TypeReferenceGenericsChecker[DocumentNode, TypeCheckingResult[DocumentNode]]())
      .chainWith(new TypeAnnotationProcessingPhase[DocumentNode, TypeCheckingResult[DocumentNode]]())
  }

  @WeaveApi(Seq("data-weave-agent"))
  def parse[T](phase: CompilationPhase[ParsingContentInput, T], input: WeaveResource, parsingContext: ParsingContext): PhaseResult[T] = {
    val parserInput = ParsingContentInput(input, parsingContext.nameIdentifier, SafeStringBasedParserInput(input.content()))
    phase.call(parserInput, parsingContext)
  }

  private def parse(parserInput: ParsingContentInput, context: ParsingContext): Either[ParseError, DocumentNode] = {
    val parser = new Grammar(
      parserInput.input,
      parserInput.nameIdentifier,
      context.errorTrace,
      context.attachDocumentation)
    parser.document.run()
  }
}
